# -*- coding: utf-8 -*-

from datetime import datetime
import logging

from passport.backend.api.common.processes import PROCESS_ACCOUNT_DELETE_V2
from passport.backend.api.email_validator.mixins import EmailValidatorMixin
from passport.backend.api.views.bundle import exceptions
from passport.backend.api.views.bundle.account.forms import (
    AccountDeleteCheckAnswerFormV2,
    AccountDeleteConfirmEmailFormV2,
    AccountDeleteConfirmPhoneFormV2,
    AccountDeleteForm,
    AccountDeleteSendEmailCodeFormV2,
)
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import (
    AccountCompletionRequiredError,
    AccountHasBlockingSIDs,
    ActionNotRequiredError,
    BaseBundleError,
    CodeInvalidError,
    EmailConfirmationsLimitExceededError,
    InvalidTrackStateError,
    PasswordRequiredError,
    UserNotVerifiedError,
)
from passport.backend.api.views.bundle.headers import (
    HEADER_CLIENT_USER_AGENT,
    HEADER_CONSUMER_CLIENT_IP,
)
from passport.backend.api.views.bundle.mixins import (
    BundleAccountFlushMixin,
    BundleAccountGetterMixin,
    BundleAccountPropertiesMixin,
    BundleAccountResponseRendererMixin,
    BundleAssertCaptchaMixin,
    BundleEditSessionMixin,
    BundleFamilyMixin,
    BundleHintAnswerCheckMixin,
    BundlePhoneMixin,
    BundleTrackedPhoneConfirmationMixin,
)
from passport.backend.api.views.bundle.mixins.kiddish import BundleKiddishMixin
from passport.backend.api.views.bundle.restore.exceptions import (
    AnswerNotMatchedError,
    PhoneChangedError,
)
from passport.backend.core.builders.blackbox.constants import BLACKBOX_EDITSESSION_OP_DELETE
from passport.backend.core.builders.mail_apis.husky import drop_mailbox
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.loggers.statbox import StatboxLogger
from passport.backend.core.mailer.utils import (
    get_tld_by_country,
    login_shadower,
    MailInfo,
    make_email_context,
    render_to_sendmail,
)
from passport.backend.core.models.account import (
    ACCOUNT_DISABLED_ON_DELETION,
    AccountDeletionOperation,
    get_preferred_language,
    MAIL_STATUS_FROZEN,
)
from passport.backend.core.models.delete_tasks import PhoneBindingsHistoryDeleteTask
from passport.backend.core.models.email import Email
from passport.backend.core.runner.context_managers import (
    CREATE,
    DELETE,
    UPDATE,
)
from passport.backend.core.services import Service
from passport.backend.core.subscription import get_blocking_sids
from passport.backend.core.types.email.email import mask_email_for_statbox
from passport.backend.core.utils.decorators import cached_property
from passport.backend.core.utils.domains import get_keyspace_by_host
from passport.backend.utils.common import generate_random_code
from passport.backend.utils.time import datetime_to_unixtime


log = logging.getLogger('passport.api.view.bundle.account.deletion')


ACCOUNT_DELETION_BASE_GRANT = 'account.delete'
ACCOUNT_DELETION_ANY_GRANT = 'account.delete_any'
ACCOUNT_DELETION_PDD_GRANT = 'account.delete_pdd'
ACCOUNT_DELETION_KINOPOISK_GRANT = 'account.delete_kinopoisk'
ACCOUNT_DELETION_SCHOLAR_GRANT = 'account.delete_scholar'
ACCOUNT_DELETION_FEDERAL_GRANT = 'account.delete_federal'
ACCOUNT_DELETION_BY_CREDENTIALS_GRANT = 'account.delete_by_credentials'

DELETION_REQUIREMENT_PASSWORD = 'password_required'
DELETION_REQUIREMENT_PHONE = 'phone_required'
DELETION_REQUIREMENT_HINT = 'question_required'
DELETION_REQUIREMENT_EMAIL = 'email_required'


class AccountDeleteBaseViewV1(
    BundleKiddishMixin,
    BundleFamilyMixin,
    BundleAccountFlushMixin,
    BundleAccountGetterMixin,
    BaseBundleView,
):
    require_track = False

    def __init__(self):
        super(AccountDeleteBaseViewV1, self).__init__()

        # Аккаунты-ребёнкиши из семьи
        self.kids = None

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode='account_delete',
            consumer=self.consumer,
            ip=self.client_ip,
            user_agent=self.user_agent,
        )

    def drop_mailbox_if_nessesary(self):
        mail_sid = Service.by_slug('mail').sid
        if self.account.has_sid(mail_sid) or self.account.mail_status == MAIL_STATUS_FROZEN:
            # От 2 сида не отписываем, так как ПДДшник без него не функционирует нормально. Но почистить ящик нужно.
            drop_mailbox(self.account.uid)

    def load_family_info_and_kids(self):
        if self.account.has_family:
            self.load_family_info_by_family_id(self.account.family_info.family_id)
            if str(self.account.family_info.admin_uid) == str(self.account.uid):
                self.load_kids()

    def disable_account_till_blocking_sids_exist(self):
        with UPDATE(self.account, self.request.env, self.events):
            self.account.disabled_status = ACCOUNT_DISABLED_ON_DELETION
            self.statbox.log(action='blocked_on_delete')

        self.response_values['blocking_sids'] = list(self.sids_blocking_deletion)
        raise AccountHasBlockingSIDs()

    def delete_account(self):
        if self.account.has_family:
            family_events = {
                'consumer': self.consumer,
            }
            if str(self.account.family_info.admin_uid) == str(self.account.uid):
                self.start_kids_deletion()
                family_events['action'] = 'family_admin_account_delete'
                with DELETE(self.family_info, self.request.env, family_events):
                    pass
            else:
                family_events['action'] = 'family_member_account_delete'
                with UPDATE(self.family_info, self.request.env, family_events):
                    self.family_info.remove_member_uid(self.account.uid)

        phone_history_delete_task = PhoneBindingsHistoryDeleteTask(
            uid=self.account.uid,
            deletion_started_at=datetime.now(),
        )
        with CREATE(phone_history_delete_task, self.request.env, {}):
            pass

        with DELETE(self.account, self.request.env, self.events):
            pass

        self.drop_social_profiles()
        self.statbox.log(action='deleted')

    @cached_property
    def sids_blocking_deletion(self):
        sids_on_account = set(self.account.subscriptions.keys())
        return sids_on_account.intersection(get_blocking_sids(self.account))

    @cached_property
    def events(self):
        return dict(action='account_delete', consumer=self.consumer)


class AccountDeleteView(AccountDeleteBaseViewV1):
    required_grants = [
        ACCOUNT_DELETION_BASE_GRANT,
    ]

    grants_for_account_type = {
        'any': ACCOUNT_DELETION_ANY_GRANT,
        'pdd': ACCOUNT_DELETION_PDD_GRANT,
        'kinopoisk': ACCOUNT_DELETION_KINOPOISK_GRANT,
        'scholar': ACCOUNT_DELETION_SCHOLAR_GRANT,
        'federal': ACCOUNT_DELETION_FEDERAL_GRANT,
    }

    basic_form = AccountDeleteForm

    def process_request(self, *args, **kwargs):
        self.process_basic_form()
        self.statbox.log(action='submitted')

        self.get_account_by_uid(
            uid=self.form_values['uid'],
            need_phones=True,
            get_family_info=True,
        )

        self.check_grants_for_account_type()

        if not self.sids_blocking_deletion:
            self.load_family_info_and_kids()

        self.drop_mailbox_if_nessesary()

        if self.sids_blocking_deletion:
            self.disable_account_till_blocking_sids_exist()
        else:
            self.delete_account()

    @cached_property
    def statbox(self):
        sbx = super(AccountDeleteView, self).statbox
        sbx.bind_context(uid=self.form_values['uid'])
        return sbx


class AccountDeleteByCredentialsView(AccountDeleteBaseViewV1):
    required_grants = [ACCOUNT_DELETION_BY_CREDENTIALS_GRANT]

    def process_request(self, *args, **kwargs):
        self.statbox.log(action='submitted')

        self.get_account_by_oauth_token(
            get_family_info=True,
            whitelisted_client_ids={a['client_id'] for a in settings.OAUTH_APPLICATIONS_FOR_MUSIC.values()},
        )

        self.check_allowed_to_delete_account_by_token()

        if not self.sids_blocking_deletion:
            self.load_family_info_and_kids()

        self.drop_mailbox_if_nessesary()

        if self.sids_blocking_deletion:
            self.disable_account_till_blocking_sids_exist()
        else:
            self.delete_account()

    def check_allowed_to_delete_account_by_token(self):
        if not self.account.is_social:
            raise exceptions.AccountInvalidTypeError()

        profiles = self.social_api.get_profiles(allow_auth=True, uid=self.account.uid)
        if not (profiles and profiles[0].get('provider_code') == 'mt'):
            raise exceptions.AccountInvalidTypeError()


class AccountDeleteBaseViewV2(
    BundleAccountResponseRendererMixin,
    BundleAccountPropertiesMixin,
    BundleAccountGetterMixin,
    BundleAssertCaptchaMixin,
    BundlePhoneMixin,
    BaseBundleView,
):
    required_headers = [
        HEADER_CONSUMER_CLIENT_IP,
        HEADER_CLIENT_USER_AGENT,
    ]

    require_track = True
    require_process = True
    process_name = PROCESS_ACCOUNT_DELETE_V2

    required_grants = [ACCOUNT_DELETION_BASE_GRANT]
    grants_for_account_type = {'any': ACCOUNT_DELETION_ANY_GRANT}

    def _get_account(self):
        self._check_track_is_personal()
        kwargs = self._kwargs_for_get_account()
        self.get_pinned_account_from_session(**kwargs)
        self.statbox.bind_context(uid=self.account.uid)

    def _kwargs_for_get_account(self):
        return dict(
            need_phones=True,
            emails=True,
            email_attributes='all',
        )

    def _get_what_required_for_deletion(self, with_details=False):
        requirements = {}

        is_password_verification_required, password_requirements = self._get_password_requirements(with_details)
        if is_password_verification_required:
            requirements.update(password_requirements)

        for get_requirements in [
            self._get_phone_requirements,
            self._get_email_requirements,
            self._get_hint_requirements,
        ]:
            is_suitable, extra_requirements = get_requirements(with_details)
            if is_suitable:
                requirements.update(extra_requirements)
                break
        return requirements

    def _get_password_requirements(self, with_details):
        if self.is_password_verification_required():
            requirements = {DELETION_REQUIREMENT_PASSWORD: None}
        else:
            requirements = {}
        return (True, requirements)

    @cached_property
    def is_fresh_account(self):
        if self.account.registration_datetime:
            timedelta_since_registration = datetime.now() - self.account.registration_datetime
            return timedelta_since_registration <= settings.ACCOUNT_FRESH_PERIOD
        return False

    def _get_phone_requirements(self, with_details):
        requirements = {}
        suitable_phone = None
        if (
            self.account.phones.secure and
            (
                self.is_fresh_account or
                datetime.now() - self.account.phones.secure.bound > settings.PHONE_SUITABLE_FOR_ACCOUNT_DELETION_SINCE_TIME
            )
        ):
            suitable_phone = self.account.phones.secure

        if suitable_phone:
            confirmed_phone_number = self.is_phone_confirmed_in_track() and self.track.phone_confirmation_phone_number
            if suitable_phone.number.e164 != confirmed_phone_number:
                requirements[DELETION_REQUIREMENT_PHONE] = None

                if with_details:
                    phones = self.phones_response()
                    requirements[DELETION_REQUIREMENT_PHONE] = {'phone': phones[suitable_phone.id]}
        return bool(suitable_phone), requirements

    def _get_email_requirements(self, with_details):
        requirements = {}
        now = datetime.now()
        suitable_emails = []
        for email in self.account.emails.suitable_for_restore:
            if self.is_fresh_account or (now - email.bound_at > settings.EMAIL_SUITABLE_FOR_ACCOUNT_DELETION_SINCE_TIME):
                suitable_emails.append(email)

        if suitable_emails:
            confirmed_email = self.track.email_confirmation_passed_at and self.track.email_confirmation_address
            if confirmed_email:
                confirmed_email = Email.normalize_address(confirmed_email)

            if all(confirmed_email != e.normalized_address for e in suitable_emails):
                requirements[DELETION_REQUIREMENT_EMAIL] = None

                if with_details:
                    requirements[DELETION_REQUIREMENT_EMAIL] = {
                        'emails': [{'address': e.address} for e in suitable_emails],
                    }
        return bool(suitable_emails), requirements

    def _get_hint_requirements(self, with_details):
        requirements = {}
        if self.account.hint.is_set:
            if not self.track.is_secure_hint_answer_checked:
                requirements[DELETION_REQUIREMENT_HINT] = None

                if with_details:
                    requirements[DELETION_REQUIREMENT_HINT] = {
                        'question': {
                            'id': self.account.hint.question.id,
                            'text': self.account.hint.question.text,
                        },
                    }
        return self.account.hint.is_set, requirements

    def _check_deletion_possible(self):
        if self.account.is_pdd:
            # Запрещаем удаление всем ПДД-пользователям, включая пользователей Коннекта
            raise UnableToDeletePddAccount()

        # Админу ПДД удаление запрещаем, но отдаем специфичную ошибку.
        if self.account.is_pdd_admin:
            raise UnableToDeleteAdmin()

        if not (
            (self.account.is_normal or self.account.is_lite) and
            self.account.have_password
        ):
            raise AccountCompletionRequiredError()

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode='account_delete',
            version='2',
            consumer=self.consumer,
            ip=self.client_ip,
            user_agent=self.user_agent,
            step=self.step_name,
        )

    def _check_track_is_personal(self):
        if not self.track.uid:
            raise InvalidTrackStateError()


class AccountDeleteCommitViewV2(BundleEditSessionMixin, AccountDeleteBaseViewV2):
    step_name = 'commit'

    def process_request(self):
        self.statbox.log(action='submitted')
        self.read_track()
        self.response_values.update({'track_id': self.track_id})

        self._check_captcha_passed()

        self._get_account()
        self.check_grants_for_account_type()
        self.fill_response_with_account()

        self._check_deletion_possible()
        self._check_deletion_authorized()
        self._start_deletion()
        self._delete_account_from_session()

        self.statbox.log(action='committed')

    def _check_captcha_passed(self):
        """
        Проверяет роботность пользователя, чтобы не допустить массовое удаление
        аккаунтов.
        """
        with self.track_transaction.rollback_on_error():
            self.track.is_captcha_required = True
        self.check_track_for_captcha()

    def _check_deletion_authorized(self):
        """
        Проверяет, что пользователь действительный владелец аккаунта.
        """
        requirements = self._get_what_required_for_deletion()
        for requirement in requirements:
            if requirement != DELETION_REQUIREMENT_PASSWORD:
                raise UserNotVerifiedError()

        if DELETION_REQUIREMENT_PASSWORD in requirements:
            raise PasswordRequiredError()

    def _start_deletion(self):
        with UPDATE(
            self.account,
            self.request.env,
            {
                'action': 'account_delete',
                'consumer': self.consumer,
            },
        ):
            timestamp = datetime.now()
            self.account.disabled_status = ACCOUNT_DISABLED_ON_DELETION
            if not self.account.deletion_operation:
                self.account.deletion_operation = AccountDeletionOperation(self, started_at=timestamp)
            self.account.global_logout_datetime = timestamp

    def _delete_account_from_session(self):
        with self.track_transaction.rollback_on_error():
            cookies, default_uid = self.edit_session(
                BLACKBOX_EDITSESSION_OP_DELETE,
                self.account.uid,
                self.session_info,
                statbox_params={},
            )
        self.response_values.update(cookies=cookies)
        if default_uid:
            self.response_values.update(default_uid=default_uid)


class AccountDeleteSubmitViewV2(AccountDeleteBaseViewV2):
    step_name = 'submit'

    def process_request(self):
        self.statbox.log(action='submitted')

        self._init_track()

        self._get_account()
        self.check_grants_for_account_type()
        self.fill_response_with_account()

        self._check_deletion_possible()

        requirements = self._get_what_required_for_deletion(with_details=True)
        self.response_values.update({'requirements': requirements})

        self._fill_response_with_services()
        self.statbox.log(action='committed')

    def _init_track(self):
        self.read_track()
        self.response_values.update({'track_id': self.track_id})
        with self.track_transaction.rollback_on_error():
            self.track.is_captcha_required = True

    def _fill_response_with_services(self):
        services = []
        for subscription in self.account.subscriptions.values():
            service = subscription.service
            if service.is_real:
                services.append({
                    'sid': service.sid,
                    'is_blocking': service.sid in get_blocking_sids(self.account),
                })
        self.response_values['subscribed_to'] = services


class AccountDeleteBasePhoneConfirmationViewV2(BundleTrackedPhoneConfirmationMixin, AccountDeleteBaseViewV2):
    def process_request(self):
        if self.basic_form:
            self.process_basic_form()

        self.read_track()
        self.response_values.update({'track_id': self.track_id})

        self.check_track_for_captcha()

        self._get_account()
        self.check_grants_for_account_type()
        self.fill_response_with_account()

        self._check_deletion_possible()
        self._check_phone_confirmation_required()

    def _check_phone_confirmation_required(self):
        if DELETION_REQUIREMENT_PHONE not in self._get_what_required_for_deletion():
            raise ActionNotRequiredError()


class AccountDeleteSendPhoneCodeViewV2(AccountDeleteBasePhoneConfirmationViewV2):
    step_name = 'send_phone_code'

    def process_request(self):
        self.statbox.log(action='submitted')

        super(AccountDeleteSendPhoneCodeViewV2, self).process_request()

        phone = self.account.phones.secure.number
        old_phone = self.track.phone_confirmation_phone_number

        with self.track_transaction.commit_on_error():
            if old_phone and old_phone != phone.e164:
                self.confirmation_info.reset_phone()
                self.confirmation_info.save()
            self.track.phone_confirmation_phone_number = phone.e164
            try:
                self.send_code(phone, account=self.account)
                self.response_values['resend_timeout'] = self.confirmation_code_resend_timeout
            finally:
                self.statbox.dump_stashes()

        self.statbox.log(action='committed')


class AccountDeleteConfirmPhoneViewV2(AccountDeleteBasePhoneConfirmationViewV2):
    basic_form = AccountDeleteConfirmPhoneFormV2
    step_name = 'confirm_phone'

    def process_request(self):
        self.statbox.log(action='submitted')

        super(AccountDeleteConfirmPhoneViewV2, self).process_request()

        if not self.track.phone_confirmation_code:
            raise InvalidTrackStateError()

        if self.track.phone_confirmation_phone_number != self.account.phones.secure.number.e164:
            raise PhoneChangedError()

        with UPDATE(self.account, self.request.env, {'action': 'confirm_phone', 'consumer': self.consumer}), self.track_transaction.commit_on_error():
            self.confirm_code(self.form_values['code'])

        self.statbox.log(action='committed')


class AccountDeleteCheckAnswerViewV2(AccountDeleteBaseViewV2, BundleHintAnswerCheckMixin):
    basic_form = AccountDeleteCheckAnswerFormV2
    step_name = 'check_answer'

    def process_request(self):
        self.statbox.log(action='submitted')
        self.process_basic_form()

        self.read_track()
        self.response_values.update({'track_id': self.track_id})

        self.check_track_for_captcha()

        self._get_account()
        self.check_grants_for_account_type()
        self.fill_response_with_account()

        self._check_deletion_possible()

        if DELETION_REQUIREMENT_HINT not in self._get_what_required_for_deletion():
            raise ActionNotRequiredError()

        compare_status = self.compare_answers()
        with self.track_transaction.commit_on_error():
            if not compare_status:
                self.track.answer_checks_count.incr()
                self.response_values['is_captcha_required'] = False
                if self._is_captcha_required_for_hint():
                    self.invalidate_captcha()
                    self.response_values['is_captcha_required'] = True
                raise AnswerNotMatchedError()
            self.track.is_secure_hint_answer_checked = True

        self.statbox.log(action='committed')

    def _is_captcha_required_for_hint(self):
        return self.track.answer_checks_count.get(default=0) >= settings.ANSWER_CHECK_ERRORS_CAPTCHA_THRESHOLD


class AccountDeleteBaseEmailConfirmationViewV2(AccountDeleteBaseViewV2):
    def process_request(self):
        self.process_basic_form()
        self.read_track()
        self.response_values.update({'track_id': self.track_id})

        self.check_track_for_captcha()

        self._get_account()
        self.check_grants_for_account_type()
        self.fill_response_with_account()

        self._check_deletion_possible()
        self._check_email_confirmation_required()

    def _check_email_confirmation_required(self):
        if DELETION_REQUIREMENT_EMAIL not in self._get_what_required_for_deletion():
            raise ActionNotRequiredError()

    def _check_email_suitable(self, email_address):
        try:
            email_address = Email.normalize_address(email_address)
        except ValueError:
            raise EmailNotSuitable()
        try:
            email = self.account.emails[email_address]
        except KeyError:
            raise EmailNotSuitable()
        if not email.is_suitable_for_restore:
            raise EmailNotSuitable()


class AccountDeleteSendEmailCodeViewV2(AccountDeleteBaseEmailConfirmationViewV2, EmailValidatorMixin):
    basic_form = AccountDeleteSendEmailCodeFormV2
    step_name = 'send_email_code'

    def process_request(self):
        self.statbox.log(action='submitted')
        super(AccountDeleteSendEmailCodeViewV2, self).process_request()
        self._check_email_suitable(self.form_values['email'])
        self._send_confirmation_code()
        self.statbox.log(action='committed')

    def _send_confirmation_code(self):
        email = self.form_values['email']
        self._check_email_counters(email)

        with self.track_transaction.rollback_on_error():
            if self._is_confirmation_email_changed(email):
                self._reset_confirmation_state(email)

            self._reset_confirmation_code_if_required()

        self._increase_email_counters(email)

        message = self._build_message({'email': email, 'confirmation_code': self.track.email_confirmation_code})
        render_to_sendmail(**message)

        self.statbox.log(
            action='email_sent',
            email=mask_email_for_statbox(self.form_values['email']),
            identity='confirmation_code_for_account_deletion',
        )

    def _check_email_counters(self, email):
        if self._email_counters_hit_limit(email):
            raise EmailLimitExceeded()

    def _is_confirmation_email_changed(self, email):
        return (
            not self.track.email_confirmation_address or
            Email.normalize_address(self.track.email_confirmation_address) != Email.normalize_address(email)
        )

    def _reset_confirmation_state(self, email):
        self.track.email_confirmation_address = email
        self.track.email_confirmation_code = self._generate_confirmation_code()
        self.track.email_confirmation_passed_at = None
        self.track.email_confirmation_checks_count.reset()

    def _reset_confirmation_code_if_required(self):
        if not self.track.email_confirmation_code or self.track.email_confirmation_checks_count.get(default=0) >= settings.ALLOWED_EMAIL_SHORT_CODE_FAILED_CHECK_COUNT:
            self.track.email_confirmation_code = self._generate_confirmation_code()
            self.track.email_confirmation_checks_count.reset()

    def _generate_confirmation_code(self):
        return generate_random_code(settings.EMAIL_VALIDATOR_SHORT_CODE_LENGTH)

    def _build_message(self, values):
        language = get_preferred_language(self.account)
        PHRASES = settings.translations.NOTIFICATIONS[language]
        mail_meta_info = MailInfo(
            subject=PHRASES['confirmation_code_for_account_deletion_email_subject'],
            from_=PHRASES['email_sender_display_name'],
            tld=get_tld_by_country(self.account.person.country),
        )
        passport_domain = '%s.%s' % (settings.PASSPORT_SUBDOMAIN, get_keyspace_by_host(self.request.env.host))
        template_context = make_email_context(
            language=language,
            account=self.account,
            context={
                'SHORT_CODE': values['confirmation_code'],
                'RESTORE_URL': 'https://%s/restoration' % passport_domain,
                'PROFILE_URL': 'https://%s/profile' % passport_domain,
            },
        )
        template_context = login_shadower(template_context)

        return dict(
            template_name='mail/confirmation_code_for_account_deletion.html',
            info=mail_meta_info,
            recipients=[values['email']],
            context=template_context,
        )


class AccountDeleteConfirmEmailViewV2(AccountDeleteBaseEmailConfirmationViewV2):
    basic_form = AccountDeleteConfirmEmailFormV2
    step_name = 'confirm_email'

    def process_request(self):
        self.statbox.log(action='submitted')

        super(AccountDeleteConfirmEmailViewV2, self).process_request()

        self._check_email_suitable(self.track.email_confirmation_address)
        self._check_confirmation_code()
        self._confirm_email()

        self.statbox.log(action='committed')

    def _check_confirmation_code(self):
        if not (self.track.email_confirmation_address and self.track.email_confirmation_code):
            raise InvalidTrackStateError()

        if self.track.email_confirmation_checks_count.get(default=0) >= settings.ALLOWED_EMAIL_SHORT_CODE_FAILED_CHECK_COUNT:
            raise EmailConfirmationsLimitExceededError()

        with self.track_transaction.commit_on_error():
            self.track.email_confirmation_checks_count.incr()
            if self.track.email_confirmation_code != self.form_values['code']:
                raise CodeInvalidError()

    def _confirm_email(self):
        timestamp = datetime.now()

        with self.track_transaction.rollback_on_error():
            self.track.email_confirmation_passed_at = datetime_to_unixtime(timestamp)

        email = self.account.emails[self.track.email_confirmation_address]
        with UPDATE(self.account, self.request.env, {'action': 'confirm_email', 'consumer': self.consumer}):
            email.confirmed_at = timestamp


class UnableToDeleteAdmin(BaseBundleError):
    error = 'unable_to_delete_pdd_admin'


class UnableToDeletePddAccount(BaseBundleError):
    error = 'unable_to_delete_pdd_account'


class EmailNotSuitable(BaseBundleError):
    error = 'email.not_suitable'


class EmailLimitExceeded(BaseBundleError):
    error = 'email_messages_limit.exceeded'
