# -*- coding: utf-8 -*-
from time import time

from passport.backend.api.common.errors import log_internal_error
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.constants import X_TOKEN_OAUTH_SCOPE
from passport.backend.api.views.bundle.mixins import BundleAccountGetterMixin
from passport.backend.core.builders.datasync_api import (
    BaseDatasyncApiError,
    DatasyncApiObjectNotFoundError,
    DatasyncApiTemporaryError,
    PersonalityApi,
)
from passport.backend.core.conf import settings
from passport.backend.core.utils.decorators import cached_property


EXTERNAL_INFO_GRANT = 'account.external_info'

DEFAULT_CACHE_TTL = 60


class BaseExternalInfoView(BaseBundleView, BundleAccountGetterMixin):
    require_track = False
    required_grants = [EXTERNAL_INFO_GRANT]
    basic_form = None

    oauth_scope = X_TOKEN_OAUTH_SCOPE

    external_service_errors = None

    use_cache = False
    cache_name = None
    cache_ttl = DEFAULT_CACHE_TTL

    @cached_property
    def personality_api_for_cache(self):
        """Отличается пониженными таймаутами и ретраями"""
        return PersonalityApi(
            timeout=settings.DATASYNC_CACHE_TIMEOUT,
            retries=settings.DATASYNC_CACHE_RETRIES,
        )

    def process_request(self):
        if self.basic_form is not None:
            self.process_basic_form()
        self.get_account_from_session_or_oauth_token(
            required_scope=self.oauth_scope,
            get_user_ticket=True,
            get_public_id=True,
        )
        data = self.try_get_data_from_cache_or_external_service_and_update_cache()
        self.response_values.update(data)

    def try_get_data_from_cache_or_external_service_and_update_cache(self):
        is_cached_data_up_to_date, cached_data = self.try_get_data_from_cache()
        if is_cached_data_up_to_date:
            return cached_data

        try:
            data = self.get_external_data()
        except tuple(self.external_service_errors):
            if cached_data:
                # лучше отдать старые данные, чем ошибку
                data = cached_data
            else:
                raise
        else:
            self.try_update_cache(data)

        return data

    def try_get_data_from_cache(self):
        """Возвращает пару (is_cache_contents_up_to_date, data)"""
        cache_contents = None
        if self.use_cache:
            assert self.cache_name, 'Cache name must be specified'
            try:
                cache_contents = self.personality_api_for_cache.passport_external_data_get(
                    uid=self.account.uid,
                    object_id=self.cache_name,
                )
            except (DatasyncApiTemporaryError, DatasyncApiObjectNotFoundError):
                pass
            except BaseDatasyncApiError as e:
                log_internal_error(e)

        is_cache_contents_applicable = (
            cache_contents and
            cache_contents.get('meta', {}).get('params', {}) == self.form_values
        )
        if is_cache_contents_applicable:
            is_cache_contents_up_to_date = time() - cache_contents['modified_at'] <= self.cache_ttl
            return is_cache_contents_up_to_date, cache_contents['data']
        else:
            return False, None

    def try_update_cache(self, data):
        if self.use_cache:
            try:
                self.personality_api_for_cache.passport_external_data_update(
                    uid=self.account.uid,
                    object_id=self.cache_name,
                    data=data,
                    meta={'params': self.form_values} if self.form_values else None,
                    modified_at=time(),
                )
            except DatasyncApiTemporaryError:
                pass
            except BaseDatasyncApiError as e:
                log_internal_error(e)

    def get_external_data(self):
        raise NotImplementedError()  # pragma: no cover
