# -*- coding: utf-8 -*-
from flask import request
from passport.backend.api.views.bundle.account.external_data.base import BaseExternalInfoView
from passport.backend.api.views.bundle.account.external_data.forms import (
    AfishaFavouritesForm,
    BiletApiOrderInfoForm,
    PaginationForm,
)
from passport.backend.core.builders.afisha import (
    BaseAfishaApiError,
    get_afisha_api,
)
from passport.backend.core.builders.bilet_api import (
    BaseBiletApiError,
    get_bilet_api,
)
from passport.backend.core.conf import settings
from passport.backend.core.geobase import Region
from passport.backend.utils.common import (
    get_default_if_none,
    remove_none_values,
)
from six.moves.urllib.parse import (
    urljoin,
    urlsplit,
    urlunsplit,
)
from werkzeug.utils import cached_property


class BaseBiletApiView(BaseExternalInfoView):
    oauth_scope = 'afisha:all'
    external_service_errors = (BaseBiletApiError,)

    @cached_property
    def bilet_api(self):
        return get_bilet_api()

    @property
    def common_args(self):
        return dict(
            user_ip=self.client_ip,
            user_agent=self.user_agent,
            request_id=request.env.request_id,
            yandexuid=self.cookies.get('yandexuid'),
            sessionid=self.cookies.get('Session_id'),
            host=self.host,
            oauth_token=self.oauth_token if self.authorization else None,
        )


class BiletApiOrdersView(BaseBiletApiView):
    basic_form = PaginationForm
    use_cache = False  # TODO: Чтобы кэшировать, надо сократить ответ

    def get_external_data(self):
        return self.bilet_api.orders(
            limit=self.form_values['page_size'],
            offset=self.form_values['page_size'] * (self.form_values['page'] - 1),
            sort='session-date',
            approved_only=True,
            **self.common_args
        )


class BiletApiOrderInfoView(BaseBiletApiView):
    basic_form = BiletApiOrderInfoForm
    use_cache = False  # Кэшировать бессмысленно - ручка вызывается много раз для разных заказов

    def get_external_data(self):
        order_data = self.bilet_api.order(
            order_id=self.form_values['order_id'],
            **self.common_args
        )
        images = order_data.get('order', {}).get('session', {}).get('event', {}).get('images', [])
        for image in images:
            url = image.get('url', '')
            split_url = urlsplit(url)
            if split_url.scheme == 'http':
                image.update(url=urlunsplit(['https'] + list(split_url[1:])))
        return order_data


class BiletApiUserPointsView(BaseBiletApiView):
    use_cache = True
    cache_name = 'afisha_user_points'

    def get_external_data(self):
        return self.bilet_api.user_points(
            **self.common_args
        )


class AfishaFavouritesView(BaseExternalInfoView):
    use_cache = True
    cache_name = 'afisha_favourite_events'
    basic_form = AfishaFavouritesForm
    external_service_errors = (BaseAfishaApiError,)

    @cached_property
    def afisha_api(self):
        return get_afisha_api()

    def get_external_data(self):
        region = Region(ip=self.client_ip)
        result = []
        next_page = None

        if region.city:
            city_id = region.city['id']

            events = self.afisha_api.events(uid=self.account.uid)
            events = events['events']
            event_ids = [v['event']['id'] for v in events if not v['isVisited']]

            events_len = len(event_ids)

            start = self.form_values['page_size'] * (self.form_values['page'] - 1)
            end = start + self.form_values['page_size']

            if events_len > start:
                # Сортируем по убыванию timestamp монговского ObjectId
                event_ids.sort(reverse=True)

                events_schedules = self.afisha_api.events_schedules(
                    city_id=city_id,
                    event_ids=event_ids[start:end],
                    period=self.form_values['period'],
                )
                for ev in events_schedules:
                    only_place = ev['scheduleInfo'].get('onlyPlace') or {}
                    only_date = ev['scheduleInfo'].get('regularity', {}).get('singleShowtime')
                    event_info = {
                        'id': ev['event']['id'],
                        'title': ev['event'].get('title'),
                        'type': ev['event'].get('type', {}).get('code'),
                        'type_name': ev['event'].get('type', {}).get('name'),
                        'url': urljoin(settings.AFISHA_FRONT_URL, ev['event'].get('url')),
                        'image': get_default_if_none(ev['event'], 'image', {}).get('base_url'),
                        'only_place': only_place.get('title'),
                        'places_total': ev['scheduleInfo'].get('placesTotal') if not only_place else None,
                        'single_date': only_date,
                        'date_end': ev['scheduleInfo'].get('dateEnd') if not only_date else None,
                    }
                    event_info = remove_none_values(event_info)
                    result.append(event_info)
            if events_len > start + end:
                next_page = self.form_values['page'] + 1
        return {
            'event_favourites': result,
            'next_page': next_page,
        }
