# -*- coding: utf-8 -*-
from passport.backend.api.common.errors import log_internal_error
from passport.backend.api.views.bundle.account.external_data.base import BaseExternalInfoView
from passport.backend.api.views.bundle.account.external_data.forms import PaginationForm
from passport.backend.core.builders.collections import get_collections_api
from passport.backend.core.builders.collections.exceptions import BaseCollectionsApiError
from werkzeug.utils import cached_property


class BaseCollectionsView(BaseExternalInfoView):
    external_service_errors = (BaseCollectionsApiError, )

    @cached_property
    def collections_api(self):
        return get_collections_api()

    @property
    def common_collections_api_args(self):
        return dict(
            sessionid=self.cookies.get('Session_id'),
            host=self.host,
        )


class PicturesCollectionsView(BaseCollectionsView):
    basic_form = PaginationForm
    use_cache = True
    cache_name = 'pictures_collections'

    def get_external_data(self):
        response = self.collections_api.collections(
            public_id=self.account.public_id,
            page=self.form_values['page'],
            page_size=self.form_values['page_size'],
            previews_count=1,
            **self.common_collections_api_args
        )
        collections = []
        for item in response:
            try:
                collections.append({
                    'id': item['id'],
                    'slug': item['slug'],
                    'title': item['title'],
                    'url': item['url'],
                    'preview': {
                        'id': item['card_previews'][0]['id'],
                        'mds_key': item['card_previews'][0]['image']['avatars_key'],
                        'mds_group_id': item['card_previews'][0]['image']['group_id'],
                    } if item.get('card_previews') else None,
                })
            except (KeyError, IndexError) as e:
                log_internal_error(e, error_details=repr(response))
        return {
            'collections': collections,
        }


class PicturesFavouritesView(BaseCollectionsView):
    basic_form = PaginationForm
    use_cache = True
    cache_name = 'pictures_favourites'

    def get_external_data(self):
        response = self.collections_api.pictures(
            public_id=self.account.public_id,
            page=self.form_values['page'],
            page_size=self.form_values['page_size'],
            **self.common_collections_api_args
        )
        pictures = []
        for item in response:
            try:
                pictures.append({
                    'id': item['content'][0]['id'],
                    'mds_key': item['content'][0]['content']['avatars_key'],
                    'mds_group_id': item['content'][0]['content']['group_id'],
                })
            except (KeyError, IndexError) as e:
                log_internal_error(e)
        return {
            'pictures': pictures,
        }
