# -*- coding: utf-8 -*-
from passport.backend.api.views.bundle.account.external_data.base import BaseExternalInfoView
from passport.backend.api.views.bundle.account.external_data.exceptions import DiskNotFoundError
from passport.backend.api.views.bundle.account.external_data.forms import PaginationForm
from passport.backend.api.views.bundle.exceptions import AccountInvalidTypeError
from passport.backend.core.builders.datasync_api import (
    BaseDatasyncApiError,
    DatasyncAccountInvalidTypeError,
    DatasyncApiObjectNotFoundError,
    DatasyncUserBlockedError,
    get_disk_api,
)
from passport.backend.core.services import Service
from werkzeug.utils import cached_property


class BaseDiskView(BaseExternalInfoView):
    oauth_scope = 'cloud_api:disk.info'
    external_service_errors = (BaseDatasyncApiError, )
    use_cache = False  # Данные уже берутся из DataSync

    @cached_property
    def disk_api(self):
        return get_disk_api()

    def check_user(self):
        # Проверим, может ли пользователь в принципе пользоваться Диском
        if not (self.account.is_normal or self.account.is_pdd):
            raise AccountInvalidTypeError()
        # Если пользователь не подписан на дисковый (59) сид - в апи можно не идти, всё и так понятно
        if not self.account.is_subscribed(Service.by_slug('cloud')):
            raise DiskNotFoundError()


class DiskInfoView(BaseDiskView):
    def get_external_data(self):
        self.check_user()
        try:
            return self.disk_api.disk_info(uid=self.account.uid)
        except DatasyncApiObjectNotFoundError:
            raise DiskNotFoundError()
        except (DatasyncAccountInvalidTypeError, DatasyncUserBlockedError):
            raise AccountInvalidTypeError()


class DiskSubscriptionsView(BaseDiskView):
    basic_form = PaginationForm

    def get_external_data(self):
        self.check_user()
        return {
            'subscriptions': self.disk_api.billing_subscriptions(
                uid=self.account.uid,
                user_ip=self.client_ip,
                limit=self.form_values['page_size'],
                offset=self.form_values['page_size'] * (self.form_values['page'] - 1),
            ),
        }
