# -*- coding: utf-8 -*-

from datetime import datetime
import logging

from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import AccountInvalidTypeError
from passport.backend.api.views.bundle.mixins import BundleAccountGetterMixin
from passport.backend.core.logging_utils.loggers.statbox import StatboxLogger
from passport.backend.core.models.email import Email
from passport.backend.core.runner.context_managers import UPDATE
from passport.backend.core.types.display_name import DisplayName
from passport.backend.core.utils.decorators import cached_property

from .exceptions import AliasNotAvailableError
from .forms import FederalChangeForm


log = logging.getLogger('passport.api.view.bundle.account.federal_change')


class FederalChangeView(BaseBundleView,
                        BundleAccountGetterMixin,
                        ):
    required_grants = ['account.federal_change']

    basic_form = FederalChangeForm
    domains = None

    @cached_property
    def statbox(self):
        return StatboxLogger(
            uid=self.account.uid,
            mode='federal_change',
            consumer=self.consumer,
        )

    def hosted_domains(self, domain):
        if self.domains is None:
            self.domains = self.blackbox.hosted_domains(domain=domain)
        return self.domains

    def process_request(self, *args, **kwargs):
        self.process_basic_form()
        userinfo = self.blackbox.userinfo(
            uid=self.form_values['uid'],
            emails=True,
        )

        self.parse_account_with_domains(userinfo)
        if not self.account.is_federal:
            raise AccountInvalidTypeError()

        firstname = self.form_values['firstname']
        lastname = self.form_values['lastname']
        active = self.form_values['active']
        display_name = self.form_values['display_name']
        emails = self.form_values['emails']

        events = {
            'action': 'federal_change',
            'consumer': self.consumer,
        }

        emails_to_add = set()
        logins_to_add = set()
        emails_to_remove = set()
        logins_to_remove = set()
        if emails is not None:
            # чтоб не пытаться удалять/добавлять основной почтовый алиас, заранее исключим его из списка
            emails_set = set([email for email in emails if not email.lower() == self.account.pdd_alias.alias.lower()])
            existed_emails = ({email.address.lower() for email in self.account.emails.all} |
                              {('%s@%s' % (login, self.account.domain.domain)).lower() for login in self.account.pdd_alias.additional_logins})
            emails_to_add = {email for email in emails_set if email.lower() not in existed_emails}
            emails_to_remove = existed_emails - set(map(lambda x: x.lower(), emails_set))

            for email in emails_to_add.copy():
                login, domain = email.split('@', 1)
                if domain.lower() == self.account.domain.domain.lower():
                    availability_info = self.blackbox.loginoccupation(
                        [email],
                        is_pdd=True,
                    )
                    if availability_info[email]['status'] != 'free':
                        raise AliasNotAvailableError()
                    emails_to_add.remove(email)
                    logins_to_add.add(login)
            for email in emails_to_remove.copy():
                login, domain = email.split('@', 1)
                if domain.lower() == self.account.domain.domain and self.account.pdd_alias.has_login(login):
                    emails_to_remove.remove(email)
                    logins_to_remove.add(login)

        with UPDATE(self.account, self.request.env, events):
            if firstname is not None:
                self.account.person.firstname = firstname
            if lastname is not None:
                self.account.person.lastname = lastname
            if active is not None:
                self.account.is_enabled = active
            if display_name is not None:
                self.account.person.display_name = DisplayName(name=display_name) if display_name else None
            if emails_to_remove or emails_to_add or logins_to_add or logins_to_remove:
                for email in emails_to_remove:
                    self.account.emails.pop(email)
                for login in logins_to_remove:
                    self.account.pdd_alias.remove_login(login)
                for email in emails_to_add:
                    new_email = Email()
                    new_email.address = email
                    new_email.created_at = datetime.now()
                    self.account.emails.add(new_email)
                for login in logins_to_add:
                    self.account.pdd_alias.add_login(login)
