# -*- coding: utf-8 -*-
import logging

from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import InvalidTrackStateError
from passport.backend.api.views.bundle.mixins import (
    BundleAccountFlushMixin,
    BundleAccountGetterMixin,
    BundlePasswordChangeMixin,
    BundlePasswordValidationMixin,
)
from passport.backend.core.logging_utils.loggers import StatboxLogger
from passport.backend.core.models.password import PASSWORD_CHANGING_REASON_FLUSHED_BY_ADMIN
from passport.backend.core.runner.context_managers import UPDATE
from passport.backend.core.utils.decorators import cached_property

from .forms import (
    FlushPddCommitForm,
    FlushPddSubmitForm,
)


log = logging.getLogger('passport.api.view.bundle.account.flush_pdd')


class FlushPddBase(BaseBundleView, BundleAccountGetterMixin):
    track_type = 'authorize'

    required_grants = ['pdd.flush']

    @cached_property
    def statbox(self):
        return StatboxLogger(
            uid=self.account.uid,
            mode='flush_pdd',
            consumer=self.consumer,
        )

    def is_account_type_allowed(self):
        return self.account.is_pdd


class FlushPddSubmit(FlushPddBase):
    require_track = False
    basic_form = FlushPddSubmitForm

    def process_request(self):
        self.read_or_create_track(self.track_type)
        self.process_basic_form()

        self.get_account_by_uid(uid=self.all_values['uid'], enabled_required=False, emails=True)

        self.statbox.log(track_id=self.track_id, action='submitted')
        self.response_values['track_id'] = self.track_id

        with self.track_transaction.commit_on_error() as track:
            self.set_uid_to_track(self.account.uid, track)
            track.is_strong_password_policy_required = self.account.is_strong_password_required
            track.login = self.account.login


class FlushPddCommit(FlushPddBase,
                     BundlePasswordValidationMixin,
                     BundlePasswordChangeMixin,
                     BundleAccountFlushMixin):
    require_track = True
    basic_form = FlushPddCommitForm

    def clean_account_info(self):
        # Эти поля выставятся при дорегистрации
        self.account.is_pdd_agreement_accepted = False

        self.account.person.firstname = None
        self.account.person.lastname = None
        self.account.person.gender = None
        self.account.person.birthday = None
        self.account.person.country = None
        self.account.person.language = None
        self.account.person.timezone = None

        self.account.hint = None

    def clean_other_info(self):
        self.account.browser_key = None

    def clean_karma(self):
        self.account.karma.prefix = 0
        self.account.karma.suffix = 0

    def drop_password(self):
        # Отключаем 2fa
        self.account.totp_secret = None
        # Отключаем sms-2fa
        self.account.sms_2fa_on = False  # игнорируем атрибут forbid_disabling_sms_2fa
        # Отключаем пароли приложений
        self.account.enable_app_password = None

        # Игнорируем настройку домена о запрете смены пароля: админу можно всё
        password, quality = self.validate_password(
            required_check_password_history=True,
            is_strong_policy=self.account.is_strong_password_required,
            old_password_hash=self.account.password.serialized_encrypted_password,
            emails=set([email.address for email in self.account.emails.native]),
        )
        self.change_password(
            password,
            quality,
            force_password_change=self.form_values['force_password_change'],
            changing_reason=PASSWORD_CHANGING_REASON_FLUSHED_BY_ADMIN,
        )

    def process_request(self):
        self.read_track()
        if not self.track.uid:
            raise InvalidTrackStateError()
        uid = int(self.track.uid)
        if uid != self.all_values['uid']:
            raise InvalidTrackStateError()

        self.get_account_by_uid(
            uid=uid,
            enabled_required=False,
            need_phones=True,
            emails=True,
            email_attributes='all',
        )

        self.process_basic_form()

        events = {
            'action': 'flush_pdd',
            'consumer': self.consumer,
        }
        with UPDATE(self.account, self.request.env, events):
            self.clean_account_info()
            self.clean_other_info()
            self.clean_karma()
            self.drop_password()
            self.drop_phones()
            self.drop_emails()
            self.drop_social_profiles()

        self.statbox.log(track_id=self.track_id, action='flushed')
