# -*- coding: utf-8 -*-
from passport.backend.core import validators
from passport.backend.core.models.password import (
    PASSWORD_CHANGING_REASON_BY_NAME,
    PASSWORD_ENCODING_VERSION_MD5_CRYPT_ARGON,
    PASSWORD_ENCODING_VERSION_RAW_MD5_ARGON,
)


class PasswordOptionsForm(validators.Schema):
    """Форма задания специальных опций пароля"""
    uid = validators.Uid(not_empty=True)

    # принудительная смена пароля
    is_changing_required = validators.StringBool(if_missing=None)
    changing_requirement_reason = validators.OneOf(list(PASSWORD_CHANGING_REASON_BY_NAME.keys()), if_missing=None)
    # не устанавиливать ТБВС, если пароль менялся N дней назад
    max_change_frequency_in_days = validators.Int(min=0, if_missing=None)
    # отправить пользователю sms о статусе его пароля(если требуется смена)
    notify_by_sms = validators.StringBool(if_missing=None)
    # время задания пароля
    update_timestamp = validators.TimestampInPast(if_missing=None)
    # глобально разлогинить
    global_logout = validators.StringBool(if_missing=None)
    # инвалидировать токены
    revoke_tokens = validators.StringBool(if_missing=None)
    # инвалидировать пароли для приложений
    revoke_app_passwords = validators.StringBool(if_missing=None)
    # инвалидировать сессионные куки
    revoke_web_sessions = validators.StringBool(if_missing=None)
    # показать пользователю промо 2ФА
    show_2fa_promo = validators.StringBool(if_missing=None)
    # Имя админа и его комментарий для лога
    admin_name = validators.String(if_missing=None, not_empty=True, strip=True)
    comment = validators.String(if_missing=None, not_empty=True, strip=True)

    chained_validators = [
        validators.RequireSome(
            [
                'is_changing_required',
                'update_timestamp',
                'global_logout',
                'revoke_tokens',
                'revoke_app_passwords',
                'revoke_web_sessions',
            ],
            max_=6,  # все 6 параметров могут быть заданы, требуем хотя бы один
        ),
        validators.RequireIfEquals(['is_changing_required'], 'notify_by_sms', True),
        validators.RequireIfPresent('is_changing_required', present='show_2fa_promo'),
        validators.RequireIfPresent('is_changing_required', present='max_change_frequency_in_days'),
        validators.RequireSet([('admin_name', 'comment')], allow_empty=True),
    ]


class AccountOptionsForm(validators.Schema):
    """Форма задания специальных опций аккаунта"""
    uid = validators.Uid(not_empty=True)

    is_enabled = validators.StringBool(if_missing=None)
    is_app_password_enabled = validators.StringBool(if_missing=None)
    # Является ли аккаунт рассылкой
    is_maillist = validators.StringBool(if_missing=None)
    # является ли аккаунт совместно используемым
    is_shared = validators.StringBool(if_missing=None)
    # Имя админа и его комментарий для лога
    admin_name = validators.String(if_missing=None, not_empty=True, strip=True)
    comment = validators.String(if_missing=None, not_empty=True, strip=True)

    chained_validators = [
        validators.RequireSome(
            [
                'is_enabled',
                'is_app_password_enabled',
                'is_shared',
                'is_maillist',
            ],
            min_=1,
            max_=4,
        ),
        validators.RequireSet([('admin_name', 'comment')], allow_empty=True),
    ]


class AccountPersonalInfoForm(validators.Schema):
    pre_validators = [
        validators.GroupDisplayName(),
    ]

    force_clean_web = validators.StringBool(if_missing=False)
    uid = validators.Uid(if_missing=None)
    track_id = validators.TrackId(if_missing=None)
    firstname = validators.AntiFraudFirstName(if_missing=None, not_empty=True)
    firstname_global = validators.AntiFraudFirstName(if_missing=None)
    lastname = validators.AntiFraudLastName(if_missing=None, not_empty=True)
    lastname_global = validators.AntiFraudLastName(if_missing=None)
    display_name = validators.DisplayName(if_missing=None)
    gender = validators.Gender(if_missing=None, not_empty=True)
    birthday = validators.Birthday(if_missing=None, if_empty='', need_full=True)
    country = validators.CountryCode(if_missing=None)
    city_id = validators.CityId(if_missing=None, not_empty=True)
    city = validators.Regex(r'^[^<>\&]{,50}$', if_missing=None, if_empty='')
    language = validators.PortalLanguage(if_missing=None)
    timezone = validators.Timezone(if_missing=None, not_empty=True)
    contact_phone_number = validators.String(
        if_missing=None,
        not_empty=True,
        strip=True,
    )
    public_id = validators.PublicId(if_missing=None, not_empty=True)

    chained_validators = [
        validators.RequireSome([
            'uid',
            'track_id',
        ]),
        validators.RequireSome([
            'firstname',
            'firstname_global',
            'lastname',
            'lastname_global',
            'display_name',
            'gender',
            'birthday',
            'country',
            'city_id',
            'city',
            'language',
            'timezone',
            'contact_phone_number',
            'public_id',
        ], min_=1, max_=14),
        validators.PhoneNumber(
            phone_number_key='contact_phone_number',
            country_key=None,
        ),
    ]


class AccountChangeLanguageForm(validators.Schema):
    uid = validators.Uid(if_missing=None)
    language = validators.PortalLanguage(not_empty=True)


class BaseAccountAliasForm(validators.Schema):
    """Базовая форма для работы с простыми алиасами"""
    uid = validators.Uid(not_empty=True)
    alias_type = validators.OneOf(
        [
            'altdomain',
            'pdddomain',
        ],
        not_empty=True,
        strip=True,
    )


class AccountAliasDeleteForm(BaseAccountAliasForm):
    """Форма для удаления простого алиаса"""


class AccountPddAliasLoginForm(validators.Schema):
    """Форма для операций с дополнительными логинами на алиасе ПДД"""
    uid = validators.Uid(not_empty=True)
    login = validators.PddLogin()


class AccountAliasCreateForm(validators.Schema):
    """Форма для добавления алиасами на альтернативных доменах Яндекса"""
    uid = validators.Uid(not_empty=True)
    alias = validators.AltDomainAliasValidator()

    chained_validators = [
        validators.Availability(login_field='alias'),
    ]


class AccountPDDCreateForm(validators.Schema):
    """Форма для добавления pdd алиаса (7й)"""
    uid = validators.Uid(not_empty=True)
    alias = validators.ExternalEmailValidator(not_empty=True, strip=True)

    chained_validators = [
        validators.Availability(login_field='alias'),
    ]


class AccountBankPhoneNumberAliasCreateForm(validators.Schema):
    """Форма для привязывания алиаса банковского номера телефона"""
    uid = validators.Uid(not_empty=True)
    phone_number = validators.String(if_missing=None, strip=True)
    phone_id = validators.PhoneId(if_missing=None)
    country = validators.String(if_missing=None, not_empty=True, strip=True)

    chained_validators = [
        validators.PhoneNumber(),
        validators.RequireSome(
            [
                'phone_number',
                'phone_id',
            ],
            min_=1,
            max_=1,
        ),
    ]


class AccountBankPhoneNumberAliasDeleteForm(validators.Schema):
    """Форма удаления алиаса банковского номера телефона"""
    uid = validators.Uid(not_empty=True)


class FlushPddSubmitForm(validators.Schema):
    """Форма получения информации о ПДД-аккаунте для последующего сброса последнего"""
    uid = validators.Uid(not_empty=True)


class FlushPddCommitForm(validators.Schema):
    """Форма сброса ПДД-аккаунта и выставления ему нового пароля"""
    uid = validators.Uid(not_empty=True)
    password = validators.String(not_empty=True)

    # Принудить пользователя к смене пароля при первом входе (причина смены пароля - сброс пароля)
    force_password_change = validators.StringBoolean(not_empty=True, if_missing=False)


class MigrateMailishForm(validators.Schema):
    email = validators.ComprehensiveEmailValidator(allow_native=False)
    mailish_id = validators.MailishId(not_empty=True)


class AccountDeleteForm(validators.Schema):
    """Форма удаления аккаунта"""
    uid = validators.Uid(not_empty=True)


class AccountDeleteConfirmPhoneFormV2(validators.Schema):
    code = validators.String(not_empty=True, strip=True)


DEFAULT_STRING_UPPER_LIMIT = 1024


class AccountDeleteCheckAnswerFormV2(validators.Schema):
    answer = validators.String(not_empty=True, strip=True, max=DEFAULT_STRING_UPPER_LIMIT)


class AccountDeleteSendEmailCodeFormV2(validators.Schema):
    email = validators.ComprehensiveEmailValidator()


class AccountDeleteConfirmEmailFormV2(validators.Schema):
    code = validators.String(not_empty=True, strip=True)


class AccountShortInfoForm(validators.Schema):
    # Намеренно не валидирую размер, чтобы появление новых размеров в Япике не требовало обновления кода в Паспорте
    avatar_size = validators.String(not_empty=True, strip=True)


class AccountFullInfoForm(validators.Schema):
    need_display_name_variants = validators.StringBoolean(not_empty=True, if_missing=True)
    need_phones = validators.StringBoolean(not_empty=True, if_missing=True)
    need_emails = validators.StringBoolean(not_empty=True, if_missing=True)
    need_social_profiles = validators.StringBoolean(not_empty=True, if_missing=True)
    need_question = validators.StringBoolean(not_empty=True, if_missing=True)
    need_additional_account_data = validators.StringBoolean(not_empty=True, if_missing=True)
    need_family_info = validators.StringBoolean(not_empty=True, if_missing=False)
    need_family_members = validators.StringBoolean(not_empty=True, if_missing=False)
    need_family_invites = validators.StringBoolean(not_empty=True, if_missing=False)
    need_family_kids = validators.StringBoolean(not_empty=True, if_missing=False)


class AccountSetPasswordForm(validators.Schema):
    """Форма проставления пароля пользователю"""

    # UID пользователя, которому меняем пароль
    uid = validators.Uid()
    password = validators.String(not_empty=True, if_missing=None)
    # "Сырой" хэш пароля, который будет записан пользователю
    password_hash = validators.PasswordHash(
        allowed_hash_versions=[PASSWORD_ENCODING_VERSION_MD5_CRYPT_ARGON, PASSWORD_ENCODING_VERSION_RAW_MD5_ARGON],
        not_empty=True,
        if_missing=None,
    )

    # Принудить пользователя к смене пароля при первом входе (причина смены пароля - сброс пароля)
    force_password_change = validators.StringBoolean(not_empty=True, if_missing=False)

    chained_validators = [
        validators.RequireSome(
            [
                'password',
                'password_hash',
            ],
            max_=1,
        ),
    ]


class AccountCheckExistsForm(validators.Schema):
    email = validators.ComprehensiveEmailValidator()


class AccountResetAvatarForm(validators.Schema):
    uid = validators.Uid(not_empty=True)
    avatar_key = validators.AvatarId()


class AccountResetDisplayNameForm(validators.Schema):
    uid = validators.Uid(not_empty=True)
    public_name = validators.String(if_missing=None)
    full_name = validators.String(if_missing=None)

    chained_validators = [
        validators.RequireSome(
            [
                'public_name',
                'full_name',
            ],
            min_=1,
            max_=1,
        ),
    ]


class AccountResetQuestionForm(validators.Schema):
    uid = validators.Uid(not_empty=True)


class AccountResetEmailForm(validators.Schema):
    uid = validators.Uid(not_empty=True)
    email = validators.ComprehensiveEmailValidator()


class AccountResetPhoneForm(validators.Schema):
    uid = validators.Uid(not_empty=True)
    phone_number = validators.String(not_empty=True, strip=True)

    chained_validators = [
        validators.PhoneNumber(allow_impossible=True)
    ]


class ScholarPersonalInfoForm(validators.Schema):
    uid = validators.Uid()
    firstname = validators.AntiFraudFirstName(if_missing=None, not_empty=True)
    lastname = validators.AntiFraudLastName(if_missing=None, not_empty=True)

    require_at_least_one = [
        'firstname',
        'lastname',
    ]
    chained_validators = [
        validators.RequireSome(require_at_least_one, min_=1, max_=len(require_at_least_one)),
    ]


class FederalChangeForm(validators.Schema):
    uid = validators.Uid(not_empty=True)

    # отключение/включение аккаунта
    active = validators.StringBool(if_missing=None)
    firstname = validators.FirstName(if_missing=None)
    lastname = validators.LastName(if_missing=None)
    display_name = validators.String(if_missing=None, strip=True)
    emails = validators.ListValidator(
        validators.ComprehensiveEmailValidator(),
        max=50,
        unique=True,
        if_missing=None,
        if_empty=tuple(),
    )

    chained_validators = [
        validators.RequireSome(
            [
                'active',
                'firstname',
                'lastname',
                'display_name',
                'emails',
            ],
            max_=5,
        ),
    ]
