# -*- coding: utf-8 -*-
from passport.backend.api.common.oauth import get_token_issue_time
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import InvalidTrackStateError
from passport.backend.api.views.bundle.mixins import (
    BundleAccountGetterMixin,
    BundleAccountResponseRendererMixin,
    BundleFamilyMixin,
    BundleKiddishMixin,
    BundleLastAuthMixin,
    BundlePhoneMixin,
)
from passport.backend.core.conf import settings

from .forms import (
    AccountFullInfoForm,
    AccountShortInfoForm,
)


ACCOUNT_FROM_TRACK_GRANT = 'account.from_track'
ACCOUNT_FULL_INFO_READ_GRANT = 'account.full_info_read'
ACCOUNT_SHORT_INFO_READ_GRANT = 'account.short_info_read'


class GetAccountInfoFromTrackView(BaseBundleView,
                                  BundleAccountGetterMixin,
                                  BundleAccountResponseRendererMixin):
    require_track = True

    required_grants = [ACCOUNT_FROM_TRACK_GRANT]

    def process_request(self, *args, **kwargs):
        self.read_track()
        if not self.track.uid:
            raise InvalidTrackStateError()

        self.get_account_from_track()
        self.fill_response_with_account(account_info_required=True)
        self.response_values['track_id'] = self.track_id


class GetAccountFullInfoView(BaseBundleView,
                             BundleAccountGetterMixin,
                             BundleAccountResponseRendererMixin,
                             BundlePhoneMixin,
                             BundleLastAuthMixin,
                             BundleFamilyMixin,
                             BundleKiddishMixin):

    required_grants = [ACCOUNT_FULL_INFO_READ_GRANT]
    basic_form = AccountFullInfoForm

    def process_request(self):
        self.process_basic_form()
        need_phones = self.form_values['need_phones']
        need_emails = self.form_values['need_emails']
        need_family_info = self.form_values['need_family_info']
        need_family_members = self.form_values['need_family_members']
        need_family_invites = self.form_values['need_family_invites']
        need_family_kids = self.form_values['need_family_kids']
        if (
            need_family_invites or
            need_family_kids or
            need_family_members
        ):
            need_family_info = True
        self.get_account_from_session_or_oauth_token(
            need_phones=need_phones,
            emails=need_emails,
            get_family_info=need_family_info,
            get_public_id=True,
        )
        if settings.IS_INTRANET:
            self.fill_response_with_account(
                with_rfc_2fa=True,
                with_emails=need_emails,
                with_family_info=need_family_info,
            )
        else:
            self.fill_response_with_account(
                account_info_required=True,
                additional_person_data=True,
                with_display_names=self.form_values['need_display_name_variants'],
                with_phones=need_phones,
                with_emails=need_emails,
                with_social_profiles=self.form_values['need_social_profiles'],
                with_question=self.form_values['need_question'],
                additional_account_data=self.form_values['need_additional_account_data'],
                with_family_info=need_family_info,
            )
        if need_family_info:
            self.fill_response_with_family_settings()
        if (need_family_members or need_family_kids) and self.account.has_family:
            self.load_family_info_by_family_id(
                self.account.family_info.family_id,
            )
            self.fill_response_with_family_roles(
                need_adults=need_family_members,
                need_kids=need_family_kids,
            )
        if need_family_invites and self.account.has_family and self.is_family_admin():
            self.load_family_invites_by_family_id(
                self.account.family_info.family_id,
            )
            self.fill_response_with_family_invites()


class GetAccountShortInfoView(BaseBundleView, BundleAccountGetterMixin, BundleAccountResponseRendererMixin):
    """
    Ручка предназначена для использования с мобильных устройств.
    Поэтому её ответ должен быть максимально коротким и плоским.
    """
    required_grants = [ACCOUNT_SHORT_INFO_READ_GRANT]
    basic_form = AccountShortInfoForm

    def process_request(self):
        self.process_basic_form()
        self.get_account_by_oauth_token(emails=True, get_public_id=True)
        self.fill_response_with_short_account_info(avatar_size=self.form_values['avatar_size'])
        self.response_values.update(
            x_token_issued_at=get_token_issue_time(self.oauth_info),
        )

        if self.request.env.user_agent and any(
            self.request.env.user_agent.startswith(version_prefix)
            for version_prefix in ['com.yandex.mobile.auth.sdk/4.210.', 'com.yandex.mobile.auth.sdk/4.211.']
        ):
            # для бажных iOS-версий заменяем нормализованный логин ненормализованным
            self.response_values.pop('normalized_display_login', None)
            if self.response_values.get('display_login'):
                self.response_values.update(normalized_display_login=self.response_values['display_login'])
