# -*- coding: utf-8 -*-
import logging

from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.mixins import (
    BundleAccountGetterMixin,
    BundleLastAuthMixin,
)
from passport.backend.core.builders.historydb_api import get_historydb_api
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.loggers.statbox import StatboxLogger
from passport.backend.core.utils.decorators import cached_property
from passport.backend.utils.time import datetime_to_unixtime

from .exceptions import ChallengeNotFoundError
from .forms import (
    ChallengeForm,
    HistoryForm,
)


log = logging.getLogger('passport.api.view.bundle.account.history')


ACCOUNT_HISTORY_BASIC_GRANT = 'history.base'
ACCOUNT_HISTORY_HISTORY_BY_UID_GRANT = 'history.by_uid'


class HistoryViewOld(BaseBundleView, BundleAccountGetterMixin, BundleLastAuthMixin):
    require_track = False

    basic_form = HistoryForm

    required_grants = [ACCOUNT_HISTORY_BASIC_GRANT]

    @cached_property
    def statbox(self):
        return StatboxLogger(
            uid=self.account.uid,
            mode='history',
            consumer=self.consumer,
            yandexuid=self.request.env.cookies.get('yandexuid'),
            useragent=self.user_agent,
            user_ip=self.client_ip,
        )

    def aggregated_auths(self, uid=None, hours_limit=None, limit=None, from_auth_row=None, password_auths=None):
        historydb = get_historydb_api()
        auths, last_row = historydb.auths_aggregated_old(
            uid=uid or self.account.uid,
            limit=limit,
            hours_limit=hours_limit,
            from_row=from_auth_row,
            password_auths=password_auths,
        )
        return auths, last_row

    def process_request(self, *args, **kwargs):
        self.process_basic_form()
        self.get_account_from_uid_or_session(
            by_uid_grant=ACCOUNT_HISTORY_HISTORY_BY_UID_GRANT,
            check_disabled_on_deletion=True,
        )

        auths, next_auth_row = self.aggregated_auths(
            uid=self.account.uid,
            limit=self.form_values['limit'],
            hours_limit=self.form_values['hours_limit'],
            from_auth_row=self.form_values['from_auth_row'],
            password_auths=self.form_values['password_auths'],
        )
        auths = sorted(auths, key=lambda x: x.get('timestamps'), reverse=True)
        if next_auth_row:
            next_auths, _ = self.aggregated_auths(
                uid=self.account.uid,
                limit=1,
                hours_limit=self.form_values['hours_limit'],
                from_auth_row=next_auth_row,
            )
            if not next_auths:
                next_auth_row = None

        self.response_values.update(
            events=auths,
            next_auth_row=next_auth_row,
        )


class HistoryView(HistoryViewOld):
    def aggregated_auths(self, uid=None, hours_limit=None, limit=None, from_auth_row=None, password_auths=None):
        historydb = get_historydb_api()
        auths, last_row = historydb.auths_aggregated(
            uid=uid or self.account.uid,
            limit=limit,
            hours_limit=hours_limit,
            from_row=from_auth_row,
            password_auths=password_auths,
        )
        return auths, last_row


class ChallengeView(BaseBundleView, BundleAccountGetterMixin):
    basic_form = ChallengeForm
    required_grants = [ACCOUNT_HISTORY_BASIC_GRANT]

    def process_request(self, *args, **kwargs):
        self.process_basic_form()
        self.get_account_from_uid_or_session(
            by_uid_grant=ACCOUNT_HISTORY_HISTORY_BY_UID_GRANT,
            check_disabled_on_deletion=True,
        )
        challenge_ts = datetime_to_unixtime(self.form_values['challenge_time'])
        threshold = settings.CHALLENGE_HISTORY_THRESHOLD
        challenges = get_historydb_api().auths_failed(
            uid=self.account.uid,
            from_ts=int(challenge_ts - threshold),
            to_ts=int(challenge_ts + threshold + 1),
            limit=settings.CHALLENGE_HISTORY_LIMIT,
            status='challenge_shown',
        )
        for challenge in challenges:
            if abs(challenge['timestamp'] - challenge_ts) <= threshold:
                self.response_values.update(challenge=challenge)
                return

        log.debug('Found %s challenges, none of them matched ts=%s', len(challenges), challenge_ts)
        raise ChallengeNotFoundError()
