# -*- coding: utf-8 -*-
import logging

from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.mixins import (
    BundleAccountGetterMixin,
    BundleLastAuthMixin,
)
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.loggers.statbox import StatboxLogger
from passport.backend.core.utils.decorators import cached_property

from .forms import LastauthForm


log = logging.getLogger('passport.api.view.bundle.account.lastauth')


LASTAUTH_BASIC_GRANT = 'lastauth.base'
LASTAUTH_BY_UID_GRANT = 'lastauth.by_uid'


class LastauthView(BaseBundleView, BundleAccountGetterMixin, BundleLastAuthMixin):
    require_track = False

    basic_form = LastauthForm

    required_grants = [LASTAUTH_BASIC_GRANT]

    @cached_property
    def statbox(self):
        return StatboxLogger(
            uid=self.account.uid,
            mode='lastauth',
            consumer=self.consumer,
            yandexuid=self.request.env.cookies.get('yandexuid'),
            useragent=self.user_agent,
            user_ip=self.client_ip,
        )

    def process_request(self, *args, **kwargs):
        self.process_basic_form()
        self.get_account_from_uid_or_session(
            by_uid_grant=LASTAUTH_BY_UID_GRANT,
            check_disabled_on_deletion=True,
        )

        lastauth = self.get_auths_aggregated(
            limit=settings.LASTAUTH_DEFAULT_LIMIT,
            hours_limit=settings.LASTAUTH_DEFAULT_HOURS_LIMIT,
        )

        if lastauth[self.WEB_PASSWORD_LASTAUTH]['apps']:
            # Запишем знание о не-вебовских авторизациях для дальнейшего анализа клиентов,
            # которые могут пострадать после включения 2фа
            protocols = [item['authtype'] for item in lastauth[self.WEB_PASSWORD_LASTAUTH]['apps']]
            self.statbox.log(
                authtypes=','.join(sorted(set(protocols))),
                enabled_2fa_track_id=self.form_values.get('enabled_2fa_track_id')
            )
        self.response_values['lastauth'] = lastauth
