# -*- coding: utf-8 -*-
import datetime

from passport.backend.api.common.authorization import (
    build_auth_cookies,
    build_cookie_mda2_beacon,
    build_non_auth_cookies,
    users_from_multisession,
)
from passport.backend.api.common.common import extract_tld
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import InvalidCSRFTokenError
from passport.backend.api.views.bundle.headers import (
    HEADER_CLIENT_COOKIE,
    HEADER_CLIENT_HOST,
    HEADER_CONSUMER_CLIENT_IP,
)
from passport.backend.api.views.bundle.mixins.account import (
    BundleAccountGetterMixin,
    BundleAccountResponseRendererMixin,
    CookieCheckStatus,
)
from passport.backend.api.views.bundle.mixins.session import BundleEditSessionMixin
from passport.backend.core.builders.blackbox.constants import BLACKBOX_EDITSESSION_OP_DELETE
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.loggers.statbox import StatboxLogger
from passport.backend.core.runner.context_managers import UPDATE
from passport.backend.core.utils.decorators import cached_property
from passport.backend.utils.string import smart_str
from passport.backend.utils.time import get_unixtime
from six.moves.urllib.parse import quote_plus

from .forms import (
    DEFAULT_TARGET,
    LogoutForm,
)


ACCOUNT_LOGOUT_GRANT = 'account.logout'

DEFAULT_RETPATH = 'https://www.yandex.%s'


class AccountLogoutView(BundleAccountGetterMixin,
                        BundleAccountResponseRendererMixin,
                        BundleEditSessionMixin,
                        BaseBundleView):

    required_grants = [ACCOUNT_LOGOUT_GRANT]
    required_headers = (
        HEADER_CONSUMER_CLIENT_IP,
        HEADER_CLIENT_COOKIE,
        HEADER_CLIENT_HOST,
    )

    require_track = False
    track_type = 'authorize'

    basic_form = LogoutForm

    @property
    def default_retpath(self):
        tld = extract_tld(self.request.env.host, settings.PASSPORT_TLDS) or settings.PASSPORT_DEFAULT_TLD
        return DEFAULT_RETPATH % tld

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode='logout',
            consumer=self.consumer,
            ip=self.client_ip,
            user_agent=self.user_agent,
            referer=self.referer,
        )

    def fill_response_with_left_accounts(self, exclude_uid):
        accounts = self.get_multisession_accounts(
            self.session_info,
            exclude_uid=exclude_uid,
        )
        if accounts:
            self.response_values['accounts'] = accounts
        self.response_values['logged_out_uids'] = [exclude_uid]

    def logout_everybody(self):
        users = users_from_multisession(self.session_info)
        for uid, user_data in users.items():
            self.log_logout(self.session_info, uid)

        empty_session = self.build_empty_session(
            with_ssl_session=bool(self.request.env.ssl_session_id),
            with_sessguard=bool(self.request.env.sessguard),
        )
        new_cookies = build_auth_cookies(self.request.env, empty_session)
        new_cookies.extend(
            build_non_auth_cookies(
                env=self.request.env,
                uid=None,
                human_readable_login=None,
                machine_readable_login=None,
                cookie_path='/',
                ignore_cookie_l=True,
                ignore_lah_for=settings.IGNORE_LAH_AFTER_LOGOUT_FOR,
            ),
        )
        new_cookies.append(build_cookie_mda2_beacon(self.request.env))

        with self.track_transaction.rollback_on_error() as track:
            track.session = ''
            track.sslsession = ''
            track.sessguard = ''
            track.session_created_at = track.session_created_at or get_unixtime()

        self.statbox.log(action='cookie_remove')
        self.response_values['cookies'] = new_cookies
        if users:
            self.response_values['logged_out_uids'] = users.keys()

    def logout_default(self, uid):
        if self.form_values['is_global'] and self.account is not None:
            events = {
                'action': 'logout',
                'consumer': self.consumer,
            }
            with UPDATE(self.account, self.request.env, events):
                self.account.global_logout_datetime = datetime.datetime.now()

        with self.track_transaction.rollback_on_error():
            cookies, default_uid = self.edit_session(
                BLACKBOX_EDITSESSION_OP_DELETE,
                uid,
                self.session_info,
                {},
            )
        self.response_values.update(cookies=cookies)
        if default_uid:
            self.response_values.update(default_uid=default_uid)
        self.fill_response_with_left_accounts(exclude_uid=uid)

    def try_get_account(self):
        self.session_info = self.check_session_cookie(allow_scholar=True)

        if self.session_info.status in (
                CookieCheckStatus.Valid,
                CookieCheckStatus.NeedReset,
                CookieCheckStatus.Disabled,
        ):
            self.parse_account(
                self.session_info.response,
                enabled_required=False,
            )
        else:
            self.account = None

    def process_request(self):
        self.process_basic_form()

        # Защищаемся от CSRF атаки
        yandexuid_cookie = self.cookies.get('yandexuid')
        if self.form_values['yu'] and self.form_values['yu'] != yandexuid_cookie:
            raise InvalidCSRFTokenError()

        # Получим дефолтный аккаунт, если переданная кука валидна
        self.try_get_account()
        default_uid = 0
        if self.session_info.response:
            if self.form_values['ci'] and self.form_values['ci'] != self.session_info.response.get('connection_id'):
                raise InvalidCSRFTokenError()
            default_uid = self.session_info.response['default_uid'] or 0

        retpath = self.form_values['retpath'] or self.default_retpath

        if self.account and self.account.is_pdd and self.account.domain.domain in settings.PDD_DOMAIN_RETPATH_WRAPPER_MAP:
            retpath = settings.PDD_DOMAIN_RETPATH_WRAPPER_MAP[self.account.domain.domain] % quote_plus(smart_str(retpath))

        self.read_or_create_track(self.track_type)
        with self.track_transaction.rollback_on_error() as track:
            track.retpath = retpath

        self.response_values['track_id'] = self.track_id

        # Глобально разлогиниваем только дефолтовый аккаунт
        target = self.form_values['target']
        if self.form_values['is_global']:
            target = DEFAULT_TARGET

        statbox_params = {
            'action': 'cookie_edit',
            'target': target,
            'global': self.form_values['is_global'],
            'yandexuid': yandexuid_cookie,
            'uid': default_uid,
            'retpath': retpath,
            'origin': self.form_values['origin'],
            'authid': self.session_info.authid,
        }

        self.statbox.bind_context(**statbox_params)

        if default_uid and target == DEFAULT_TARGET:
            self.logout_default(uid=default_uid)
        else:
            self.logout_everybody()
