# -*- coding: utf-8 -*-
import logging

from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import (
    AccountNotFoundError,
    LoginNotavailableError,
    RateLimitExceedError,
)
from passport.backend.api.views.bundle.mixins import BundleAccountGetterMixin
from passport.backend.core.counters import migrate_mailish
from passport.backend.core.logging_utils.loggers import StatboxLogger
from passport.backend.core.runner.context_managers import UPDATE
from passport.backend.core.types.display_name import DisplayName
from passport.backend.core.utils.decorators import cached_property

from .forms import MigrateMailishForm


log = logging.getLogger('passport.api.view.bundle.account.migrate_mailish')


class MigrateMailish(BaseBundleView, BundleAccountGetterMixin):
    require_track = False
    basic_form = MigrateMailishForm

    required_grants = ['account.migrate_mailish']

    def __init__(self):
        super(MigrateMailish, self).__init__()
        self.counter = migrate_mailish.get_per_consumer_counter()

    @cached_property
    def statbox(self):
        return StatboxLogger(
            uid=self.account.uid,
            mode='migrate_mailish',
            consumer=self.consumer,
        )

    def check_counter(self):
        if self.counter.hit_limit(self.consumer):
            raise RateLimitExceedError()

    def increment_counter(self):
        self.counter.incr(self.consumer)

    def is_account_type_allowed(self):
        return self.account.is_mailish

    def process_request(self):
        self.check_counter()
        self.process_basic_form()

        # Не запрашиваем display_name, так как хотим его в любом случае записать в БД (если ЧЯ вернёт
        # вычисленный display_name, то диффер решит, что ничего не изменилось)
        self.get_account_by_login(
            login=self.form_values['email'],
            sid='mailish',
            enabled_required=False,
            need_display_name=False,
        )

        try:
            self.get_account_by_login(
                login=self.form_values['mailish_id'],
                sid='mailish',
                enabled_required=False,
            )
        except AccountNotFoundError:
            pass
        else:
            raise LoginNotavailableError()

        events = {
            'action': 'migrate_mailish',
            'consumer': self.consumer,
        }
        with UPDATE(self.account, self.request.env, events):
            self.account.default_email = self.account.default_email or self.account.mailish_alias.alias
            self.account.mailish_alias.mailish_id = self.form_values['mailish_id']
            self.account.person.display_name = DisplayName(name=self.account.default_email)

        self.increment_counter()

        self.statbox.log(
            action='migrated',
            email=self.account.default_email,
            mailish_id=self.account.mailish_alias.alias,
        )
