# -*- coding: utf-8 -*-
from operator import attrgetter

from passport.backend.api.views.bundle.exceptions import ActionNotRequiredError
from passport.backend.core.grants import GrantsError

from .grants import (
    ACCOUNT_TYPE_TO_POSTFIX,
    POSTFIX_BY_UID,
)
from .utils import attrsetter


class OptionsBaseHandler(object):
    def __init__(self, view, option_name, option_params):
        self.view = view
        self.account = self.view.account
        self.form_values = self.view.form_values
        self.option_name = option_name
        self.option_form_value = self.view.form_values[self.option_name]
        self.option_params = option_params

    @property
    def _account_attribute_name(self):
        return self.option_params.get('account_attribute_name', self.option_name)

    @property
    def option(self):
        return attrgetter(self._account_attribute_name)(self.account)

    @option.setter
    def option(self, value):
        attrsetter(self._account_attribute_name, value)(self.account)

    def is_work_required(self):
        raise NotImplementedError()  # pragma: no cover

    def check_grants(self):
        old_style_grants = self.option_params['old_style_grants']
        new_style_grant_prefix = self.option_params['new_style_grant_prefix']
        if old_style_grants is None and new_style_grant_prefix is None:
            raise ValueError('No grants configured for `%s`' % self.option_name)  # pragma: no cover

        if old_style_grants is not None:
            try:
                self.view.check_grants_for_account_type()
                for grant in old_style_grants:
                    self.view.check_grant(grant)
                return
            except GrantsError:
                if new_style_grant_prefix is None:
                    # новые гранты не настроены - кидаем ошибку о нехватке старых грантов
                    raise

        # старых грантов не хватает - проверим новые
        grants = {
            account_type: '.'.join([new_style_grant_prefix, postfix])
            for account_type, postfix in ACCOUNT_TYPE_TO_POSTFIX.items()
        }
        self.view.check_grants_for_account_type(grants)
        if self.view.account_got_by_uid:
            grant = '.'.join([new_style_grant_prefix, POSTFIX_BY_UID])
            self.view.check_grant(grant)

    def set_option(self):
        self.option = self.option_form_value

    def pre_update(self):
        if self.is_work_required():
            self.check_grants()

    def update(self):
        if self.is_work_required():
            if self.option == self.option_form_value and self.option_params.get('raise_if_nothing_to_do'):
                raise ActionNotRequiredError()
            self.set_option()

    def post_update(self):
        pass


class BaseValueHandler(OptionsBaseHandler):
    def is_work_required(self):
        return self.option_form_value is not None


class BaseFlaggedHandler(OptionsBaseHandler):
    def is_work_required(self):
        return bool(self.option_form_value)
