# -*- coding: utf-8 -*-

import logging

from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.constants import AUTHENTICATION_MEDIA_UID
from passport.backend.api.views.bundle.mixins import (
    BundleAccountGetterMixin,
    BundlePhoneMixin,
    UserMetaDataMixin,
)
from passport.backend.api.views.bundle.mixins.common import (
    BundleAdminActionMixin,
    BundleTvmUserTicketMixin,
)
from passport.backend.api.views.bundle.mixins.kolmogor import KolmogorMixin
from passport.backend.api.views.bundle.mixins.mail import MailMixin
from passport.backend.api.views.bundle.mixins.federal import BundleFederalMixin
from passport.backend.api.views.bundle.mixins.push import BundlePushMixin
from passport.backend.core.logging_utils.loggers.statbox import StatboxLogger
from passport.backend.core.runner.context_managers import UPDATE
from passport.backend.core.utils.decorators import cached_property

from .config import (
    ACCOUNT_OPTIONS_CONFIG,
    PLUS_ATTRIBUTES_CONFIG,
)
from .forms import (
    AccountOptionsForm,
    PlusChangeAttributesForm,
)
from .grants import (
    ACCOUNT_OPTIONS_ANY_GRANT,
    ACCOUNT_OPTIONS_BASE_GRANT,
    ACCOUNT_OPTIONS_NORMAL_GRANT,
    ACCOUNT_OPTIONS_PDD_GRANT,
    ACCOUNT_OPTIONS_PHONISH_GRANT,
    LOG_ADMIN_ACTION_GRANT,
    PLUS_OPTIONS_BASE_GRANT,
)
from .options import Options


log = logging.getLogger(__name__)


class AccountOptionsView(
    BaseBundleView,
    BundleAccountGetterMixin,
    BundlePhoneMixin,
    BundleTvmUserTicketMixin,
    BundleAdminActionMixin,
    BundlePushMixin,
    KolmogorMixin,
    MailMixin,
    UserMetaDataMixin,
    BundleFederalMixin,
):
    require_track = False
    required_grants = [ACCOUNT_OPTIONS_BASE_GRANT]
    grants_for_account_type = {
        'any': ACCOUNT_OPTIONS_ANY_GRANT,
        'normal': ACCOUNT_OPTIONS_NORMAL_GRANT,
        'pdd': ACCOUNT_OPTIONS_PDD_GRANT,
        'phonish': ACCOUNT_OPTIONS_PHONISH_GRANT,
    }

    basic_form = AccountOptionsForm

    @cached_property
    def options(self):
        return Options(view=self, config=ACCOUNT_OPTIONS_CONFIG)

    @cached_property
    def statbox(self):
        return StatboxLogger(
            uid=self.account.uid,
            mode='account_options',
            consumer=self.consumer,
            consumer_ip=self.request.env.consumer_ip,
        )

    def process_request(self, *args, **kwargs):
        self.process_basic_form()

        if self.is_admin_action:
            self.check_grant(LOG_ADMIN_ACTION_GRANT)

        media, _ = self.get_account_from_available_media(
            multisession_uid=self.form_values['multisession_uid'],
            ignore_grants=True,  # проверим в хэндлерах
            enabled_required=False,
            need_phones=True,
            get_billing_features=True,
            emails=True,
        )
        self.account_got_by_uid = media == AUTHENTICATION_MEDIA_UID

        events = {
            'action': 'account',
            'consumer': self.consumer,
        }
        self.mark_admin_action(events)

        self.options.pre_update()

        with UPDATE(self.account, self.request.env, events):
            self.options.update()

        self.options.post_update()


class PlusChangeAttributesView(BaseBundleView, BundleAccountGetterMixin):
    basic_form = PlusChangeAttributesForm
    required_grants = [PLUS_OPTIONS_BASE_GRANT]
    grants_for_account_type = {
        'any': PLUS_OPTIONS_BASE_GRANT,
    }

    @cached_property
    def options(self):
        return Options(view=self, config=PLUS_ATTRIBUTES_CONFIG)

    def process_request(self):
        self.process_basic_form()

        self.get_account_by_uid(self.form_values['uid'])
        self.account_got_by_uid = True

        events = {
            'action': 'account',
            'consumer': self.consumer,
        }

        self.options.pre_update()
        with UPDATE(self.account, self.request.env, events):
            self.options.update()

        self.options.post_update()
