# -*- coding: utf-8 -*-
from datetime import (
    datetime,
    timedelta,
)

from passport.backend.core import validators


ZERO_DATETIME = datetime.fromtimestamp(0)
# Музыка хочет уметь выдавать столетние подписки
DISTANT_FUTURE_DATETIME = datetime.fromtimestamp(2 ** 31) + timedelta(days=365 * 100)


OTT_SUBSCRIPTIONS = [
    # https://st.yandex-team.ru/PASSP-21788
    'YA_PLUS_3M',
    'YA_PLUS',
    'YA_PREMIUM',
    'KP_BASIC',
    # https://st.yandex-team.ru/PASSP-24155
    'YA_PLUS_KP',
    # https://st.yandex-team.ru/PASSP-24939
    'YA_PLUS_KP_3M',
    # https://st.yandex-team.ru/PASSP-30200
    'YA_PLUS_SUPER',
]


PLUS_FAMILY_ROLES = [
    # https://st.yandex-team.ru/PASSP-23642
    'PARENT',
    'CHILD',
]


class AccountOptionsForm(validators.Schema):
    """Форма задания специальных опций аккаунта"""
    uid = validators.Uid(if_missing=None, not_empty=True)
    multisession_uid = validators.Uid(if_missing=None, not_empty=True)

    # Имя админа и его комментарий для лога
    admin_name = validators.String(if_missing=None, not_empty=True, strip=True)
    comment = validators.String(if_missing=None, not_empty=True, strip=True)

    # доступен ли аккаунт
    is_enabled = validators.StringBool(if_missing=None)
    # включены ли ПП
    is_app_password_enabled = validators.StringBool(if_missing=None)
    # является ли аккаунт рассылкой
    is_maillist = validators.StringBool(if_missing=None)
    # заморожен ли почтовый ящик
    is_mailbox_frozen = validators.StringBool(if_missing=None)
    # является ли аккаунт совместно используемым
    is_shared = validators.StringBool(if_missing=None)
    # является ли аккаунт админом Коннекта
    is_connect_admin = validators.StringBool(if_missing=None)
    # является ли аккаунт часто взламываемым
    is_easily_hacked = validators.StringBool(if_missing=None)
    # принудительная смена пароля
    is_password_change_required = validators.StringBool(if_missing=None)
    # не устанавиливать ТБВС, если пароль менялся N дней назад
    max_change_frequency_in_days = validators.Int(min=0, if_missing=None)
    # отправить пользователю sms о статусе его пароля(если требуется смена)
    notify_by_sms = validators.StringBool(if_missing=None)
    # время задания пароля
    password_update_timestamp = validators.TimestampInPast(if_missing=None)
    # глобально разлогинить
    global_logout = validators.StringBool(if_missing=None)
    # инвалидировать токены
    revoke_tokens = validators.StringBool(if_missing=None)
    # инвалидировать пароли для приложений
    revoke_app_passwords = validators.StringBool(if_missing=None)
    # инвалидировать сессионные куки
    revoke_web_sessions = validators.StringBool(if_missing=None)
    # показать пользователю промо 2ФА
    show_2fa_promo = validators.StringBool(if_missing=None)
    # подключить сервис Аудитория
    audience_on = validators.StringBool(if_missing=None)
    # список id ненативных организаций, к которым относится аккаунт
    external_organization_ids = validators.ListValidator(
        validators.Int(min=0),
        max=500,
        unique=True,
        if_missing=None,
        if_empty=tuple(),
    )
    # Вход по мессенджеру/волшебному письму разрешен/запрещен
    magic_link_login_forbidden = validators.StringBool(if_missing=None)
    # Вход по Qr коду разрешен/запрещен
    qr_code_login_forbidden = validators.StringBool(if_missing=None)
    # Вход по sms разрешен/запрещен
    sms_code_login_forbidden = validators.StringBool(if_missing=None)
    # Разрешен ли доступ к takeout
    takeout_subscription = validators.StringBool(if_missing=None)
    # PASSP-25400 Билинговые фичи
    billing_features = validators.BillingFeatures(if_missing=None)
    # использовать полную фамилию для public name
    show_fio_in_public_name = validators.StringBool(if_missing=None)
    # обязательно требовать челлендж
    force_challenge = validators.StringBool(if_missing=None)
    # требовать смс-челлендж при входах с нового устройства
    sms_2fa_on = validators.StringBool(if_missing=None)
    # запретить отключение sms-2fa
    forbid_disabling_sms_2fa = validators.StringBool(if_missing=None)
    # разрешён ли доступ к удалению в takeout
    takeout_delete_subscription = validators.StringBool(if_missing=None)
    # разрешён ли публичный доступ к персональным данным
    personal_data_public_access_allowed = validators.StringBool(if_missing=None)
    # разрешена ли обработка персональных данных третьими лицами
    personal_data_third_party_processing_allowed = validators.StringBool(if_missing=None)
    family_pay_enabled = validators.Regex(r'^[\w-]+(,[\w-]+)*$', if_missing=None, if_empty='')

    plus_enabled = validators.StringBoolean(if_missing=None, not_empty=True)
    plus_trial_used_ts = validators.Unixtime(
        if_missing=None,
        if_empty=ZERO_DATETIME,
        max=DISTANT_FUTURE_DATETIME,
        not_empty=False,
    )
    plus_subscription_stopped_ts = validators.Unixtime(
        if_missing=None,
        if_empty=ZERO_DATETIME,
        max=DISTANT_FUTURE_DATETIME,
        not_empty=False,
    )
    plus_subscription_expire_ts = validators.Unixtime(
        if_missing=None,
        if_empty=ZERO_DATETIME,
        max=DISTANT_FUTURE_DATETIME,
        not_empty=False,
    )
    plus_next_charge_ts = validators.Unixtime(
        if_missing=None,
        if_empty=ZERO_DATETIME,
        max=DISTANT_FUTURE_DATETIME,
        not_empty=False,
    )
    ott_subscription = validators.OneOf(
        OTT_SUBSCRIPTIONS,
        if_missing=None,
        not_empty=False,
        if_empty='',
    )
    plus_family_role = validators.OneOf(
        PLUS_FAMILY_ROLES,
        if_missing=None,
        not_empty=False,
        if_empty='',
    )
    plus_cashback_enabled = validators.StringBoolean(if_missing=None, if_empty=False)
    plus_subscription_level = validators.Int(if_missing=None, not_empty=True, min=0, max=1000)
    plus_is_frozen = validators.StringBoolean(if_missing=None, not_empty=True)
    plus_subscriber_state = validators.PlusSubscriberStateValidator(if_missing=None)
    is_documents_agreement_accepted = validators.StringBool(if_missing=None, not_empty=True)
    is_dzen_sso_prohibited = validators.StringBool(if_missing=None, not_empty=True)
    can_manage_children = validators.StringBool(if_missing=None, not_empty=True)

    chained_validators = [
        validators.RequireSet([('admin_name', 'comment')], allow_empty=True),
        validators.RequireSome(
            [
                'is_enabled',
                'is_app_password_enabled',
                'is_shared',
                'is_maillist',
                'is_mailbox_frozen',
                'is_connect_admin',
                'is_easily_hacked',
                'is_password_change_required',
                'password_update_timestamp',
                'global_logout',
                'revoke_tokens',
                'revoke_app_passwords',
                'revoke_web_sessions',
                'audience_on',
                'external_organization_ids',
                'magic_link_login_forbidden',
                'qr_code_login_forbidden',
                'sms_code_login_forbidden',
                'takeout_subscription',
                'billing_features',
                'show_fio_in_public_name',
                'force_challenge',
                'sms_2fa_on',
                'forbid_disabling_sms_2fa',
                'takeout_delete_subscription',
                'personal_data_public_access_allowed',
                'personal_data_third_party_processing_allowed',
                'family_pay_enabled',
                'plus_enabled',
                'plus_trial_used_ts',
                'plus_subscription_stopped_ts',
                'plus_subscription_expire_ts',
                'plus_next_charge_ts',
                'ott_subscription',
                'plus_family_role',
                'plus_cashback_enabled',
                'plus_subscription_level',
                'plus_is_frozen',
                'plus_subscriber_state',
                'is_documents_agreement_accepted',
                'is_dzen_sso_prohibited',
                'can_manage_children',
            ],
            min_=1,
            max_=42,
        ),
        validators.RequireSome(['uid', 'multisession_uid'], min_=0, max_=1),
        validators.RequireIfEquals(['is_password_change_required'], 'notify_by_sms', True),
        validators.RequireIfPresent('is_password_change_required', present='show_2fa_promo'),
        validators.RequireIfPresent('is_password_change_required', present='max_change_frequency_in_days'),
    ]


class PlusChangeAttributesForm(validators.Schema):
    uid = validators.Uid()
    plus_enabled = validators.StringBoolean(if_missing=None, not_empty=True)
    plus_trial_used_ts = validators.Unixtime(
        if_missing=None,
        if_empty=ZERO_DATETIME,
        max=DISTANT_FUTURE_DATETIME,
        not_empty=False,
    )
    plus_subscription_stopped_ts = validators.Unixtime(
        if_missing=None,
        if_empty=ZERO_DATETIME,
        max=DISTANT_FUTURE_DATETIME,
        not_empty=False,
    )
    plus_subscription_expire_ts = validators.Unixtime(
        if_missing=None,
        if_empty=ZERO_DATETIME,
        max=DISTANT_FUTURE_DATETIME,
        not_empty=False,
    )
    plus_next_charge_ts = validators.Unixtime(
        if_missing=None,
        if_empty=ZERO_DATETIME,
        max=DISTANT_FUTURE_DATETIME,
        not_empty=False,
    )
    ott_subscription = validators.OneOf(
        OTT_SUBSCRIPTIONS,
        if_missing=None,
        not_empty=False,
        if_empty='',
    )
    plus_family_role = validators.OneOf(
        PLUS_FAMILY_ROLES,
        if_missing=None,
        not_empty=False,
        if_empty='',
    )
    plus_cashback_enabled = validators.StringBoolean(if_missing=None, if_empty=False)
    plus_subscription_level = validators.Int(if_missing=None, not_empty=True, min=0, max=1000)
    plus_is_frozen = validators.StringBoolean(if_missing=None, not_empty=True)

    chained_validators = [
        validators.RequireSome(
            [
                'plus_enabled',
                'plus_trial_used_ts',
                'plus_subscription_stopped_ts',
                'plus_subscription_expire_ts',
                'plus_next_charge_ts',
                'ott_subscription',
                'plus_family_role',
                'plus_cashback_enabled',
                'plus_subscription_level',
                'plus_is_frozen',
            ],
            min_=1,
            max_=10,
        ),
    ]
