# -*- coding: utf-8 -*-
from datetime import datetime
from time import time

from passport.backend.api.common.mail import is_mail_occupied_by_another_user
from passport.backend.api.views.bundle.exceptions import (
    Account2FAEnabledError,
    AccountInvalidTypeError,
    AccountNotSubscribedError,
    ActionImpossibleError,
    SecurePhoneNotFoundError,
    TooFrequentPasswordChangeError,
)
from passport.backend.core.builders.blackbox import get_blackbox
from passport.backend.core.builders.datasync_api import DiskApi
from passport.backend.core.builders.datasync_api.exceptions import BaseDatasyncApiError
from passport.backend.core.conf import settings
from passport.backend.core.models.account import (
    get_preferred_language,
    MAIL_STATUS_ACTIVE,
    MAIL_STATUS_FROZEN,
)
from passport.backend.core.services import Service
from passport.backend.utils.time import (
    datetime_to_unixtime,
    unixtime_to_datetime,
)

from .base_handlers import (
    BaseFlaggedHandler,
    BaseValueHandler,
)
from .grants import ACCOUNT_SHOW_2FA_PROMO_GRANT


SMS_NOTIFICATION_TANKER_KEY = 'password_compromised_sms'


class DatetimeNowHandler(BaseFlaggedHandler):
    def set_option(self):
        self.option = datetime.now()


class UnixtimeHandler(BaseValueHandler):
    def set_option(self):
        self.option = unixtime_to_datetime(self.option_form_value)


class DisableAuthMethodsHandler(BaseValueHandler):
    def set_option(self):
        if self.account.is_superlite or self.account.is_neophonish:
            raise AccountInvalidTypeError()
        super(DisableAuthMethodsHandler, self).set_option()

    def post_update(self):
        self.view.send_account_modification_push(
            event_name='login_method_change',
        )
        self.view.send_account_modification_mail(
            event_name='login_method_change',
        )


class IsMailListHandler(BaseValueHandler):
    def set_option(self):
        if not self.account.is_pdd or self.account.is_pdd_admin or self.account.is_connect_admin:
            raise AccountInvalidTypeError()
        if self.option_form_value is False and self.view.get_saml_settings(domain_id=self.account.domain.id, only_enabled=True):
            raise ActionImpossibleError()
        if self.option_form_value is not None:
            self.account.is_employee = not self.option_form_value
        super(IsMailListHandler, self).set_option()


class IsMailboxFrozenHandler(BaseValueHandler):
    def pre_update(self):
        # чтоб поймать именно событие удаление флага и не реагировать на None проверяю на False
        if self.form_values['is_mailbox_frozen'] is False and self.account.mail_status == MAIL_STATUS_FROZEN:
            if is_mail_occupied_by_another_user(get_blackbox(), self.account):
                raise ActionImpossibleError('Mailbox is already used by another user')

    def update(self):
        if not self.is_work_required():
            return

        if self.account.is_pdd:
            raise AccountInvalidTypeError()
        if not (
            self.account.mail_status or
            self.account.is_subscribed(Service.by_slug('mail'))
        ):
            raise AccountNotSubscribedError()

        self.account.mail_status = MAIL_STATUS_FROZEN if self.option_form_value else MAIL_STATUS_ACTIVE


class IsPasswordChangeRequiredHandler(BaseValueHandler):
    def get_sms_notification_text(self, language):
        return settings.translations.SMS[language][SMS_NOTIFICATION_TANKER_KEY]

    def send_notifications(self):
        notifications = {}
        if self.view.has_secure_number:
            language = get_preferred_language(self.account)
            sms_text = self.get_sms_notification_text(language)
            self.view.send_sms(self.view.secure_number, sms_text, 'password_options')
            notifications['is_sms_sent'] = True

        return notifications

    def check_password_change_frequency(self, not_frequently_than_N_days):
        """
        Проверим, когда последний раз менялся пароль.
        Бросим ошибку, если слишком часто
        """
        if not_frequently_than_N_days is None:
            return  # Ничего не проверяем, если пришел None

        if self.view.account.password.update_datetime:
            last_password_change_ts = datetime_to_unixtime(
                self.account.password.update_datetime,
            )

            delta_ts = settings.TIMESTAMP_DELTA_ONE_DAY * not_frequently_than_N_days
            is_too_early = time() < last_password_change_ts + delta_ts
            if is_too_early:
                raise TooFrequentPasswordChangeError()

    def pre_update(self):
        if self.is_work_required():
            self.check_grants()
            if self.form_values['show_2fa_promo'] is not None:
                self.view.check_grant(ACCOUNT_SHOW_2FA_PROMO_GRANT)

            # PASSP-10198 - принудительная смена пароля не имеет смысла для пользователя со включенным 2FA
            # PASSP-21607 - также нет смысла отправлять на смену пользователей без пароля
            # PASSP-24857 - снимать принуждение позволяем всегда
            if self.option_form_value and (self.account.totp_secret.is_set or not self.account.password.is_set):
                raise AccountInvalidTypeError()

            if self.option_form_value:
                self.check_password_change_frequency(self.form_values['max_change_frequency_in_days'])

    def set_option(self):
        self.account.password.setup_password_changing_requirement(is_required=self.option_form_value)
        if self.form_values['show_2fa_promo'] is not None:
            self.account.show_2fa_promo = self.form_values['show_2fa_promo']

    def post_update(self):
        if self.option_form_value and self.form_values['notify_by_sms']:
            notifications = self.send_notifications()
            self.view.response_values['notifications'] = notifications


class PlusEnableDisableHandler(BaseValueHandler):
    """Включаем или выключаем бонус от Диска и Денег для Плюса"""
    def post_update(self):
        if self.account.is_yandexoid or settings.DISK_PLUS_ENABLED:
            try:
                disk_api = DiskApi(retries=1)  # Диск в этих ручках подвержен гонкам, так что не ретраимся
                if self.option:
                    disk_api.plus_subscribe(self.account.uid, settings.DISK_PLUS_PARTNER_ID, settings.DISK_PLUS_PRODUCT_ID)
                else:
                    disk_api.plus_unsubscribe(self.account.uid, settings.DISK_PLUS_PARTNER_ID, settings.DISK_PLUS_PRODUCT_ID)
            except BaseDatasyncApiError:
                pass


class Sms2FaOnHandler(BaseValueHandler):
    def set_option(self):
        if self.option_form_value:
            if not self.view.has_secure_number:
                raise SecurePhoneNotFoundError()
            elif self.account.totp_secret.is_set:
                raise Account2FAEnabledError()
            elif self.account.is_neophonish:
                raise AccountInvalidTypeError()
        else:
            if self.account.forbid_disabling_sms_2fa:
                raise ActionImpossibleError()
        super(Sms2FaOnHandler, self).set_option()

    def post_update(self):
        self.view.send_account_modification_push(
            event_name='login_method_change',
        )
        self.view.send_account_modification_mail(
            event_name='login_method_change',
        )


class CanManageChildrenHandler(BaseValueHandler):
    def set_option(self):
        if self.account.is_child:
            raise AccountInvalidTypeError()
