# -*- coding: utf-8 -*-

from datetime import datetime
import logging
import time

from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import (
    AccountInvalidTypeError,
    ActionNotRequiredError,
    TooFrequentPasswordChangeError,
)
from passport.backend.api.views.bundle.mixins import (
    BundleAccountGetterMixin,
    BundlePhoneMixin,
)
from passport.backend.api.views.bundle.mixins.common import BundleAdminActionMixin
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.loggers.statbox import StatboxLogger
from passport.backend.core.models.account import get_preferred_language
from passport.backend.core.models.password import PASSWORD_CHANGING_REASON_BY_NAME
from passport.backend.core.runner.context_managers import UPDATE
from passport.backend.core.utils.decorators import cached_property
from passport.backend.utils.time import (
    datetime_to_unixtime,
    unixtime_to_datetime,
)

from .forms import PasswordOptionsForm


log = logging.getLogger('passport.api.view.bundle.account.change_password_options')

PASSWORD_OPTIONS_BASE_GRANT = 'password_options.base'
PASSWORD_IS_CHANGING_REQUIRED_GRANT = 'password.is_changing_required'
PASSWORD_CHANGING_REQUIREMENT_REASON_GRANT = 'password.changing_requirement_reason'
ACCOUNT_GLOBAL_LOGOUT_GRANT = 'account.global_logout'
ACCOUNT_REVOKE_TOKENS_GRANT = 'account.revoke_tokens'
ACCOUNT_REVOKE_APP_PASSWORDS_GRANT = 'account.revoke_app_passwords'
ACCOUNT_REVOKE_WEB_SESSIONS_GRANT = 'account.revoke_web_sessions'
PASSWORD_UPDATE_DATETIME_GRANT = 'password.update_datetime'
SMS_NOTIFICATION_TANKER_KEY = 'password_compromised_sms'
LOG_ADMIN_ACTION_GRANT = 'admin.log_action'
ACCOUNT_SHOW_2FA_PROMO_GRANT = 'account.show_2fa_promo'


class PasswordOptionsView(BaseBundleView,
                          BundlePhoneMixin,
                          BundleAccountGetterMixin,
                          BundleAdminActionMixin):

    require_track = False
    required_grants = [PASSWORD_OPTIONS_BASE_GRANT]

    basic_form = PasswordOptionsForm

    @cached_property
    def statbox(self):
        return StatboxLogger(
            uid=self.account.uid,
            mode='password_options',
            consumer=self.consumer,
        )

    def get_sms_notification_text(self, language):
        return settings.translations.SMS[language][SMS_NOTIFICATION_TANKER_KEY]

    def send_notifications(self):
        notifications = {}
        if self.has_secure_number:
            log.debug('Send information with sms to account\'s secure phone number')
            language = get_preferred_language(self.account)
            sms_text = self.get_sms_notification_text(language)
            self.send_sms(self.secure_number, sms_text, 'password_options')
            notifications['is_sms_sent'] = True

        return notifications

    def check_password_change_frequency(self, not_frequently_than_N_days):
        """Проверим, когда последний раз менялся пароль. Бросим ошибку, если слишком часто"""
        if not_frequently_than_N_days is None:
            return  # Ничего не проверяем, если пришел None

        if self.account.password.update_datetime:
            last_password_change_ts = datetime_to_unixtime(self.account.password.update_datetime)
            if self.account.registration_datetime == self.account.password.update_datetime:
                return

            delta_ts = settings.TIMESTAMP_DELTA_ONE_DAY * not_frequently_than_N_days
            is_too_early = time.time() < (last_password_change_ts + delta_ts)
            if is_too_early:
                raise TooFrequentPasswordChangeError()

    def process_request(self, *args, **kwargs):
        self.process_basic_form()
        if self.is_admin_action:
            self.check_grant(LOG_ADMIN_ACTION_GRANT)

        self.get_account_by_uid(
            self.form_values['uid'],
            enabled_required=False,
            need_phones=True,
        )

        is_changing_required = self.form_values['is_changing_required']
        changing_requirement_reason = self.form_values['changing_requirement_reason']
        max_change_frequency_in_days = self.form_values['max_change_frequency_in_days']
        update_timestamp = self.form_values['update_timestamp']
        show_2fa_promo = self.form_values['show_2fa_promo']
        global_logout = self.form_values['global_logout']
        revoke_tokens = self.form_values['revoke_tokens']
        revoke_app_passwords = self.form_values['revoke_app_passwords']
        revoke_web_sessions = self.form_values['revoke_web_sessions']
        comment = self.form_values['comment']

        events = {
            'action': 'password',
            'consumer': self.consumer,
            'comment': comment,
        }

        self.mark_admin_action(events)

        with UPDATE(self.account, self.request.env, events):
            if is_changing_required is not None:
                # Принудительная смена пароля не имеет смысла для пользователя со включенным 2FA. (PASSP-10198)
                # Также нет смысла отправлять на смену пользователей без пароля (PASSP-21607)
                # Снимать принуждение позволяем всегда (PASSP-24857)
                if is_changing_required and (self.account.totp_secret.is_set or not self.account.password.is_set):
                    raise AccountInvalidTypeError()

                self.check_grant(PASSWORD_IS_CHANGING_REQUIRED_GRANT)
                if is_changing_required:
                    if self.account.password.is_changing_required:
                        # PASSP-13047 - не выставлять glogout, если передали is_changing_required=1 и он уже установлен.
                        # чтобы не плодить записи в historydb о glogout
                        raise ActionNotRequiredError()

                    self.check_password_change_frequency(max_change_frequency_in_days)
                requirement_kwargs = dict(is_required=is_changing_required)

                if changing_requirement_reason:
                    self.check_grant(PASSWORD_CHANGING_REQUIREMENT_REASON_GRANT)
                    changing_reason = PASSWORD_CHANGING_REASON_BY_NAME[changing_requirement_reason]
                    requirement_kwargs.update(changing_reason=changing_reason)

                self.account.password.setup_password_changing_requirement(**requirement_kwargs)

                if show_2fa_promo is not None:
                    self.check_grant(ACCOUNT_SHOW_2FA_PROMO_GRANT)
                    self.account.show_2fa_promo = show_2fa_promo

            if update_timestamp is not None:
                self.check_grant(PASSWORD_UPDATE_DATETIME_GRANT)
                update_datetime = unixtime_to_datetime(update_timestamp)
                self.account.password.update_datetime = update_datetime

            if global_logout:
                self.check_grant(ACCOUNT_GLOBAL_LOGOUT_GRANT)
                self.account.global_logout_datetime = datetime.now()

            if revoke_tokens:
                self.check_grant(ACCOUNT_REVOKE_TOKENS_GRANT)
                self.account.tokens_revoked_at = datetime.now()

            if revoke_app_passwords:
                self.check_grant(ACCOUNT_REVOKE_APP_PASSWORDS_GRANT)
                self.account.app_passwords_revoked_at = datetime.now()

            if revoke_web_sessions:
                self.check_grant(ACCOUNT_REVOKE_WEB_SESSIONS_GRANT)
                self.account.web_sessions_revoked_at = datetime.now()

        if is_changing_required and self.form_values.get('notify_by_sms'):
            notifications = self.send_notifications()
            self.response_values['notifications'] = notifications
