# -*- coding: utf-8 -*-
from datetime import datetime
import logging

from passport.backend.api.common.processes import PROCESS_RESTORE
from passport.backend.api.views.bundle.exceptions import (
    AccountDisabledError,
    AccountInvalidTypeError,
    AccountNotFoundError,
    AccountUidMismatchError,
    UserNotVerifiedError,
)
from passport.backend.api.views.bundle.mixins.common import BundleAdminActionMixin
from passport.backend.core.runner.context_managers import UPDATE
from passport.backend.core.types.phone_number.phone_number import (
    InvalidPhoneNumber,
    PhoneNumber,
)
from passport.backend.utils.time import unixtime_to_datetime

from .base import BaseAccountPhonishBundleView
from .forms import (
    AccountPhonishCanLoginForm,
    AccountPhonishDisableAuthForm,
    AccountPhonishUidByPhoneForm,
)


# Логически это время "нулевое", то есть меньше него времён не ожидаем. И номер, подтверждённый в это время,
# гарантированно будет достаточно старым, чтобы фониш не мог по нему авторизоваться.
# Но технически тут нужно нечто >=1, чтобы не спутать с отсутствием значения.
ZERO_DATETIME = unixtime_to_datetime(1)

log = logging.getLogger('passport.api.view.bundle.account.phonish')


class AccountPhonishCanLoginView(BaseAccountPhonishBundleView):
    basic_form = AccountPhonishCanLoginForm
    required_grants = ['phonish.can_login']

    def process_request(self):
        self.process_basic_form()
        account, _ = self.get_account_by_phone_number(self.form_values['phone_number'])
        if account is None:
            raise AccountNotFoundError()
        elif not account.is_enabled:
            raise AccountDisabledError()
        if account.uid != self.form_values['uid']:
            raise AccountUidMismatchError(known_uids=[account.uid])


class AccountPhonishUidByPhoneView(BaseAccountPhonishBundleView):
    basic_form = AccountPhonishUidByPhoneForm
    require_track = True

    required_grants = ['phonish.uid_by_phone']
    allowed_processes = [PROCESS_RESTORE]  # для связок с фонишами после рестора

    def process_request(self):
        self.process_basic_form()
        self.read_track()
        if not self.is_phone_confirmed_in_track():
            raise UserNotVerifiedError()

        try:
            phone_number = PhoneNumber.parse(self.track.phone_confirmation_phone_number)
        except InvalidPhoneNumber:
            raise AccountNotFoundError()

        # Временно, чтобы посмотреть, как часто здесь в треке не будет device_id
        use_device_id = self.form_values['use_device_id']
        if use_device_id and not self.track.device_id:
            log.debug('Was asked for use_device_id, but track.device_id is None')

        account, _ = self.get_account_by_phone_number(
            phone_number,
            device_id=self.track.device_id if use_device_id else None,
        )
        if account is None:
            raise AccountNotFoundError()
        elif not account.is_enabled:
            raise AccountDisabledError()

        self.response_values['uid'] = account.uid


class AccountPhonishDisableAuth(BaseAccountPhonishBundleView, BundleAdminActionMixin):
    basic_form = AccountPhonishDisableAuthForm
    required_grants = ['phonish.disable_auth']

    def process_request(self):
        self.process_basic_form()
        if self.is_admin_action:
            self.check_grant('admin.log_action')

        self.get_account_by_uid(self.form_values['uid'], enabled_required=False, need_phones=True)
        if not self.account.is_phonish:
            raise AccountInvalidTypeError()

        events = {
            'action': 'disable_phonish_auth',
            'consumer': self.consumer,
        }
        self.mark_admin_action(events)
        with UPDATE(self.account, self.request.env, events):
            self.account.global_logout_datetime = datetime.now()
            # У фониша должен быть ровно один номер. Но перестрахуемся на случай всяких ненатуралов.
            for phone in self.account.phones.all().values():
                phone.confirmed = ZERO_DATETIME


class AccountPhonishDisableAuthByXToken(BaseAccountPhonishBundleView):
    required_grants = ['phonish.disable_auth_by_xtoken']

    def process_request(self):
        self.get_account_by_oauth_token(enabled_required=False)

        if not self.account.is_phonish:
            raise AccountInvalidTypeError()

        events = {
            'action': 'disable_phonish_auth_by_xtoken',
            'consumer': self.consumer,
        }

        with UPDATE(self.account, self.request.env, events):
            self.account.global_logout_datetime = datetime.now()
            # У фониша должен быть ровно один номер. Но перестрахуемся на случай всяких ненатуралов.
            for phone in self.account.phones.all().values():
                phone.confirmed = ZERO_DATETIME
