# -*- coding: utf-8 -*-
import json
import logging

from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import (
    AccountInvalidTypeError,
    AccountNoAnswerError,
    AccountNoQuestionError,
    AnswerEmptyError,
    CaptchaRequiredError,
    CompareNotMatchedError,
    InvalidTrackStateError,
    RateLimitExceedError,
)
from passport.backend.api.views.bundle.headers import (
    HEADER_CLIENT_HOST,
    HEADER_CLIENT_USER_AGENT,
    HEADER_CONSUMER_CLIENT_IP,
)
from passport.backend.api.views.bundle.mixins import (
    BundleAccountGetterMixin,
    BundleAssertCaptchaMixin,
    BundleHintAnswerCheckMixin,
    BundlePhoneMixin,
)
from passport.backend.api.views.bundle.mixins.account import (
    BundleAccountPropertiesMixin,
    BundleAccountResponseRendererMixin,
    UserMetaDataMixin,
)
from passport.backend.api.views.bundle.mixins.kolmogor import KolmogorMixin
from passport.backend.api.views.bundle.mixins.mail import MailMixin
from passport.backend.api.views.bundle.mixins.password import BundleAuthenticateMixin
from passport.backend.api.views.bundle.mixins.push import BundlePushMixin
from passport.backend.api.views.bundle.utils import make_hint_question
from passport.backend.core.conf import settings
from passport.backend.core.counters import (
    check_answer_counter,
    question_change_email_counter,
)
from passport.backend.core.host.host import get_current_host
from passport.backend.core.logging_utils.loggers.statbox import StatboxLogger
from passport.backend.core.mailer.utils import (
    get_tld_by_country,
    login_shadower,
    MailInfo,
    make_email_context,
    send_mail_for_account,
)
from passport.backend.core.models.account import get_preferred_language
from passport.backend.core.models.hint import Hint
from passport.backend.core.runner.context_managers import UPDATE
from passport.backend.core.serializers.logs.historydb.runner import HistorydbActionRunner
from passport.backend.core.utils.decorators import cached_property

from .forms import (
    QuestionsAddOptionalForm,
    QuestionsChangeForm,
    QuestionsCheckAnswerForm,
    QuestionsGetQuestionForm,
    QuestionsSetQuestionForm,
)


log = logging.getLogger('passport.api.view.bundle.account.questions')

QUESTIONS_BASE_GRANT = 'questions.base'
QUESTIONS_CHANGE_GRANT = 'questions.change'
QUESTIONS_EDIT_NORMAL_GRANT = 'questions.edit_normal'
QUESTIONS_EDIT_PDD_GRANT = 'questions.edit_pdd'
QUESTIONS_OPTIONAL_GRANT = 'questions.optional'
QUESTIONS_SECURE_GRANT = 'questions.secure'
QUESTIONS_SECURE_BY_UID_GRANT = 'questions.secure_by_uid'


class QuestionsSetQuestionView(
    MailMixin,
    UserMetaDataMixin,
    BaseBundleView,
    BundleAccountGetterMixin,
    KolmogorMixin,
    BundlePushMixin,
):

    require_track = False
    required_headers = (
        HEADER_CONSUMER_CLIENT_IP,
        HEADER_CLIENT_USER_AGENT,
    )
    required_grants = [QUESTIONS_BASE_GRANT]

    basic_form = QuestionsSetQuestionForm

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode='questions_set',
            action='set_hintq',
            ip=self.client_ip,
            user_agent=self.user_agent,
        )

    def process_request(self, *args, **kwargs):
        self.process_basic_form()
        self.statbox.log(action='submitted')

        self.get_account_from_uid_or_session(
            by_uid_grant=QUESTIONS_EDIT_PDD_GRANT,
            emails=True,
        )

        if self.account.is_pdd:
            self.check_grant(QUESTIONS_EDIT_PDD_GRANT)
        elif self.account.is_normal:
            self.check_grant(QUESTIONS_EDIT_NORMAL_GRANT)
        else:
            raise AccountInvalidTypeError()

        if not self.account.is_pdd and self.form_values['answer'] is None:
            raise AnswerEmptyError()

        question = None
        if not self.account.is_pdd or self.form_values['question'] or self.form_values['question_id']:
            question = make_hint_question(
                question=self.form_values['question'],
                question_id=self.form_values['question_id'],
                display_language=self.form_values['display_language'],
            )

        events = {
            'action': 'questions_set',
            'consumer': self.consumer,
        }
        with UPDATE(self.account, self.request.env, events):
            if question is not None:
                self.account.hint.question = question
            if self.form_values['answer'] is not None:
                self.account.hint.answer = self.form_values['answer']

        self.send_account_modification_push(
            event_name='hint_change',
        )
        self.send_account_modification_mail(event_name='hint_change')
        self.statbox.log(action='hint_set')


class QuestionsChangeView(
    BaseBundleView,
    BundleAccountGetterMixin,
    BundleAccountResponseRendererMixin,
    BundleAccountPropertiesMixin,
    BundleHintAnswerCheckMixin,
    BundlePhoneMixin,
    BundleAuthenticateMixin,
    KolmogorMixin,
    BundlePushMixin,
):
    required_headers = (
        HEADER_CONSUMER_CLIENT_IP,
        HEADER_CLIENT_HOST,
        HEADER_CLIENT_USER_AGENT,
    )
    required_grants = [QUESTIONS_BASE_GRANT, QUESTIONS_CHANGE_GRANT]
    require_track = True
    track_type = 'authorize'

    basic_form = QuestionsChangeForm

    email_template = 'mail/account_hint_notification.html'
    email_subject_key_created = 'hint_created.subject'
    email_subject_key_changed = 'hint_changed.subject'

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode='questions_change',
            action='change_hint',
            ip=self.client_ip,
            user_agent=self.user_agent,
            yandexuid=self.cookies.get('yandexuid'),
            host=self.host,
        )

    def send_email_notifications(self, language=None, is_hint_created=False):
        language = language or get_preferred_language(self.account)
        template_name, info, context = self.get_email_notification_data(language, is_hint_created)
        send_mail_for_account(template_name, info, context, self.account, login_shadower, send_to_native=True)

    def get_email_notification_data(self, language, is_hint_created):
        """Возвращает template_name, info, context"""
        translations = settings.translations.NOTIFICATIONS[language]
        template_name = self.email_template
        subject_key = self.email_subject_key_created if is_hint_created else self.email_subject_key_changed

        info = MailInfo(
            subject=translations[subject_key],
            from_=translations['email_sender_display_name'],
            tld=get_tld_by_country(self.account.person.country),
        )
        context = make_email_context(
            language=language,
            account=self.account,
            context={
                'is_2fa_on': self.account.totp_secret.is_set,
                'is_hint_created': is_hint_created,
            },
        )
        return template_name, info, context

    def process_request(self, *args, **kwargs):
        self.statbox.log(action='submitted')
        self.process_basic_form()

        self.read_track()

        if not self.track.uid:
            raise InvalidTrackStateError('No uid in track')

        self.get_account_from_session(
            multisession_uid=int(self.track.uid),
            check_disabled_on_deletion=True,
            emails=True,
        )
        self.statbox.bind(
            track_id=self.track_id,
            uid=self.account.uid,
        )
        # Проверим состояние глобальных счетчиков, не увеличивая их
        if check_answer_counter.is_limit_exceeded(self.client_ip, self.account.uid, increment=False):
            self.statbox.log(action='limit_exceeded')
            raise RateLimitExceedError()

        # Пароля может не быть у аккаунта, отправляем на дорегистрацию или кидаем ошибку
        self.state = self.check_have_password()
        if self.state:
            # Отдаем инфу для дорегистрации социального пользователя
            self.response_values['track_id'] = self.track_id
            self.fill_response_with_account()
            return

        # Проверяем текущий ответ на КВ
        with self.track_transaction.commit_on_error():
            if self.account.hint.is_set and not self.compare_answers():
                # Увеличиваем счетчики на капчу
                check_answer_counter.increment_counters(
                    self.client_ip,
                    self.account.uid,
                )
                raise CompareNotMatchedError()

        # Сохраняем новый КВ/КО
        language = self.form_values['display_language']
        events = {
            'action': 'questions_change',
            'consumer': self.consumer,
        }
        is_hint_created = not self.account.hint.is_set
        with UPDATE(self.account, self.request.env, events):
            self.account.hint = Hint(parent=self.account)
            self.account.hint.question = make_hint_question(
                question=self.form_values['question'],
                question_id=self.form_values['question_id'],
                display_language=language,
            )
            self.account.hint.answer = self.form_values['new_answer']

        # Передадим фронту знание о необходимости привязки телефона
        if not self.account_phones.suitable_for_restore:
            self.response_values.update(
                need_to_bind_phone=True,
            )
        # Отправляем письмо с подтверждением
        if settings.SEND_EMAIL_AFTER_HINT_QUESTION_CHANGE:
            if not question_change_email_counter.hit_limit(self.account.uid):
                self.send_email_notifications(
                    language=language,
                    is_hint_created=is_hint_created,
                )
                question_change_email_counter.incr(self.account.uid)

        self.send_account_modification_push(
            event_name='hint_change',
            context=dict(track_id=self.track_id),
        )
        self.statbox.log(
            action='committed',
            authid=self.session_info.authid,
        )


class QuestionsAddOptional(BaseBundleView, BundleAccountGetterMixin):

    require_track = False

    required_headers = (
        HEADER_CONSUMER_CLIENT_IP,
    )
    required_grants = [QUESTIONS_BASE_GRANT, QUESTIONS_OPTIONAL_GRANT]

    basic_form = QuestionsAddOptionalForm

    EVENT_NAME = 'info.hints_optional'

    def process_request(self, *args, **kwargs):
        self.process_basic_form()
        self.get_account_from_session(
            check_disabled_on_deletion=True,
            multisession_uid=self.form_values['uid'],
        )

        values = {k: v for k, v in self.form_values.items() if k != 'uid'}

        event_value = json.dumps(values)
        runner = HistorydbActionRunner(
            {},
            uid=self.account.uid,
            user_ip=self.client_ip,
            user_agent=self.user_agent,
            yandexuid=self.cookies.get('yandexuid'),
            host_id=get_current_host().get_id(),
        )
        runner.execute((self.EVENT_NAME, event_value))


class QuestionsGetQuestionView(BaseBundleView, BundleAccountGetterMixin):

    required_grants = [QUESTIONS_BASE_GRANT, QUESTIONS_SECURE_GRANT]

    basic_form = QuestionsGetQuestionForm

    def _fill_track_with_account_data(self):
        self.set_uid_to_track(self.account.uid)

    def _get_question(self):
        hint = self.account.hint
        if not hint or not hint.question:
            raise AccountNoQuestionError()
        self.response_values['question'] = dict(
            id=hint.question.id,
            text=hint.question.text,
        )

    def process_request(self, *args, **kwargs):
        self.process_basic_form()
        self.read_or_create_track('authorize')
        self.response_values['track_id'] = self.track_id

        with self.track_transaction.commit_on_error():
            self.get_account_from_track_or_uid_or_session(
                by_uid_grant=QUESTIONS_SECURE_BY_UID_GRANT,
                check_disabled_on_deletion=True,
                multisession_uid=self.track_uid if self.track.uid else None,
            )
            self._fill_track_with_account_data()
            self._get_question()


class QuestionsCheckAnswerView(BaseBundleView,
                               BundleHintAnswerCheckMixin,
                               BundleAccountGetterMixin):

    require_track = True
    track_type = 'authorize'

    required_grants = [QUESTIONS_BASE_GRANT, QUESTIONS_SECURE_GRANT]

    required_headers = [HEADER_CONSUMER_CLIENT_IP]

    basic_form = QuestionsCheckAnswerForm

    def on_compare_not_matched(self):
        pass

    def _check_limit(self):
        if check_answer_counter.is_limit_exceeded(self.client_ip, self.account.uid):
            self.statbox.log(action='limit_exceeded')
            raise RateLimitExceedError()

    def process_request(self, *args, **kwargs):
        self.process_basic_form()
        self.read_track()

        self.get_account_from_track(check_disabled_on_deletion=True)
        self.statbox = StatboxLogger(
            mode='account_questions_secure_check',
            track_id=self.track.track_id,
            uid=self.account.uid,
            ip=self.client_ip,
        )

        with self.track_transaction.commit_on_error():
            if not self.account.hint.is_set:
                raise AccountNoAnswerError()

            self._check_limit()
            compare_result = self.compare_answers()

            self.statbox.log(
                action='compared',
                check_passed=compare_result,
            )
            if not compare_result:
                self.on_compare_not_matched()
                raise CompareNotMatchedError()

            self.track.is_secure_hint_answer_checked = True


class QuestionsCheckAnswerViewV2(QuestionsCheckAnswerView, BundleAssertCaptchaMixin):

    def _check_limit(self):
        # Проверим состояние глобальных счетчиков, не увеличивая их
        if check_answer_counter.is_limit_exceeded(self.client_ip, self.account.uid, increment=False):
            self.statbox.log(action='limit_exceeded')
            raise RateLimitExceedError()

        self.check_track_for_captcha()
        # Если капча не пройдена - надо проверить лимит на капчу
        if not self.is_captcha_passed:
            # Проверяем счетчики на капчу, но не увеличиваем их
            if check_answer_counter.is_limit_exceeded(
                self.client_ip,
                self.account.uid,
                ip_limit=settings.CHECK_ANSWER_UNTIL_CAPTCHA_LIMIT_BY_IP,
                uid_limit=settings.CHECK_ANSWER_UNTIL_CAPTCHA_LIMIT_BY_UID,
                increment=False,
            ):
                self.track.is_captcha_required = True
                self.invalidate_captcha()
                self.statbox.log(action='captcha_limit_exceeded')
                raise CaptchaRequiredError()

    def on_compare_not_matched(self):
        check_answer_counter.get_per_uid_buckets().incr(self.account.uid)
        check_answer_counter.get_per_ip_buckets().incr(self.client_ip)
        self.invalidate_captcha()
