# -*- coding: utf-8 -*-

from passport.backend.core import validators


class QuestionsAddOptionalForm(validators.Schema):
    """Форма задания дополнительных КВ/КО"""
    uid = validators.Uid(not_empty=True, if_missing=None)

    MAX_HINT_LENGTH = 512
    QUESTION_COUNT = 3

    def __init__(self, *args, **kwargs):
        question_base_field = 'question'
        answer_base_field = 'answer'
        for i in range(1, self.QUESTION_COUNT + 1):
            self.add_field(
                question_base_field + str(i),
                validators.HintString(max=self.MAX_HINT_LENGTH),
            )
            self.add_field(
                answer_base_field + str(i),
                validators.HintString(max=self.MAX_HINT_LENGTH),
            )
        super(QuestionsAddOptionalForm, self).__init__(*args, **kwargs)


class QuestionsSetQuestionForm(validators.Schema):
    """
    Форма установки КВ-КО на аккаунте.
    """
    uid = validators.Uid(if_missing=None)
    answer = validators.HintAnswer(if_missing=None)
    question_id = validators.Int(strip=True, not_empty=True, if_missing=None)
    question = validators.HintQuestion()
    display_language = validators.DisplayLanguage(if_missing=None)

    chained_validators = [
        validators.RequireSome(
            ['question', 'question_id', 'answer'],
            min_=1,
            max_=2,
            rule_name='question_or_answer',
        ),
        validators.RequireIfPresent('display_language', present='question_id'),
    ]


class QuestionsChangeForm(validators.Schema):
    """
    Форма изменения КВ-КО на аккаунте.
    """
    answer = validators.String(not_empty=True, strip=True, if_missing=None, max=1024)

    question_id = validators.HintQuestionId(strip=True)
    question = validators.HintQuestion()
    new_answer = validators.HintAnswer()
    display_language = validators.DisplayLanguage(if_missing=None)

    chained_validators = [
        validators.RequireSet(
            allowed_sets=[
                ['question'],
                ['question_id', 'display_language'],
            ],
        ),
    ]


class QuestionsGetQuestionForm(validators.Schema):
    """Форма получения КВ аккаунта"""
    uid = validators.Uid(if_missing=None, not_empty=True)
    is_pdd = validators.StringBool(if_missing=False, not_empty=True)


class QuestionsCheckAnswerForm(validators.Schema):
    """Форма проверки КО аккаунта"""
    # max=1024 предохранитель от слишком длинных входных данных
    answer = validators.String(max=1024, not_empty=True, strip=True)


class CheckQuestionAnswerHistoryForm(validators.Schema):
    question_id = validators.Int(not_empty=True, strip=True)
    # max=1024 предохранитель от слишком длинных входных данных
    question = validators.String(max=1024, not_empty=True, strip=True)
    answer = validators.String(max=1024, not_empty=True, strip=True)
