# -*- coding: utf-8 -*-
import logging

from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import (
    AccountInvalidTypeError,
    CompareNotMatchedError,
    InvalidTrackStateError,
    RateLimitExceedError,
)
from passport.backend.api.views.bundle.headers import HEADER_CONSUMER_CLIENT_IP
from passport.backend.api.views.bundle.mixins import (
    BundleAccountGetterMixin,
    BundlePhoneMixin,
)
from passport.backend.core.compare import (
    compare_answers,
    compress_string_factor,
)
from passport.backend.core.conf import settings
from passport.backend.core.counters import check_answer_per_ip_and_uid_counter
from passport.backend.core.logging_utils.loggers.statbox import StatboxLogger
from passport.backend.core.utils.decorators import cached_property

from .controllers import QUESTIONS_BASE_GRANT
from .forms import CheckQuestionAnswerHistoryForm


log = logging.getLogger('passport.api.view.bundle.account.questions')


class BaseQuestionsHistoryView(BaseBundleView, BundleAccountGetterMixin, BundlePhoneMixin):

    require_track = True

    required_headers = (
        HEADER_CONSUMER_CLIENT_IP,
    )

    # TODO: а какие гранты в точности нужны тут?
    required_grants = [QUESTIONS_BASE_GRANT]

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode='account_questions_history_check',
            track_id=self.track.track_id,
            uid=self.account.uid,
            ip=self.client_ip,
        )

    def check_track(self):
        # Пока эти ручки планируем использовать только в процессе
        # принудительной смены пароля. Ручки доступны из браузера.
        # Убедимся, что работаем только в режиме смены
        if not self.track.is_password_change:
            raise InvalidTrackStateError('Track is not for change password')

    def check_account(self):
        # Убедимся, что у пользователя нет защищенного телефона (PASSP-35087)
        if not settings.DONT_SHOW_HINT_IF_SECURE_NUMBER_AVAILABLE:
            return

        has_suitable_phone = bool(self.account_phones.suitable_for_restore)
        if has_suitable_phone:
            raise AccountInvalidTypeError('Account has secure phone')

    def _process(self):
        raise NotImplementedError()  # pragma: no cover

    def process_request(self, *args, **kwargs):
        self.read_track()
        self.check_track()
        if self.basic_form:
            self.process_basic_form()
        self.get_account_from_track(check_disabled_on_deletion=True, need_phones=True)
        self.check_account()
        with self.track_transaction.rollback_on_error():
            self._process()


class GetQuestionsHistoryView(BaseQuestionsHistoryView):
    def _process(self):
        self.response_values['questions'] = []

        if self.account.hint and self.account.hint.is_set:
            self.response_values['questions'].append({
                'id': 0,
                'text': self.account.hint.question.text,
            })


class CheckQuestionAnswerHistoryView(BaseQuestionsHistoryView):

    basic_form = CheckQuestionAnswerHistoryForm

    def stringify_compare_factors(self):
        return '; '.join([compress_string_factor(factor) for factor in self.compare_factors])

    def check_hint(self, entered_answer):
        compare_result = compare_answers(
            self.account.hint.answer,
            entered_answer,
            distance_threshold=settings.HINT_ANSWER_DISTANCE_THRESHOLD,
        )
        self.compare_factors.append(compare_result.factors)
        if not compare_result.status:
            raise CompareNotMatchedError()

    def _process(self):
        self.compare_factors = []
        entered_answer = self.form_values['answer']

        if check_answer_per_ip_and_uid_counter.is_limit_exceeded(self.client_ip, self.account.uid):
            self.statbox.log(action='limit_exceeded')
            raise RateLimitExceedError()

        try:
            self.check_hint(entered_answer)

            self.track.is_fuzzy_hint_answer_checked = True
            self.statbox.log(
                action='compared',
                check_passed=True,
                answer_compare_factors=self.stringify_compare_factors(),
            )
        except CompareNotMatchedError:
            self.statbox.log(
                action='compared',
                check_passed=False,
                answer_compare_factors=self.stringify_compare_factors(),
            )
            raise
