# -*- coding: utf-8 -*-
import json

import jinja2
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import ActionImpossibleError
from passport.backend.api.views.bundle.mixins import BundleAccountGetterMixin
from passport.backend.core.avatars import delete_avatar
from passport.backend.core.builders.clean_web_api import (
    BaseCleanWebError,
    get_clean_web_api,
)
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.loggers.statbox import StatboxLogger
from passport.backend.core.mailer.utils import (
    get_tld_by_country,
    MailInfo,
    make_email_context,
    send_mail_for_account,
)
from passport.backend.core.models.account import get_preferred_language
from passport.backend.core.models.email import Email
from passport.backend.core.runner.context_managers import UPDATE
from passport.backend.core.types.display_name import DisplayName
from passport.backend.core.utils.decorators import cached_property

from .forms import (
    AccountResetAvatarForm,
    AccountResetDisplayNameForm,
    AccountResetEmailForm,
    AccountResetPhoneForm,
    AccountResetQuestionForm,
)


class BaseResetView(BaseBundleView,
                    BundleAccountGetterMixin):
    action = None

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode=self.action,
            uid=self.form_values['uid'],
            consumer=self.consumer,
            ip=self.client_ip,
            user_agent=self.user_agent,
        )

    def is_applicable(self):
        return True

    def get_account(self):
        self.get_account_by_uid(
            uid=self.form_values['uid'],
            enabled_required=False,
            emails=True,  # запрашиваем всегда так как необходимо слать нотификацию пользователю
        )

    def _process(self):
        raise NotImplementedError()

    def process_request(self, *args, **kwargs):
        self.process_basic_form()
        self.get_account()
        if not self.is_applicable():
            raise ActionImpossibleError()

        self._process()

        self.statbox.log(action=self.action)


class ResetAvatarView(BaseResetView):
    required_grants = [
        'account.reset_avatar',
    ]
    basic_form = AccountResetAvatarForm
    action = 'avatar_reset'

    def send_email_notifications(self, language):
        translations = settings.translations.NOTIFICATIONS[language]
        template_name = 'mail/reset_avatar.html'
        info = MailInfo(
            subject=translations['change.avatar.notify.subject'],
            from_=translations['email_sender_display_name'],
            tld=get_tld_by_country(self.account.person.country),
        )

        context = make_email_context(
            language=language,
            account=None,
            context=dict(
                feedback_key='feedback',
                greeting_key='greeting.noname',
            )
        )

        send_mail_for_account(template_name, info, context, self.account, send_to_native=True)

    def _process(self):
        if not self.account.person or not self.account.person.default_avatar or self.account.person.is_avatar_empty:
            return

        if self.account.person.default_avatar != self.form_values['avatar_key']:
            return

        delete_avatar(self.account.person.default_avatar)

        events = {
            'action': 'delete_default_avatar',
            'consumer': self.consumer,
        }
        language = get_preferred_language(self.account)
        with UPDATE(self.account, self.request.env, events):
            self.account.person.default_avatar = None

        self.send_email_notifications(language)


class ResetDisplayNameView(BaseResetView):
    required_grants = [
        'account.reset_display_name',
    ]
    basic_form = AccountResetDisplayNameForm
    action = 'display_name_reset'

    def get_new_display_name(self, language):
        user = settings.translations.TEXTS[language]['displayname.name.censored']

        if self.account.person.lastname:
            lastname_symbol = self.account.person.lastname[0]
            return u'{} {}.'.format(user, lastname_symbol)
        else:
            return user

    @cached_property
    def clean_web_api(self):
        return get_clean_web_api()

    def is_applicable(self):
        if not self.account.person:
            return False

        if self.form_values.get('public_name'):
            return self.account.person.display_name.public_name == self.form_values.get('public_name')

        if self.form_values.get('full_name') and self.account.person.is_display_name_empty:
            full_names = list()
            if self.account.person.firstname and self.account.person.lastname:
                full_names.append(u'{} {}'.format(self.account.person.firstname, self.account.person.lastname))
                full_names.append(u'{} {}'.format(self.account.person.lastname, self.account.person.firstname))
            elif self.account.person.firstname:
                full_names.append(self.account.person.firstname)
            elif self.account.person.lastname:
                full_names.append(self.account.person.lastname)

            if self.form_values.get('full_name') not in full_names:
                return False
            if self.account.person.display_name.public_name in full_names:
                return True
            try:
                auto_verdict = self.clean_web_api.check_user_data(
                    key='public_name_cl_check_{}'.format(self.form_values['uid']),
                    auto_only=False,
                    uid=self.form_values['uid'],
                    verdict_data=json.dumps(dict(
                        uid=self.form_values['uid'],
                        value=self.account.person.display_name.public_name,
                    )),
                    display_name=self.account.person.display_name.public_name,
                    simple_response=True,
                )
                if not (auto_verdict is True or auto_verdict is None):
                    return True
            except BaseCleanWebError:
                pass

        return False

    def send_email_notifications(self, language, old_display_name, new_display_name):
        translations = settings.translations.NOTIFICATIONS[language]
        template_name = 'mail/reset_display_name.html'
        info = MailInfo(
            subject=translations['change.displayname.notify.subject'],
            from_=translations['email_sender_display_name'],
            tld=get_tld_by_country(self.account.person.country),
        )

        context = make_email_context(
            language=language,
            account=None,
            context=dict(
                feedback_key='feedback',
                greeting_key='greeting.noname',
                last_displayname=old_display_name,
                new_username=new_display_name,
            )
        )

        send_mail_for_account(template_name, info, context, self.account, send_to_native=True)

    def _process(self):
        events = {
            'action': 'reset_display_name',
            'consumer': self.consumer,
        }
        language = get_preferred_language(self.account)
        old_display_name = jinja2.escape(self.account.person.display_name.public_name)
        new_display_name = self.get_new_display_name(language)
        with UPDATE(self.account, self.request.env, events):
            self.account.person.display_name = DisplayName(name=new_display_name)

        self.send_email_notifications(language, old_display_name, new_display_name)


class ResetQuestionView(BaseResetView):
    required_grants = [
        'account.reset_question',
    ]
    basic_form = AccountResetQuestionForm
    action = 'question_reset'

    def is_applicable(self):
        return self.account.hint.is_set

    def get_account(self):
        self.get_account_by_uid(
            uid=self.form_values['uid'],
            enabled_required=False,
        )

    def _process(self):
        events = {
            'action': 'reset_question',
            'consumer': self.consumer,
        }
        with UPDATE(self.account, self.request.env, events):
            self.account.hint = None


class ResetEmailView(BaseResetView):
    required_grants = [
        'account.reset_email',
    ]
    basic_form = AccountResetEmailForm
    action = 'email_reset'

    def is_applicable(self):
        if self.account.is_lite and Email.normalize_address(self.form_values['email']) == self.account.lite_alias.alias:
            return False

        if Email.normalize_address(self.form_values['email']) not in [email.normalized_address for email in self.account.emails.external]:
            return False
        return True

    def get_account(self):
        self.get_account_by_uid(
            uid=self.form_values['uid'],
            enabled_required=False,
            emails=True,
            email_attributes='all',
        )

    def _process(self):
        events = {
            'action': 'reset_email',
            'consumer': self.consumer,
        }
        with UPDATE(self.account, self.request.env, events):
            del self.account.emails[self.form_values['email']]


class ResetPhoneView(BaseResetView):
    required_grants = [
        'account.reset_phone',
    ]
    basic_form = AccountResetPhoneForm
    action = 'phone_reset'

    def is_applicable(self):
        if self.account.is_phonish or self.account.is_neophonish:
            return False
        if not self.account.phones.has_number(self.form_values['phone_number']):
            return False

        return True

    def get_account(self):
        self.get_account_by_uid(
            uid=self.form_values['uid'],
            enabled_required=False,
            need_phones=True,
        )

    def _process(self):
        events = {
            'action': 'reset_phone',
            'consumer': self.consumer,
        }
        phone = self.account.phones.by_number(self.form_values['phone_number'])
        with UPDATE(self.account, self.request.env, events):
            self.account.phones.remove(phone.id)
