# -*- coding: utf-8 -*-
from base64 import b64encode
import logging

from passport.backend.api.views.bundle.exceptions import (
    AccountWithoutPasswordError,
    PasswordRequiredError,
    RateLimitExceedError,
)
from passport.backend.api.views.bundle.headers import HEADER_CONSUMER_CLIENT_IP
from passport.backend.api.views.bundle.mixins import BundleAccountPropertiesMixin
from passport.backend.core.counters import passman_recovery_key_counter
from passport.backend.core.models.passman_recovery_key import PassManRecoveryKey
from passport.backend.core.runner.context_managers import UPDATE

from .base import SecretsBaseView
from .exceptions import PassmanRecoveryKeyExistsError
from .forms import (
    SecretsReadPassmanRecoveryKeyForm,
    SecretsWritePassmanRecoveryKeyForm,
)


log = logging.getLogger('passport.api.view.bundle.account.secrets')

PASSMAN_RECOVERY_KEY_GRANT = 'account.passman_recovery_key'
PASSMAN_RECOVERY_KEY_READ_SCOPE = 'browser:passman_recovery_key.read'
PASSMAN_RECOVERY_KEY_WRITE_SCOPE = 'browser:passman_recovery_key.write'

PASSWORD_VERIFICATION_AGE = 5 * 60


class SecretsPassmanRecoveryKeyBaseView(SecretsBaseView):
    mode = 'account_secrets_passman_recovery_key'
    required_grants = [PASSMAN_RECOVERY_KEY_GRANT]


class SecretsReadPassmanRecoveryKeyView(SecretsPassmanRecoveryKeyBaseView, BundleAccountPropertiesMixin):
    oauth_scope = PASSMAN_RECOVERY_KEY_READ_SCOPE

    basic_form = SecretsReadPassmanRecoveryKeyForm
    required_headers = [HEADER_CONSUMER_CLIENT_IP]

    def process_request(self):
        self.process_basic_form()
        self.get_account_from_session_or_oauth_token(
            multisession_uid=self.form_values.get('uid'),
            required_scope=self.oauth_scope,
        )
        self.statbox.bind(uid=self.account.uid)

        if not self.account.have_password:
            raise AccountWithoutPasswordError()

        if (
            self.session_info is not None and
            self.is_password_verification_required(max_age_threshold=PASSWORD_VERIFICATION_AGE)
        ):
            # Проверяем давность ввода пароля, только если нам принесли сессию. Если на входе токен - мы и так знаем,
            # что пароль вводился (токен можно получить только по паролю, и живёт он 5 минут)
            raise PasswordRequiredError()

        recovery_key = self.blackbox.get_recovery_keys(
            uid=self.account.uid,
            key_id_b64=b64encode(self.form_values['key_id']),
        )
        # не используем b64encode - от ЧЯ уже пришёл base64
        self.response_values['recovery_key'] = recovery_key
        self.statbox.log(action='read')


class SecretsWritePassmanRecoveryKeyView(SecretsPassmanRecoveryKeyBaseView):
    oauth_scope = PASSMAN_RECOVERY_KEY_WRITE_SCOPE

    basic_form = SecretsWritePassmanRecoveryKeyForm

    def process_request(self):
        self.process_basic_form()
        self.get_account_by_oauth_token(
            required_scope=self.oauth_scope,
        )
        self.statbox.bind(uid=self.account.uid)

        if not self.account.have_password:
            raise AccountWithoutPasswordError()

        counter = passman_recovery_key_counter.get_counter()
        if counter.hit_limit(self.account.uid):
            raise RateLimitExceedError()

        existing_key = self.blackbox.get_recovery_keys(
            uid=self.account.uid,
            key_id_b64=b64encode(self.form_values['key_id']),
        )
        if existing_key:
            raise PassmanRecoveryKeyExistsError()

        counter.incr(self.account.uid)

        events = {
            'action': 'passman_recovery_key',
            'consumer': self.consumer,
        }
        with UPDATE(
            self.account,
            self.request.env,
            events,
        ):
            self.account.passman_recovery_key = PassManRecoveryKey(
                parent=self.account,
                key_id=self.form_values['key_id'],
                recovery_key=self.form_values['recovery_key'],
            )

        self.response_values.update(
            uid=str(self.account.uid),  # некоторые реализации JSON-парсеров не умеют int64
            login=self.account.login,
        )
        self.statbox.log(action='saved')
