# -*- coding: utf-8 -*-

from datetime import datetime

from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import AccountInvalidTypeError
from passport.backend.api.views.bundle.mixins import (
    BundleAccountGetterMixin,
    BundlePasswordChangeMixin,
    BundlePasswordValidationMixin,
)
from passport.backend.core.logging_utils.loggers.statbox import StatboxLogger
from passport.backend.core.models.password import PASSWORD_CHANGING_REASON_FLUSHED_BY_ADMIN
from passport.backend.core.runner.context_managers import UPDATE
from passport.backend.core.utils.decorators import cached_property

from .forms import AccountSetPasswordForm


ACCOUNT_SET_PASSWORD_BASE_GRANT = 'account.set_password'
ACCOUNT_SET_ANY_PASSWORD_GRANT = 'account.set_password_any'
ACCOUNT_SET_PDD_PASSWORD_GRANT = 'account.set_password_pdd'
ACCOUNT_SET_KINOPOISK_PASSWORD_GRANT = 'account.set_password_kinopoisk'
ACCOUNT_SET_PASSWORD_HASH_GRANT = 'account.set_password_hash'


class AccountSetPasswordView(BaseBundleView,
                             BundleAccountGetterMixin,
                             BundlePasswordChangeMixin,
                             BundlePasswordValidationMixin):

    basic_form = AccountSetPasswordForm
    required_grants = [ACCOUNT_SET_PASSWORD_BASE_GRANT]
    grants_for_account_type = {
        'any': ACCOUNT_SET_ANY_PASSWORD_GRANT,
        'pdd': ACCOUNT_SET_PDD_PASSWORD_GRANT,
        'kinopoisk': ACCOUNT_SET_KINOPOISK_PASSWORD_GRANT,
    }

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode='account_set_password',
            consumer=self.consumer,
            ip=self.client_ip,
            user_agent=self.user_agent,
        )

    def process_request(self):
        self.process_basic_form()
        self.create_track('authorize')
        self.statbox.log(action='submitted')

        self.get_account_by_uid(uid=self.form_values['uid'])
        self.statbox.bind_context(uid=self.account.uid)

        if not (self.account.is_pdd or self.account.is_kinopoisk):
            raise AccountInvalidTypeError()

        self.check_grants_for_account_type()

        hash_version, raw_password_hash = None, None
        if self.form_values['password_hash']:
            hash_version, raw_password_hash = self.form_values['password_hash']

        password = self.form_values['password']
        with UPDATE(self.account, self.request.env, {'action': 'set_password'}):
            if password:
                password, password_quality = self.validate_password(
                    login=self.account.login,
                    emails={self.account.login} if self.account.is_pdd else {},
                )
                self.change_password(password, quality=password_quality)

            elif raw_password_hash:
                self.check_grant(ACCOUNT_SET_PASSWORD_HASH_GRANT)

                self.account.password.set_hash(
                    raw_password_hash,
                    version=hash_version,
                )
                self.account.global_logout_datetime = datetime.now()

            if self.form_values['force_password_change']:
                self.account.password.setup_password_changing_requirement(
                    is_required=True,
                    changing_reason=PASSWORD_CHANGING_REASON_FLUSHED_BY_ADMIN,
                )

            self.statbox.log(
                action='set_password',
                is_hash=bool(raw_password_hash),
            )

            if self.account.totp_secret.is_set:
                # Для ПДД-шника необходимо безусловно сбрасывать установленный
                # 2FA при смене пароля, т.к. она производится администратором.
                self.account.totp_secret = None
                self.statbox.log(action='2fa_disabled')
