# -*- coding: utf-8 -*-
import logging

from passport.backend.api.common.mail import is_mail_occupied_by_another_user
from passport.backend.api.views.bundle.auth.base import BundleBaseAuthorizationMixin
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import (
    AccountWithoutLoginError,
    AccountWithoutPasswordError,
    SubscriptionBlockedError,
    SubscriptionForbiddenError,
)
from passport.backend.api.views.bundle.headers import (
    HEADER_CLIENT_COOKIE,
    HEADER_CLIENT_HOST,
    HEADER_CLIENT_USER_AGENT,
    HEADER_CONSUMER_CLIENT_IP,
)
from passport.backend.api.views.bundle.mixins import (
    BundleAccountGetterMixin,
    BundleAccountPropertiesMixin,
    BundleAccountResponseRendererMixin,
    BundleAccountSubscribeMixin,
    BundleFixPDDRetpathMixin,
    BundlePasswordChangeMixin,
)
from passport.backend.api.views.bundle.utils import assert_valid_host
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.loggers import StatboxLogger
from passport.backend.core.subscription import (
    can_be_subscribed,
    is_subscription_blocked_by_zero,
    SubscriptionImpossibleError,
    SubscriptionNotAllowedError,
    SubscriptionRequiresAccountWithLoginError,
    SubscriptionRequiresAccountWithPasswordError,
)
from passport.backend.core.utils.decorators import cached_property
from passport.backend.core.utils.domains import get_keyspace_by_host

from .forms import SubscribeForm


log = logging.getLogger('passport.api.view.bundle.account.subscriptions')


class SubscribeView(BaseBundleView,
                    BundleAccountGetterMixin,
                    BundleAccountPropertiesMixin,
                    BundleAccountResponseRendererMixin,
                    BundleFixPDDRetpathMixin,
                    BundlePasswordChangeMixin,
                    BundleBaseAuthorizationMixin,
                    BundleAccountSubscribeMixin):
    """
    Подписка аккаунта на сервис по кукам
    """

    required_headers = (
        HEADER_CLIENT_HOST,
        HEADER_CONSUMER_CLIENT_IP,
        HEADER_CLIENT_COOKIE,
        HEADER_CLIENT_USER_AGENT,
    )

    required_grants = ['account.subscribe']

    require_track = False

    basic_form = SubscribeForm

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode='subscribe',
            consumer=self.consumer,
            ip=self.client_ip,
            user_agent=self.user_agent,
        )

    @property
    def default_retpath(self):
        return 'https://%s.%s' % (settings.PASSPORT_SUBDOMAIN, get_keyspace_by_host(self.host))

    def fill_response_with_account(self, **kwargs):
        super(SubscribeView, self).fill_response_with_account(**kwargs)
        if self.account.is_pdd:
            self.response_values['account']['domain_id'] = self.account.domain.id

    def assert_can_be_subscribed(self, service):
        # Надо понять, подписан ли пользователь уже на сервис
        is_subscribed_already = self.account.is_subscribed(service)

        # Это не обычная подписка, проверять больше нечего.
        if is_subscribed_already and not self.account.has_sid(service.sid):
            return

        # Если пользователь уже подписан на сервис, но пользоваться сервисом ему
        # запрещено — выводится сообщение «Доступ к сервису заблокирован».
        # Пруфлинк: https://doc.yandex-team.ru/Passport/passport-modes/reference/subscribe.xml
        if is_subscribed_already:
            subscription = self.account.subscriptions[service.sid]
            if is_subscription_blocked_by_zero(subscription):
                message = 'Account is already subscribed to blocked service ' \
                          'with login_rule %s' % subscription.login_rule
                log.error(message)
                raise SubscriptionBlockedError(message)
            else:
                return

        if service.slug == 'mail':
            if is_mail_occupied_by_another_user(self.blackbox, self.account):
                raise SubscriptionBlockedError('Another user already occupied this email')

        # Теперь остается подписать пользователя,
        # если он удовлетворяет всем условиям данной подписки
        try:
            can_be_subscribed(self.account, service)
        except (SubscriptionImpossibleError, SubscriptionNotAllowedError):
            log.error('Subscription to service %s is forbidden', service.slug)
            raise SubscriptionForbiddenError()
        except SubscriptionRequiresAccountWithLoginError:
            log.debug('Subscription to service %s requires login', service.slug)
            raise AccountWithoutLoginError()
        except SubscriptionRequiresAccountWithPasswordError:
            log.debug('Subscription to service %s requires password', service.slug)
            raise AccountWithoutPasswordError()

        # Ну и неизменный костыль для galatasaray
        if service.sid == 61 and not self.account.is_normal:
            log.error('Subscription to service %s is forbidden for uid %s', service.slug, self.account.uid)
            raise SubscriptionForbiddenError()

    def process_request(self):
        assert_valid_host(self.request.env)
        self.process_basic_form()

        service = self.form_values['service']
        if service is None:
            log.debug('in subscribe: subscription for service is not implemented')
            self.response_values['retpath'] = self.default_retpath
            return

        self.get_account_from_session(
            check_disabled_on_deletion=True,
        )
        self.statbox.bind_context(
            uid=self.account.uid,
            sid=service.sid,
        )
        self.statbox.log(action='submitted')

        if self.account.is_pdd:
            self.fix_pdd_retpath()

        retpath = self.form_values['retpath'] or self.default_retpath

        self.fill_response_with_account(account_info_required=True)

        self.response_values.update(
            service=service.slug,
            retpath=retpath,
        )

        self.assert_can_be_subscribed(service)

        self.subscribe_if_allow_and_update_account(
            self.account,
            service,
        )
