# -*- coding: utf-8 -*-

import logging

from passport.backend.api.common.authorization import (
    set_authorization_track_fields,
    user_session_scope,
)
from passport.backend.api.views.bundle.auth.base import BundleBaseAuthorizationMixin
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import (
    AccountNotSubscribedError,
    CaptchaRequiredError,
    InvalidTrackStateError,
    SubscriptionBlockedError,
    UserNotVerifiedError,
    ValidationFailedError,
)
from passport.backend.api.views.bundle.headers import (
    HEADER_CLIENT_COOKIE,
    HEADER_CLIENT_HOST,
    HEADER_CLIENT_USER_AGENT,
    HEADER_CONSUMER_CLIENT_IP,
)
from passport.backend.api.views.bundle.mixins import (
    BundleAccountGetterMixin,
    BundleAccountPropertiesMixin,
    BundleAccountResponseRendererMixin,
    BundleAssertCaptchaMixin,
    BundleFixPDDRetpathMixin,
    BundlePasswordChangeMixin,
    BundlePhoneMixin,
    BundleVerifyPasswordMixin,
)
from passport.backend.api.views.bundle.phone import helpers as phone_helpers
from passport.backend.api.views.bundle.utils import assert_valid_host
from passport.backend.core.logging_utils.loggers import StatboxLogger
from passport.backend.core.runner.context_managers import UPDATE
from passport.backend.core.services import get_service
from passport.backend.core.subscription import (
    can_be_unsubscribed,
    delete_subscription,
    is_subscription_blocked,
    is_subscription_blocked_by_zero,
    UnsubscribeBlockingServiceError,
    UnsubscribeProtectedServiceError,
)
from passport.backend.core.utils.decorators import cached_property

from .forms import (
    UnsubscribeCommitForm,
    UnsubscribeSubmitForm,
)


log = logging.getLogger('passport.api.view.bundle.account.subscriptions')


class BaseUnsubscribeView(BaseBundleView,
                          BundleAccountGetterMixin,
                          BundleAccountPropertiesMixin,
                          BundleAccountResponseRendererMixin,
                          BundleAssertCaptchaMixin,
                          BundleVerifyPasswordMixin,
                          BundlePasswordChangeMixin,
                          BundleBaseAuthorizationMixin,
                          BundlePhoneMixin):

    """Отписка аккаунта от сервиса"""

    required_headers = (
        HEADER_CLIENT_HOST,
        HEADER_CLIENT_COOKIE,
        HEADER_CONSUMER_CLIENT_IP,
        HEADER_CLIENT_USER_AGENT,
    )

    required_grants = ['account.unsubscribe']

    require_track = False
    track_type = 'authorize'

    basic_form = UnsubscribeSubmitForm

    @cached_property
    def statbox(self):
        return StatboxLogger(
            ip=self.client_ip,
            user_agent=self.user_agent,
            track_id=self.track_id,
            mode='unsubscribe',
        )

    @staticmethod
    def assert_account_can_be_unsubscribed(account, service):
        """
        Проверим, возможно ли отписать аккаунт
        - Нельзя НИКОГО отписать от Я.СМС. Притворяемся что его вообще не существует
        - Lite аккаунтам позволяем отписаться от чего угодно

        - Нельзя отписать от Блокирующих сервисов
          см. https://wiki.yandex-team.ru/passport/sids/
        - Нельзя отписать от некоторых Защищенных подписок
        - ПДД пользователь не может отписаться от Почты

        - Нельзя отписать некоторые сервисы, если login_rule меньше некоторого значения

        Тезисы после разбора этой ручки в perl-коде Паспорта
        https://wiki.yandex-team.ru/passport/backend/apiold/#unsubscribe
        """
        if service.slug == 'yasms':
            # Притворяемся, что такого сервиса вообще не существует
            raise ValidationFailedError(['service.invalid'])

        if account.is_lite:
            return

        try:
            can_be_unsubscribed(account, service)
        except UnsubscribeBlockingServiceError:
            log.debug('Can\'t unsubscribe blocking service')
            raise SubscriptionBlockedError()
        except UnsubscribeProtectedServiceError:
            log.debug('Can\'t unsubscribe protected service')
            raise SubscriptionBlockedError()

        # Это не обычная подписка, проверять больше нечего.
        if not account.has_sid(service.sid):
            return

        subscription = account.subscriptions[service.sid]
        if is_subscription_blocked(subscription):
            message = 'Can\'t unsubcribe account from service because of login_rule=%s' % subscription.login_rule
            log.error(message)
            raise SubscriptionBlockedError(message)

        if is_subscription_blocked_by_zero(subscription):
            log.debug('Unsubscribe attempt with login_rule=%s' % subscription.login_rule)
            raise SubscriptionBlockedError('Bad login rule value for unsubscribe')

    @staticmethod
    def assert_account_is_subscribed(account, service):
        """Ошибка, если пользователь не подписан на этот сервис"""
        if not account.is_subscribed(service):
            raise AccountNotSubscribedError()

    def check_track(self):
        if self.track.is_unsubscribed:
            raise InvalidTrackStateError('Track was already used for this action')

        if not self.track.uid:
            raise InvalidTrackStateError('Bad uid in track')

        if self.track.is_captcha_required and not self.is_captcha_passed:
            raise CaptchaRequiredError()

    @property
    def is_validated(self):
        """
        Считается что, пользователь подтвердил своё действие:
        - телефоном
        - кв
        - анкетой в точности совпадающей с анкетой на принудительной смене пароля

        Капча может потребоваться при любой ситации.
        """
        return (
            (
                self.is_phone_confirmed_in_track() and
                self.saved_number and self.saved_number == self.secure_number
            ) or
            self.track.is_secure_hint_answer_checked or
            self.track.is_short_form_factors_checked
        ) and (
            self.is_captcha_passed if self.track.is_captcha_required else True
        )

    def check_specific_requirements(self, service):
        """
        При отписке от сервиса Почта, требуем одно из следующих условий:
          - подтверждение через код из sms на защищенный номер И правильный ввод капчи
          - успешный ответ на КВ
          - прохождение короткой анкеты восстановления
        """
        if service.slug == 'mail' and not self.is_validated:
            raise UserNotVerifiedError()

    def unsubscribe(self, service):
        """
        Отписываем
        Удаляем атрибуты и алиасы
        """
        events = {
            'consumer': self.consumer,
            'action': 'unsubscribe',
        }
        with UPDATE(self.account, self.request.env, events):
            delete_subscription(self.account, service)

    def process_request(self):
        assert_valid_host(self.request.env)
        self.process_basic_form()
        self.process()

    def process(self):
        raise NotImplementedError()  # pragma: no cover


class UnsubscribeSubmitView(BaseUnsubscribeView,
                            BundleFixPDDRetpathMixin):

    def fill_track_with_account(self):
        self.track.uid = self.account.uid
        if self.account.is_pdd:
            self.track.domain = self.account.domain.domain
        if self.account.emails:
            self.track.emails = ' '.join([email.address for email in self.account.emails.native])

    def process(self):
        service = self.form_values['service']
        self.statbox.bind_context(sid=service.sid)

        self.get_account_from_session(
            check_disabled_on_deletion=True,
            need_phones=True,
            force_show_mail_subscription=True,
        )
        self.statbox.bind_context(uid=self.account.uid)

        self.fill_response_with_account()

        self.assert_account_is_subscribed(self.account, service)
        self.assert_account_can_be_unsubscribed(self.account, service)

        # Проверяем тип пользователя и его возможности
        redirect_state = self.check_have_password()
        if redirect_state:
            self.state = redirect_state
            return

        self.create_track(self.track_type)
        self.response_values['track_id'] = self.track_id

        with self.track_transaction.commit_on_error():
            self.fill_track_with_account()

            self.track.service = service.slug

            if self.form_values['retpath']:
                self.track.retpath = self.form_values['retpath']

            if self.account.is_pdd:
                self.fix_pdd_retpath()

            self.save_secure_number_in_track()
            if self.secure_number:
                self.track.has_secure_phone_number = True
                self.response_values['number'] = phone_helpers.dump_number(self.secure_number)
            self.statbox.log(
                action='submitted',
            )


class UnsubscribeCommitView(BaseUnsubscribeView):

    require_track = True
    basic_form = UnsubscribeCommitForm

    def fill_track(self):
        self.track.is_unsubscribed = True
        set_authorization_track_fields(
            self.account,
            self.track,
            allow_create_session=True,
            allow_create_token=False,
            old_session_info=self.session_info,
            password_passed=True,
            session_scope=user_session_scope(self.session_info, self.account.uid),
        )

    def process(self):
        self.read_track()
        service = get_service(slug=self.track.service)
        self.statbox.bind_context(sid=service.sid)

        self.check_track()
        self.response_values['track_id'] = self.track_id
        self.response_values['retpath'] = self.track.retpath

        self.get_account_from_session(
            check_disabled_on_deletion=True,
            emails=True,
            multisession_uid=self.track_uid,
            need_phones=True,
            force_show_mail_subscription=True,
        )

        self.statbox.bind_context(uid=self.account.uid)

        self.fill_response_with_account()

        self.assert_account_is_subscribed(self.account, service)
        self.assert_account_can_be_unsubscribed(self.account, service)

        # Проверяем тип пользователя и его возможности
        redirect_state = self.check_have_password()
        if redirect_state:
            self.state = redirect_state
            return

        self.check_specific_requirements(service)

        with self.track_transaction.commit_on_error():
            # Здесь произойдет повторный парсинг модели Account
            self.verify_password(
                uid=self.account.uid,
                password=self.form_values['password'],
            )

            self.unsubscribe(service)
            self.fill_track()
            self.fill_response_with_account_and_session(
                personal_data_required=True,
                cookie_session_info=self.session_info,
            )

            self.statbox.log(
                action='removed',
            )
