# -*- coding: utf-8 -*-
import logging

from passport.backend.api.common.account import fill_person_from_args
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import (
    DisplayNameUpdateNotAllowed,
    PublicIdUpdateNotAllowed,
    ValidationFailedError,
)
from passport.backend.api.views.bundle.mixins import (
    BundleAccountGetterMixin,
    BundleAccountResponseRendererMixin,
    BundleAccountPropertiesMixin,
)
from passport.backend.api.views.bundle.mixins.common import BundleCleanWebMixin
from passport.backend.core import validators
from passport.backend.core.geobase import Region
from passport.backend.core.logging_utils.loggers.statbox import StatboxLogger
from passport.backend.core.runner.context_managers import UPDATE
from passport.backend.core.types.login.login import normalize_login
from passport.backend.core.utils.decorators import cached_property
from passport.backend.utils.string import smart_text

from .forms import (
    AccountChangeLanguageForm,
    AccountPersonalInfoForm,
    ScholarPersonalInfoForm,
)


log = logging.getLogger('passport.api.view.bundle.account.personal_info')


ACCOUNT_PERSON_BY_UID_GRANT = 'account_person.by_uid'
ACCOUNT_PERSON_PDD_BY_UID_GRANT = 'account_person.pdd_by_uid'
ACCOUNT_PERSON_SCHOLAR_BY_UID_GRANT = 'account_person.scholar_by_uid'
ACCOUNT_PERSON_WRITE_GRANT = 'account_person.write'

ACCOUNT_PERSON_CHANGE_LANGUAGE_GRANT = 'account_person.change_language'


class UpdateAccountPersonalInfoView(BaseBundleView,
                                    BundleAccountGetterMixin,
                                    BundleAccountResponseRendererMixin,
                                    BundleAccountPropertiesMixin,
                                    BundleCleanWebMixin):
    """
    Ручка предполагает два варианта использования: длинный и короткий.
    Короткий: с грантом и по UID из формы (пока только для ПДД).
    Длинный: по токену или сессионной куке через предварительное получение трека.
    """

    required_grants = [ACCOUNT_PERSON_WRITE_GRANT]
    basic_form = AccountPersonalInfoForm
    grants_for_account_type = {
        'any': ACCOUNT_PERSON_BY_UID_GRANT,
        'pdd': ACCOUNT_PERSON_PDD_BY_UID_GRANT,
    }

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode='account_personal_info',
            user_agent=self.user_agent,
            consumer=self.consumer,
            ip=self.client_ip,
        )

    def process_request(self, *args, **kwargs):
        self.process_basic_form()
        self.clean_web_check_form_values()

        uid = self.form_values['uid']

        if uid is not None:
            self.get_account_by_uid(uid)
            self.check_grants_for_account_type()
        else:
            self.read_track()
            self.get_pinned_account_from_session_or_oauth_token()

        self._check_child_update_restrictions()
        self.statbox.log(action='submitted')
        events = {
            'action': 'person',
            'consumer': self.consumer,
        }
        # Приоритет у введенного руками города выше: если пользователь
        # не поленился что-то написать, наверно, это важно.
        city_id = self.form_values.pop('city_id')
        if city_id and not self.form_values['city']:
            self.form_values['city'] = smart_text(Region(id=city_id).name)

        if self.form_values['display_name']:
            if self.account.is_verified:
                if self.consumer == 'mobileproxy':
                    raise ValidationFailedError(
                        ['display_name.invalid'],
                    )
                else:
                    raise DisplayNameUpdateNotAllowed()

        # Если из формы пришёл display_name, а на модели он дефолтный, то сбросим значение на модели,
        # чтобы запись в БД произошла (иначе диффер может сравнить значение из формы с дефолтом из ЧЯ и решить,
        # что ничего не изменилось): https://st.yandex-team.ru/PASSP-20807#1534414618000
        # или если установлен флаг dont_use_displayname_as_public_name и аккаунт социальный: https://st.yandex-team.ru/PASSP-24975
        if self.form_values['display_name'] and \
                (self.account.person.is_display_name_empty or (self.account.person.dont_use_displayname_as_public_name and self.account.is_social)):
            self.account.person.display_name.name = None

        with UPDATE(self.account, self.request.env, events):
            self.account.person = fill_person_from_args(self.account.person, self.form_values)

            contact_phone_number = self.form_values['contact_phone_number']
            if contact_phone_number:
                contact_phone_number = contact_phone_number.e164
                self.account.person.contact_phone_number = contact_phone_number

            public_id = self.form_values['public_id']
            if public_id and public_id != self.account.user_defined_public_id:
                normalized_public_id = normalize_login(public_id)

                if self.account.public_id_alias.alias != normalized_public_id:
                    if self.account.public_id_alias.alias and len(self.account.public_id_alias.old_public_ids):
                        raise PublicIdUpdateNotAllowed()
                    try:
                        availability_validator = validators.PublicIdAvailability(uid=self.account.uid)
                        availability_validator.to_python(self.form_values, validators.State(self.request.env))
                    except validators.Invalid as exc:
                        raise ValidationFailedError(
                            ['public_id.not_available'],
                            invalid_exception=exc,
                        )
                    if self.account.public_id_alias.alias and not self.account.public_id_alias.has_old_public_id(self.account.public_id_alias.alias):
                        self.account.public_id_alias.add_old_public_id(self.account.public_id_alias.alias)
                    self.account.public_id_alias.alias = normalized_public_id

                self.account.user_defined_public_id = public_id

            # Глобальные имя и фамилия в форме есть только у портальных пользователей yandex-team
            if self.form_values['firstname_global']:
                self.account.person.firstname_global = self.form_values['firstname_global']
            if self.form_values['lastname_global']:
                self.account.person.lastname_global = self.form_values['lastname_global']


class ChangeLanguageView(BaseBundleView,
                         BundleAccountGetterMixin,
                         BundleAccountResponseRendererMixin):
    required_grants = [ACCOUNT_PERSON_CHANGE_LANGUAGE_GRANT]
    basic_form = AccountChangeLanguageForm

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode='account_change_language',
            user_agent=self.user_agent,
            consumer=self.consumer,
            ip=self.client_ip,
        )

    def process_request(self, *args, **kwargs):
        self.process_basic_form()

        self.get_account_from_session_or_oauth_token(multisession_uid=self.form_values['uid'])
        self.statbox.log(action='submitted')
        events = {
            'action': 'change_language',
            'consumer': self.consumer,
        }

        with UPDATE(self.account, self.request.env, events):
            self.account.person.language = self.form_values['language']


class UpdateScholarPersonalInfoView(
    BaseBundleView,
    BundleAccountGetterMixin,
    BundleCleanWebMixin,
):
    required_grants = [ACCOUNT_PERSON_SCHOLAR_BY_UID_GRANT]
    basic_form = ScholarPersonalInfoForm

    @cached_property
    def statbox(self):
        return StatboxLogger(
            consumer=self.consumer,
            ip=self.client_ip,
            mode='scholar_personal_info',
            user_agent=self.user_agent,
        )

    def process_request(self):
        self.process_basic_form()
        self.clean_web_check_form_values()
        self.get_account_by_uid(self.form_values.get('uid'))

        self.statbox.log(action='submitted')

        with UPDATE(
            self.account,
            self.request.env,
            dict(action='person', consumer=self.consumer),
        ):
            self.account.person = fill_person_from_args(self.account.person, self.form_values)

    def is_account_type_allowed(self):
        return self.account.scholar_alias
