# -*- coding: utf-8 -*-
import logging

from passport.backend.api.common.common import extract_tld
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import (
    ActionNotRequiredError,
    CaptchaRequiredError,
)
from passport.backend.api.views.bundle.headers import (
    HEADER_CLIENT_COOKIE,
    HEADER_CLIENT_HOST,
    HEADER_CLIENT_USER_AGENT,
    HEADER_CONSUMER_CLIENT_IP,
)
from passport.backend.api.views.bundle.mixins import (
    BundleAccountGetterMixin,
    BundleAssertCaptchaMixin,
    BundlePhoneMixin,
)
from passport.backend.api.yasms.utils import does_binding_allow_washing
from passport.backend.core.conf import settings
from passport.backend.core.mailer import PlainTextMessage
from passport.backend.core.mailer.utils import (
    render_message_text,
    send_message,
)
from passport.backend.core.models.account import get_preferred_language
from passport.backend.core.runner.context_managers import UPDATE
from passport.backend.core.utils.decorators import cached_property

from .forms import (
    UserApproveCommitForm,
    UserApproveSubmitForm,
)


log = logging.getLogger('passport.api.view.bundle.account.userapprove')

ACCOUNT_USERAPPROVE_GRANT = 'account.userapprove'

DEFAULT_RETPATH = 'https://www.yandex.%s'

KARMA_WASHED_STATE = 'karma_washed'
EMAIL_SENT_STATE = 'email_sent'

USER_APPROVE_TEMPLATE_NAME = 'mail/user_approve_message.txt'


class UserApproveBaseView(
    BaseBundleView,
    BundleAccountGetterMixin,
    BundleAssertCaptchaMixin,
    BundlePhoneMixin,
):
    required_headers = [
        HEADER_CLIENT_HOST,
        HEADER_CONSUMER_CLIENT_IP,
        HEADER_CLIENT_USER_AGENT,
        HEADER_CLIENT_COOKIE,
    ]
    required_grants = [ACCOUNT_USERAPPROVE_GRANT]

    @cached_property
    def tld(self):
        return extract_tld(self.request.env.host, settings.PASSPORT_TLDS) or settings.PASSPORT_DEFAULT_TLD

    def invalidate_track_and_require_captcha(self):
        with self.track_transaction.commit_on_error():
            self.track.is_captcha_required = True
            self.invalidate_captcha()

    def process_request(self):
        self.process_basic_form()
        self.response_values['retpath'] = self.form_values['retpath'] or DEFAULT_RETPATH % self.tld

        self.get_account_from_session(
            check_disabled_on_deletion=True,
            need_phones=True,
            emails=True,
        )

        # Если карма уже обелена или она идеальна, нам нечего делать
        if self.account.karma.is_washed() or self.account.karma.value == 0:
            raise ActionNotRequiredError()

        self.process()

    def process(self):
        raise NotImplementedError()     # pragma: no cover


class UserApproveSubmitView(UserApproveBaseView):
    """
    Первый шаг самообеления пользователя:
    если у пользователя есть чистый привязанный телефон, обеляем его.
    """
    basic_form = UserApproveSubmitForm

    def process(self):
        self.read_or_create_track('authorize')

        if self.account.phones.bound():
            bound_phones = self.account.phones.bound().values()
            bound_phone_numbers = [p.number.e164 for p in bound_phones]
            bindings_history = self.yasms_api.phone_bindings_history(bound_phone_numbers)
            for phone in bound_phones:
                if does_binding_allow_washing(
                        self.account,
                        phone.number.e164,
                        phone.bound,
                        phone.binding.should_ignore_binding_limit,
                        bindings_history,
                ):
                    events = {
                        'action': 'karma',
                        'consumer': self.consumer,
                    }
                    old_karma_value = self.account.karma.value
                    with UPDATE(self.account, self.request.env, events):
                        self.account.karma.prefix = settings.KARMA_PREFIX_WASHED
                    log.info(
                        'In userapprove: karma washed by bound phone (uid=%s; karma=%s)',
                        self.account.uid,
                        old_karma_value,
                    )
                    self.response_values['state'] = KARMA_WASHED_STATE
                    return

        # Затребуем капчу для следующего шага
        self.invalidate_track_and_require_captcha()
        self.response_values['track_id'] = self.track_id


class UserApproveCommitView(UserApproveBaseView):
    """
    Второй шаг самообеления пользователя:
    телефона не нашлось, отправим письмо пользователя в службу поддержки.
    """
    basic_form = UserApproveCommitForm
    require_track = True

    def create_user_approve_message(self):
        language = get_preferred_language(self.account)

        # Для поля From перл всегда создает адрес вида <login>@yandex.<tld>
        if self.account.emails.default:
            account_email = self.account.emails.default.address
        elif self.account.is_pdd:
            account_email = self.account.login
        else:
            account_email = '%s@yandex.%s' % (self.account.login, self.tld)

        context = dict(
            text=self.form_values['text'],
            uid=self.account.uid,
            login=self.account.login,
            email=account_email,
            karma=self.account.karma.value,
            user_ip=self.client_ip.normalized,
            adm_url=settings.RECOVERY_ADM_USERS_URL,
            referer=self.referer or None,
        )
        body = render_message_text(USER_APPROVE_TEMPLATE_NAME, context)
        recipient = settings.RECOVERY_EMAIL_BY_TLD.get(self.tld, settings.RECOVERY_EMAIL_BY_TLD['default'])

        return PlainTextMessage(
            subject=settings.translations.NOTIFICATIONS[language]['userapprove.subject'],
            recipients=[recipient],
            from_=(self.account.login, account_email),
            sender=account_email,
            reply_to=account_email,
            body=body,
            extra_headers={
                'X-Browser': self.user_agent,
                'X-Address': self.client_ip.normalized,
            },
        )

    def process(self):
        self.read_track()

        # Капча нужна всегда
        if not self.is_captcha_passed:
            raise CaptchaRequiredError()

        message = self.create_user_approve_message()
        send_message(message)

        self.response_values['state'] = EMAIL_SENT_STATE
        # На всякий случай снова требуем капчу, чтобы не завалить саппорт письмами
        self.invalidate_track_and_require_captcha()
