# -*- coding: utf-8 -*-

from datetime import datetime
import logging

from passport.backend.api.common import (
    build_auth_cookies_and_session,
    set_authorization_track_fields,
)
from passport.backend.api.common.authorization import (
    is_session_created,
    set_old_session_track_fields,
    users_from_multisession,
)
from passport.backend.api.exceptions import (
    AccountGlobalLogoutError,
    SessionExpiredError,
)
from passport.backend.api.views.bundle.constants import (
    CHANGE_PASSWORD_REASON_EXPIRED,
    CHANGE_PASSWORD_REASON_FLUSHED,
    CHANGE_PASSWORD_REASON_HACKED,
    CHANGE_PASSWORD_REASON_PWNED,
    CHANGE_PASSWORD_REASON_TOO_WEAK,
)
from passport.backend.api.views.bundle.exceptions import (
    AccountGlobalLogoutError as BundleAccountGlobalLogoutError,
    AccountStrongPasswordPolicyError,
)
from passport.backend.api.views.bundle.mixins import CookieCheckStatus
from passport.backend.api.views.bundle.mixins.password import NO_VALIDATION
from passport.backend.api.views.bundle.phone import helpers as phone_helpers
from passport.backend.api.views.bundle.states import (
    PasswordChangeForbidden,
    RedirectToCompletion,
    RedirectToFederalCompletion,
    RedirectToForcedLiteCompletion,
    RedirectToPasswordChange,
    RedirectToPDDCompletion,
    RedirectToPDDCompletionWithPassword,
)
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.loggers.statbox import to_statbox
from passport.backend.core.models.password import PASSWORD_CHANGING_REASON_PWNED
from passport.backend.core.runner.context_managers import UPDATE
from passport.backend.core.utils.experiments import is_experiment_enabled_by_uid
from passport.backend.core.utils.shakur import (
    check_password_is_pwned,
    is_login_in_shakur_whitelist,
)

from .exceptions import (
    AuthAlreadyPassedError,
    AuthSessionExpiredError,
    AuthSessionOverflowError,
)


log = logging.getLogger('passport.authorization')


class BundleBaseAuthorizationMixin(object):

    # Полный ответ ЧЯ о домене пользователя. Имеет смысл только для ПДД.
    domains = None

    def hosted_domains(self, domain):
        if self.domains is None:
            self.domains = self.blackbox.hosted_domains(domain=domain)
        return self.domains

    def check_auth_not_passed(self):
        """Проверим, не прошел ли пользователь авторизацию и пришел сюда повторно"""
        if is_session_created(self.track):
            raise AuthAlreadyPassedError()

    def prepare_track_for_second_step(self, allowed_second_steps):
        self.track.is_second_step_required = True
        self.track.allowed_second_steps = allowed_second_steps
        if (
            self.account and
            self.account.password and
            self.account.password.is_expired
        ):
            # После второго шага знание о необходимости смены пароля станет недоступно (в ятиме оно приходит от AD,
            # поэтому доступно только в ответе метода login ЧЯ). По этой причине сохраним его в треке.
            # При этом expired может означать как протухание по ttl, так и досрочное протухание
            # по инициативе AD (т.е. фактически принуждение к смене).
            self.track.change_password_reason = CHANGE_PASSWORD_REASON_EXPIRED

    def check_redirect_state(self):
        return (
            self.track.is_complete_pdd_with_password or
            self.track.is_complete_pdd or
            self.track.is_complete_autoregistered or
            self.track.is_password_change or
            self.track.is_force_complete_lite
        )

    def experiment_change_password_with_sms(self):
        if self.is_sms_validation_required and self.has_secure_number:
            self.track.has_secure_phone_number = True
            self.track.secure_phone_number = self.secure_number.e164
            self.response_values['number'] = phone_helpers.dump_number(self.secure_number)

    def try_experiments(self):
        """Эта функция по ситуации определяет, нужно ли применить какие-то дополнительные
        действия к пользователю в рамках какого-либо эксперимета"""
        if self.track.is_password_change:
            self.experiment_change_password_with_sms()

    @property
    def is_password_expired(self):
        # Смотрим и на модель, и на ранее сохранённое в треке знание (см. коммент в prepare_track_for_second_step)
        return self.account.password.is_expired or self.track.change_password_reason == CHANGE_PASSWORD_REASON_EXPIRED

    def check_user_policies(self, check_strong_password_policy=False):
        """
        Проверим, нужно ли перенаправить пользователя на страницу смены пароля,
        или на страницу завершения регистрации (анкета)
        """
        if self.account.is_pdd:
            if not self.account.is_complete_pdd:
                if not self.account.is_pdd_workspace_user:
                    self.track.is_captcha_required = True
                if self.account.password.is_new_password_required and self.account.domain.can_users_change_password:
                    self.track.is_complete_pdd_with_password = True
                    return RedirectToPDDCompletionWithPassword(self.account.hint)
                else:
                    self.track.is_complete_pdd = True
                    return RedirectToPDDCompletion(self.account.hint)
            elif not self.account.totp_secret.is_set:
                # Если ПДД пользователь дорегистрирован и у него стоит
                # принудительная смена пароля и запрещена смена пароля на домене,
                # то пускать такого пользователя нельзя (этот флаг могла поставить ФО
                # или наши скрипты - игнорировать нельзя)
                # Если ПДД пользователь дорегистрирован и у него
                # внезапно остался флаг требования завести новый пароль,
                # а смена пароля на домене запрещена, то пускаем.
                # Вероятнее всего такие образовались в результате бага
                # на дорегистрации - не удаляли этот флаг
                if self.account.password.is_changing_required_by_any_reason and not self.account.domain.can_users_change_password:
                    return PasswordChangeForbidden()

        if self.account.is_federal and not self.account.is_complete_federal:
            return RedirectToFederalCompletion()

        if self.account.is_lite and is_experiment_enabled_by_uid(
                self.account.uid,
                settings.LITE_ACCOUNTS_ENFORCED_COMPLETION_DENOMINATOR,
        ):
            self.track.is_force_complete_lite = True
            return RedirectToForcedLiteCompletion()

        self.track.is_strong_password_policy_required = self.account.is_strong_password_required
        if check_strong_password_policy and self.account.is_strong_password_required:
            # Запрещать авторизацию социальным профилем при выставленной политике сложного пароля
            raise AccountStrongPasswordPolicyError()

        # Если пользователь создан автоматическим образом и не создал пароль - перенаправляем на установку пароля
        if self.account.is_incomplete_autoregistered:
            self.track.is_complete_autoregistered = True
            return RedirectToCompletion()

        # Если установлен специальный флаг - включен 2FA или нет пароля,
        # то все состояния качества пароля не имеют смысла.
        if self.account.totp_secret.is_set or not self.account.have_password:
            return

        if self.account.password.is_password_flushed_by_admin:
            # Специальный случай смены пароля: администратор меняет пароль пользователю и при первом входе
            # требует его сменить
            self.track.is_password_change = True
            self.track.is_force_change_password = True
            self.track.change_password_reason = CHANGE_PASSWORD_REASON_FLUSHED
            to_statbox({
                'mode': 'change_flushed_password',
                'action': 'redirect_to_password_change',
                'track_id': self.track.track_id,
                'uid': self.account.uid,
            })
            return RedirectToPasswordChange(
                reason=CHANGE_PASSWORD_REASON_FLUSHED,
                validation_method=NO_VALIDATION,
            )

        # Смена пароля для пользователей со слитым паролем
        is_password_pwned = False
        if (
            self.account.password.is_password_pwned or
            (not self.is_password_pwn_check_suspended() and self.is_password_pwned())
        ):
            is_password_pwned = True
            is_experiment_enabled = is_experiment_enabled_by_uid(
                self.account.uid,
                settings.PWNED_PASSWORD_CHANGE_DENOMINATOR,
            )
            is_redirect_required = is_experiment_enabled or self.account.password.is_password_pwned
            if self.account.sms_2fa_on or (self.account.is_pdd and not self.account.domain.can_users_change_password):
                is_redirect_required = False

            log.debug('Password for track %s is pwned, is redirect - %s' % (self.track.track_id, is_redirect_required))
            to_statbox({
                'mode': 'change_pwned_password',
                'action': 'redirect_to_password_change' if is_redirect_required else 'no_action',
                'track_id': self.track.track_id,
                'uid': self.account.uid,
                'is_weak': self.account.password.is_weak,
                'is_pwned': is_password_pwned,
                'is_pwn_check_suspended': self.is_password_pwn_check_suspended(),
                'is_experiment_enabled': is_experiment_enabled,
                'sms_2fa_on': self.account.sms_2fa_on or False,
                'is_no_change_password_pdd': self.account.is_pdd and not self.account.domain.can_users_change_password,
            })

            if is_redirect_required:
                self.track.is_password_change = True
                self.track.is_force_change_password = True
                self.track.change_password_reason = CHANGE_PASSWORD_REASON_PWNED
                self.try_experiments()

                if self.request and self.request.env:
                    with UPDATE(self.account, self.request.env, dict(action='password_update', consumer=self.consumer)):
                        self.account.password.setup_password_changing_requirement(
                            changing_reason=PASSWORD_CHANGING_REASON_PWNED,
                        )

                validation_method = self.get_validation_method()

                return RedirectToPasswordChange(
                    reason=CHANGE_PASSWORD_REASON_PWNED,
                    validation_method=validation_method,
                )
        elif not self.account.password.is_weak:
            to_statbox({
                'mode': 'change_pwned_password',
                'action': 'no_action',
                'track_id': self.track.track_id,
                'uid': self.account.uid,
                'is_weak': False,
                'is_pwned': False,
                'is_pwn_check_suspended': self.is_password_pwn_check_suspended(),
                'is_experiment_enabled': False,
                'sms_2fa_on': self.account.sms_2fa_on or False,
                'is_no_change_password_pdd': self.account.is_pdd and not self.account.domain.can_users_change_password,
            })

        if self.account.password.is_changing_required:
            # Смена пароля для пользователей с подозрением на взлом аккаунта
            self.track.is_password_change = True
            self.track.is_force_change_password = True
            self.track.is_captcha_required = True
            if self.track.is_captcha_checked:
                self.track.is_captcha_checked = False
            if self.track.is_captcha_recognized:
                self.track.is_captcha_recognized = False
            self.track.change_password_reason = CHANGE_PASSWORD_REASON_HACKED
            self.try_experiments()

            validation_method = self.get_validation_method()
            to_statbox({
                'mode': 'change_password_force',
                'action': 'defined_validation_method',
                'track_id': self.track.track_id,
                'validation_method': validation_method,
                'uid': self.account.uid,
            })

            return RedirectToPasswordChange(
                reason=CHANGE_PASSWORD_REASON_HACKED,
                validation_method=validation_method,
            )

        # Дополнительно проверим качество пароля и, если он слабый, перенаправляем на смену пароля
        if self.account.password.is_complication_required:
            self.track.is_password_change = True
            self.track.is_force_change_password = True
            self.track.change_password_reason = CHANGE_PASSWORD_REASON_TOO_WEAK

            log.debug('Password for track %s is weak and need complication' % self.track.track_id)
            to_statbox({
                'mode': 'change_weak_password',
                'action': 'redirect_to_password_change',
                'track_id': self.track.track_id,
                'uid': self.account.uid,
                'is_weak': self.account.password.is_weak,
                'is_pwned': is_password_pwned,
            })

            return RedirectToPasswordChange(
                reason=CHANGE_PASSWORD_REASON_TOO_WEAK,
                validation_method=NO_VALIDATION,
            )

        # Дополнительно убедимся, что пароль не нуждается в смене
        elif self.is_password_expired:
            self.track.is_password_change = True
            self.track.is_force_change_password = True
            self.track.change_password_reason = CHANGE_PASSWORD_REASON_EXPIRED
            return RedirectToPasswordChange(
                reason=CHANGE_PASSWORD_REASON_EXPIRED,
                validation_method=NO_VALIDATION,
            )

    def fill_track_with_account_data(self, allow_create_session=False, allow_create_token=False,
                                     password_passed=False, **kwargs):
        """Запишем в трек все что уже узнали о пользователе"""
        set_authorization_track_fields(
            self.account,
            self.track,
            allow_create_session=allow_create_session,
            allow_create_token=allow_create_token,
            password_passed=password_passed,
            **kwargs
        )

        if self.account.is_pdd:
            self.track.domain = self.account.domain.domain

        self.track.is_strong_password_policy_required = self.account.is_strong_password_required

    def fill_response_with_account_and_session(
        self,
        personal_data_required=False,
        cookie_session_info=None,
        auth_type=None,
        is_lite=None,
        ignore_non_auth_cookies=False,
        otp_magic_passed=None,
        x_token_magic_passed=False,
        logout_check_enabled=True,
        is_2fa_enabled_yp=None,
        retpath=None,
        session_scope=None,
    ):
        # Если кука целиком валидна, то вызываем editsession,
        # который разберется, как добавить аккаунт.
        # editsession работает и с кукой 2ой версии (выдавая третью)
        extend_session = cookie_session_info.cookie_status not in (
            CookieCheckStatus.Invalid,
            CookieCheckStatus.WrongGuard,
        )
        multi_session_users = users_from_multisession(cookie_session_info)
        # Подстрахуемся, и если ЧЯ явно не говорит,
        # что нельзя добавлять еще пользоваталей, то считаем что можно
        if extend_session and not cookie_session_info.response.get('allow_more_users', True) and (
            self.account.uid not in multi_session_users
        ):
            raise AuthSessionOverflowError()
        self.response_values['accounts'] = self.get_multisession_accounts(cookie_session_info)

        try:
            logout_datetime = self.account.web_sessions_logout_datetime if logout_check_enabled else None

            if is_2fa_enabled_yp is None:
                is_2fa_enabled = self.account.totp_secret is not None and self.account.totp_secret.is_set
                is_2fa_enabled_yp = True if is_2fa_enabled else None

            cookies, _, service_guard_container = build_auth_cookies_and_session(
                self.request.env,
                self.track,
                account_type=self.account.type,
                auth_type=auth_type,
                authorization_session_policy=self.track.authorization_session_policy,
                display_name=self.account.person.display_name,
                extend_session=extend_session,
                ignore_non_auth_cookies=ignore_non_auth_cookies,
                is_2fa_enabled=self.account.totp_secret is not None and self.account.totp_secret.is_set,
                is_2fa_enabled_yp=is_2fa_enabled_yp,
                is_betatester=self.account.is_betatester,
                is_lite=is_lite,
                is_yandexoid=self.account.is_yandexoid,
                logout_datetime=logout_datetime,
                multi_session_users=multi_session_users,
                need_extra_sessguard=True,
                otp_magic_passed=otp_magic_passed,
                session_scope=session_scope,
                x_token_magic_passed=x_token_magic_passed,
                is_child=self.account.is_child,
            )

            if service_guard_container:
                self.response_values.update({
                    'service_guard_container': service_guard_container.pack(),
                })

            self.response_values.update({
                'cookies': cookies,
                'default_uid': self.account.uid,
                'sensitive_fields': ['cookies'],
            })  # последняя стадия - track_id не возвращаем
            self.fill_response_with_account(personal_data_required=personal_data_required)

        except AccountGlobalLogoutError:
            raise BundleAccountGlobalLogoutError()

        except SessionExpiredError:
            raise AuthSessionExpiredError()

    def is_magnitola_device(self):
        return (
            self.track.device_hardware_model and
            self.track.device_hardware_model.lower() in settings.MAGNITOLA_MODELS and
            self.track.device_application and
            self.track.device_application in settings.MAGNITOLA_APP_IDS
        )

    def is_password_pwn_check_suspended(self):
        suspend_time = self.account.password.pwn_forced_changing_suspended_at
        if not suspend_time:
            return False

        time_diff = datetime.now() - suspend_time
        return time_diff.days < settings.PASSWORD_PWN_CHECK_SUSPENSION_DAYS

    def is_password_pwned(self):
        """
        Сходим в шакур и посмотрим, не слит ли пароль,
        если есть обычный плейнтекст пароль
        """
        if self.form_values.get('password') is None:
            return False

        if self.is_magnitola_device():
            return False

        if is_login_in_shakur_whitelist(self.account.login):
            return False

        if self.account.scholar_alias:
            return False

        return check_password_is_pwned(self.form_values['password'])

    def set_old_session_track_fields(self, cookie_session_info):
        set_old_session_track_fields(self.track, cookie_session_info)
