# -*- coding: utf-8 -*-

import logging

from passport.backend.api.views.bundle.auth.drive.exceptions import (
    DriveSessionNotFoundError,
    NeedNewNonceError,
    PublicKeyNotFoundError,
)
from passport.backend.api.views.bundle.auth.drive.forms import (
    BuildNonceForm,
    IssueAuthorizationCodeForm,
    StartForm,
    StopForm,
)
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import (
    InternalPermanentError,
    OAuthUnavailableError,
    ResourceUnavailableError,
)
from passport.backend.api.views.bundle.mixins.common import BundleTvmUserTicketMixin
from passport.backend.api.views.bundle.mixins.oauth import BundleOAuthMixin
from passport.backend.core.builders.blackbox.constants import (
    BLACKBOX_CHECK_DEVICE_SIGNATURE_RETRIABLE_STATUS,
    BLACKBOX_GET_DEVICE_PUBLIC_KEY_STATUS,
)
from passport.backend.core.builders.blackbox.exceptions import (
    BlackboxInvalidDeviceSignature,
    BlackboxInvalidResponseError,
    BlackboxTemporaryError,
)
from passport.backend.core.builders.drive_api.drive_api import (
    DriveApiPermanentError,
    DriveApiTemporaryError,
    get_drive_api,
)
from passport.backend.core.builders.oauth.oauth import (
    get_oauth,
    OAUTH_CODE_STRENGTH_LONG,
)
from passport.backend.core.conf import settings
from passport.backend.core.device_public_key import find_device_public_key
from passport.backend.core.grants import check_grant
from passport.backend.core.logging_utils.loggers.statbox import StatboxLogger
from passport.backend.core.models.drive import DriveSession
from passport.backend.core.utils.decorators import cached_property
from passport.backend.core.ydb.exceptions import YdbTemporaryError
from passport.backend.core.ydb.processors.drive import (
    delete_drive_session,
    find_drive_session,
    save_drive_session,
)


log = logging.getLogger(__name__)


DRIVE_SCOPES = ['carsharing:all', 'yataxi:write']
DRIVE_DEVICE_CONSUMER = 'drive_device'


class BaseDriveView(BaseBundleView):
    def check_api_enabled(self):
        if not settings.DRIVE_AUTH_FORWARDING_API_ENABLED:
            log.debug('Drive authentication forwarding disabled')
            raise InternalPermanentError()

    def revoke_tokens_async(self, drive_session):
        # Токены отзываем через Логброкер
        if not drive_session.sandbox_device_id:
            log.debug('no sandbox_device_id in session, nothing to revoke: %s' % drive_session)
            return

        log.debug('revoke drive session tokens: %s' % drive_session)
        self.statbox.log(
            action='revoke_drive_device',
            uid=drive_session.uid,
            device_id=drive_session.sandbox_device_id,
        )

    def check_device_ip(self, device_ip):
        check_grant(
            'auth_forward_drive.public_access',
            device_ip,
            DRIVE_DEVICE_CONSUMER,
        )

    def find_device_public_key(self, device_id):
        try:
            return find_device_public_key(device_id)
        except BlackboxTemporaryError:
            log.debug('Blackbox temporary unavailable')
            raise ResourceUnavailableError()

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode='forward_auth_to_drive_device',
            consumer=self.consumer,
            ip=self.client_ip,
        )

    @cached_property
    def oauth(self):
        return get_oauth()


class AuthDriveStartView(BaseDriveView, BundleTvmUserTicketMixin, BundleOAuthMixin):
    basic_form = StartForm
    required_grants = ['auth_forward_drive.start']
    required_user_ticket_scopes = DRIVE_SCOPES

    def process_request(self):
        self.check_api_enabled()

        self.process_basic_form()

        user_ticket = self.check_user_ticket()
        uid = self.get_uid_or_default_from_user_ticket(user_ticket)

        old_drive_session = find_drive_session(self.form_values['drive_device_id'])
        new_drive_session = DriveSession(
            drive_device_id=self.form_values['drive_device_id'],
            drive_session_id=self.form_values['drive_session_id'],
            uid=uid,
        )
        if new_drive_session.is_same(old_drive_session):
            log.debug('Nothing to do, because drive session already exists: %s' % old_drive_session)
            return

        if old_drive_session and new_drive_session.uid != old_drive_session.uid:
            self.revoke_tokens_async(old_drive_session)

        log_message = 'Create new drive session: %s' % new_drive_session
        if old_drive_session:
            log_message += ' (replace %s)' % old_drive_session
        log.debug(log_message)

        save_drive_session(new_drive_session)

        self.statbox.log(
            action='create_drive_session',
            delete_old_session=bool(old_drive_session),
            uid=uid,
            old_uid=old_drive_session.uid if old_drive_session else None,
            drive_device_id=self.form_values['drive_device_id'],
        )


class AuthDriveBuildNonceView(BaseDriveView):
    basic_form = BuildNonceForm
    required_grants = ['auth_forward_drive.build_nonce']

    def process_request(self):
        self.process_basic_form()

        device_public_key = self.find_device_public_key(self.form_values['drive_device_id'])
        if (
            device_public_key and
            device_public_key.owner_id == settings.DRIVE_PRODUCTION_PUBLIC_KEY_OWNER_ID
        ):
            self.check_device_ip(self.client_ip)

        try:
            blackbox_response = self.blackbox.sign(
                value=self.form_values['drive_device_id'],
                ttl=settings.DRIVE_AUTH_FORWARDING_NONCE_TTL,
                sign_space=settings.DRIVE_NONCE_SIGN_SPACE,
            )
        except BlackboxTemporaryError:
            log.debug('Blackbox temporary unavailable')
            raise ResourceUnavailableError()
        nonce = blackbox_response['signed_value']

        self.statbox.log(
            action='build_nonce',
            drive_device_id=self.form_values['drive_device_id'],
        )
        self.response_values.update(nonce=nonce)


class AuthDriveIssueAuthorizationCodeView(BaseDriveView, BundleOAuthMixin):
    basic_form = IssueAuthorizationCodeForm
    required_grants = ['auth_forward_drive.issue_authorization_code']

    def process_request(self):
        self.check_api_enabled()

        self.process_basic_form()

        oauth_client_id = self.form_values['oauth_client_id']
        oauth_client_secret = self.form_values['oauth_client_secret']

        if oauth_client_id is None and oauth_client_secret is None:
            oauth_client_id = settings.OAUTH_APPLICATION_AM_XTOKEN['client_id']
            oauth_client_secret = settings.OAUTH_APPLICATION_AM_XTOKEN['client_secret']

        is_virtual = False
        device_public_key = self.find_device_public_key(self.form_values['drive_device_id'])
        if device_public_key:
            if device_public_key.owner_id == settings.DRIVE_PRODUCTION_PUBLIC_KEY_OWNER_ID:
                self.check_device_ip(self.client_ip)
            elif device_public_key.owner_id == settings.DRIVE_VIRTUAL_PUBLIC_KEY_OWNER_ID:
                is_virtual = True

        self.check_nonce(
            self.form_values['nonce'],
            self.form_values['drive_device_id'],
            self.form_values['signature'],
        )

        drive_session = self.find_drive_session(self.form_values['drive_device_id'])
        if not drive_session:
            log.debug('Drive session not found in YDB: %s' % self.form_values['drive_device_id'])
            raise DriveSessionNotFoundError()

        if self.form_values['check_drive_session']:
            self.check_drive_knows_session(drive_session)
        else:
            if is_virtual:
                log.debug('Skipping checking drive session for virtual device')
            else:
                log.debug('Cannot skip checking drive session for non-virtual device')
                raise InternalPermanentError()

        sandbox_device_id = self.form_values['sandbox_device_id']
        if drive_session.sandbox_device_id != sandbox_device_id and settings.ALLOW_DRIVE_SANDBOX_IDS:
            self.revoke_tokens_async(drive_session)
            log.debug(
                'Changing sandbox_device_id from %r to %r for session: %s',
                drive_session.sandbox_device_id,
                sandbox_device_id,
                drive_session,
            )
            drive_session.sandbox_device_id = sandbox_device_id
            save_drive_session(drive_session)

        log.debug('Issue authorization code for drive session: %s' % drive_session)

        authorization_code = self.issue_authorization_code(
            client_id=oauth_client_id,
            client_secret=oauth_client_secret,
            uid=drive_session.uid,
        )

        self.statbox.log(
            action='issue_authorization_code',
            uid=drive_session.uid,
            drive_device_id=drive_session.drive_device_id,
        )

        self.response_values.update(authorization_code=authorization_code)

    def check_nonce(self, nonce, drive_device_id, signature):
        try:
            self.blackbox.check_device_signature(
                nonce,
                settings.DRIVE_NONCE_SIGN_SPACE,
                drive_device_id,
                signature,
            )
            status = 'ok'
        except BlackboxInvalidDeviceSignature as e:
            log.debug('Invalid signature: %s, %s' % (e, e.security_info))
            status = (e.status or '').lower()
            if e.status in BLACKBOX_CHECK_DEVICE_SIGNATURE_RETRIABLE_STATUS:
                raise NeedNewNonceError()
            elif e.status == BLACKBOX_GET_DEVICE_PUBLIC_KEY_STATUS.PUBLIC_KEY_NOT_FOUND:
                raise PublicKeyNotFoundError()
            else:
                raise InternalPermanentError()
        except BlackboxTemporaryError:
            log.debug('Blackbox temporary unavailable')
            status = 'temporary_error'
            raise ResourceUnavailableError()
        except BlackboxInvalidResponseError:
            log.debug('Blackbox invalid response')
            status = 'invalid_response'
            raise ResourceUnavailableError()
        finally:
            self.statbox.log(
                action='check_nonce',
                status=status,
            )

    def check_drive_knows_session(self, drive_session):
        if (
            settings.UNKNOWN_DRIVE_SESSIONS_ALLOWED and
            self.form_values['test_drive_session_id']
        ):
            drive_session_id2 = self.form_values['test_drive_session_id']
        else:
            try:
                drive_session_id2 = self.drive_api.find_drive_session_id(drive_session.drive_device_id)
            except DriveApiTemporaryError:
                log.debug('Drive API temporary unavailable')
                raise ResourceUnavailableError()
            except DriveApiPermanentError:
                raise InternalPermanentError()
        if not drive_session_id2:
            log.debug('Drive session not found in Drive: %s' % drive_session)
            raise InternalPermanentError()
        if drive_session.drive_session_id != drive_session_id2:
            log.debug(
                'Drive session id from Drive does not match drive session id from YDB: %s != %s (%s)' % (
                    drive_session_id2,
                    drive_session.drive_session_id,
                    drive_session,
                ),
            )
            raise InternalPermanentError()

    def issue_authorization_code(
        self,
        client_id,
        client_secret,
        uid,
    ):
        try:
            return self.oauth_authorization_code_by_uid(
                client_id=client_id,
                client_secret=client_secret,
                uid=uid,
                code_strength=OAUTH_CODE_STRENGTH_LONG,
                require_activation=False,
            )
        except OAuthUnavailableError:
            log.debug('OAuth unavailable error')
            raise ResourceUnavailableError()

    def find_drive_session(self, drive_device_id):
        try:
            return find_drive_session(drive_device_id)
        except YdbTemporaryError:
            log.debug('YDB temporary unavailable')
            raise ResourceUnavailableError()

    @cached_property
    def drive_api(self):
        return get_drive_api()


class AuthDriveStopView(BaseDriveView, BundleTvmUserTicketMixin, BundleOAuthMixin):
    basic_form = StopForm
    required_grants = ['auth_forward_drive.stop']
    required_user_ticket_scopes = DRIVE_SCOPES

    def process_request(self):
        self.check_api_enabled()

        self.process_basic_form()

        user_ticket = self.check_user_ticket()
        uid = self.get_uid_or_default_from_user_ticket(user_ticket)

        drive_session = find_drive_session(self.form_values['drive_device_id'])

        if drive_session:
            self.revoke_tokens_async(drive_session)
            log.debug('Delete drive session: %s' % drive_session)
            delete_drive_session(drive_session.drive_device_id)

        self.statbox.log(
            action='delete_drive_session',
            uid=uid,
            old_uid=drive_session.uid if drive_session else None,
            drive_device_id=self.form_values['drive_device_id'],
        )
