# -*- coding: utf-8 -*-
import logging

from passport.backend.api.common.authorization import (
    authorize_oauth,
    is_oauth_token_created,
    set_authorization_track_fields,
    user_session_scope,
)
from passport.backend.api.common.common import extract_tld
from passport.backend.api.common.processes import PROCESS_FORWARD_AUTH_TO_MOBILE_BY_TRACK
from passport.backend.api.forms.base import DeviceInfoForm
from passport.backend.api.views.bundle.auth.base import BundleBaseAuthorizationMixin
from passport.backend.api.views.bundle.auth.exceptions import AuthAlreadyPassedError
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import (
    AccountGlobalLogoutError,
    AccountStrongPasswordPolicyError,
    InvalidTrackStateError,
    OAuthUnavailableError,
    PasswordRequiredError,
    RateLimitExceedError,
    ResourceUnavailableError,
    SecurePhoneNotFoundError,
    ValidationFailedError,
)
from passport.backend.api.views.bundle.headers import (
    HEADER_CLIENT_COOKIE,
    HEADER_CLIENT_HOST,
    HEADER_CLIENT_USER_AGENT,
    HEADER_CONSUMER_CLIENT_IP,
)
from passport.backend.api.views.bundle.mixins import (
    BundleFixPDDRetpathMixin,
    BundlePhoneMixin,
)
from passport.backend.api.views.bundle.mixins.account import (
    BundleAccountGetterMixin,
    BundleAccountPropertiesMixin,
    BundleAuthNotificationsMixin,
)
from passport.backend.api.views.bundle.phone.helpers import dump_number
from passport.backend.api.views.bundle.restore.exceptions import PhoneChangedError
from passport.backend.core import authtypes
from passport.backend.core.builders.blackbox.exceptions import BlackboxTemporaryError
from passport.backend.core.builders.oauth.oauth import (
    get_oauth,
    OAuthPermanentError,
    OAuthTemporaryError,
)
from passport.backend.core.conf import settings
from passport.backend.core.counters.sms_per_ip import get_auth_forwarding_by_sms_counter
from passport.backend.core.logging_utils.loggers.statbox import StatboxLogger
from passport.backend.core.redis_manager.redis_manager import RedisError
from passport.backend.core.tracks.exceptions import TrackNotFoundError
from passport.backend.core.utils.decorators import cached_property
from passport.backend.core.utils.domains import get_keyspace_by_host
from passport.backend.utils.time import get_unixtime
from six.moves.urllib.parse import urlparse

from .forms import (
    AUTH_LINK_PLACEHOLDER,
    CommitForm,
    ExchangeByTrackAuthForwardingForm,
    SubmitForm,
)
from .helpers import get_short_track_link


log = logging.getLogger('passport.api.views.bundle.auth.forwarding.controllers')


class BaseAuthForwardingView(BaseBundleView, BundleAccountGetterMixin, BundleBaseAuthorizationMixin):
    track_type = 'authorize'

    required_headers = (HEADER_CONSUMER_CLIENT_IP,)

    def raise_error_with_logging(self, exception_type):
        self.statbox.log(action='finished_with_error', error=exception_type.error)
        raise exception_type()


class BaseBySmsAuthForwardingView(BundlePhoneMixin, BaseAuthForwardingView):
    required_headers = BaseAuthForwardingView.required_headers + (
        HEADER_CLIENT_COOKIE,
        HEADER_CLIENT_USER_AGENT,
        HEADER_CLIENT_HOST,
    )

    def check_counters(self):
        counter = get_auth_forwarding_by_sms_counter(self.client_ip)
        if counter.hit_limit_by_ip(self.client_ip):
            raise RateLimitExceedError()

    def increase_counters(self):
        counter = get_auth_forwarding_by_sms_counter(self.client_ip)
        counter.incr(self.client_ip)

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode='auth_forwarding',
            host=self.host,
            uid=self.account.uid,
            track_id=self.track.track_id,
            ip=self.client_ip,
            user_agent=self.user_agent,
            yandexuid=self.cookies.get('yandexuid'),
            consumer=self.consumer,
        )


class ForwardAuthBySmsSubmitView(BaseBySmsAuthForwardingView):

    require_track = False

    required_grants = ['auth_forwarding.by_sms']

    basic_form = SubmitForm

    def process_request(self, *args, **kwargs):
        self.process_basic_form()
        self.check_counters()
        self.create_track(self.track_type)

        self.get_account_from_session(
            multisession_uid=self.form_values['uid'],
            need_phones=True,
        )

        secure_number = self.account.phones.secure and self.account.phones.secure.number
        if not secure_number:
            self.raise_error_with_logging(SecurePhoneNotFoundError)

        with self.track_transaction.rollback_on_error():
            self.track.secure_phone_number = secure_number.e164
            self.track.uid = self.account.uid

        self.response_values.update(
            track_id=self.track_id,
            number=dump_number(secure_number, only_masked=True),
        )
        self.statbox.log(
            action='submitted',
        )


class ForwardAuthBySmsCommitView(BaseBySmsAuthForwardingView, BundleFixPDDRetpathMixin):

    require_track = True

    required_grants = ['auth_forwarding.by_sms']

    basic_form = CommitForm

    def process_request(self, *args, **kwargs):
        self.process_basic_form()
        self.check_counters()
        self.read_track()
        self.assert_retpath_within_auth_domain()
        if not self.track.secure_phone_number or not self.track.uid:
            raise InvalidTrackStateError()
        self.get_pinned_account_from_session(
            need_phones=True,
        )

        secure_number = self.account.phones.secure and self.account.phones.secure.number
        if not secure_number or secure_number.e164 != self.track.secure_phone_number:
            self.raise_error_with_logging(PhoneChangedError)

        if self.account.is_pdd:
            self.fix_pdd_retpath()

        auth_track = self.track_manager.create_short(
            'authorize',
            self.consumer,
            ttl=settings.AUTH_FORWARDING_TRACK_TTL,
        )
        with self.track_manager.transaction(track=auth_track).rollback_on_error() as auth_track:
            set_authorization_track_fields(
                self.account,
                auth_track,
                allow_create_session=True,
                allow_create_token=False,
                auth_source=authtypes.AUTH_SOURCE_FOREIGN_COOKIE,
                password_passed=False,
                session_scope=user_session_scope(self.session_info, self.account.uid),
                source_authid=self.session_info.authid,
                # Не сохраняем текущую сессию в трек, т.к. предполагается авторизация на другом устройстве,
                # где может быть другая сессия
            )
            auth_track.retpath = self.form_values['retpath']

        with self.track_transaction.rollback_on_error():
            # Записываем ID нового трека для удобства тестирования
            self.track.next_track_id = auth_track.track_id

        sms_text = self.get_sms_text(auth_track)
        self.send_sms(secure_number, sms_text, settings.AUTH_FORWARDING_SMS_IDENTITY, ignore_errors=False)
        self.increase_counters()
        self.response_values.update(
            track_id=self.track_id,
            number=dump_number(secure_number, only_masked=True),
        )
        self.statbox.log(
            action='committed',
        )

    def get_sms_text(self, auth_track):
        tld = extract_tld(self.host, settings.PASSPORT_TLDS) or settings.PASSPORT_DEFAULT_TLD
        link = get_short_track_link(track_id=auth_track.track_id, tld=tld)
        return self.form_values['template'].replace(AUTH_LINK_PLACEHOLDER, link)

    def assert_retpath_within_auth_domain(self):
        if not self.form_values['retpath']:
            return
        retpath_host = urlparse(self.form_values['retpath']).hostname
        expected_keyspace = get_keyspace_by_host(self.host)
        if not retpath_host.endswith('.' + expected_keyspace) and retpath_host != expected_keyspace:
            log.debug('Retpath host "%s" does not match auth domain "%s"', retpath_host, expected_keyspace)
            raise ValidationFailedError(['retpath.invalid'])


class BaseByTrackAuthForwardingView(BaseAuthForwardingView):
    def check_track_is_auth_forwarding_track(self):
        """
        Проверяет, что по треку когда-то (возможно в будущем) можно было
        получить токен.
        """
        if self.track.track_type != 'authorize':
            raise InvalidTrackStateError()
        if not self.track.allow_oauth_authorization:
            raise InvalidTrackStateError()

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode='auth_forwarding_by_track',
            host=self.host,
            uid=self.account.uid,
            track_id=self.track.track_id,
            ip=self.client_ip,
            user_agent=self.user_agent,
            yandexuid=self.cookies.get('yandexuid'),
            consumer=self.consumer,
        )


class CreateTrackByTrackForwardingView(BundleAccountPropertiesMixin, BaseByTrackAuthForwardingView):
    """
    Ручка для проброса аутентикации через QR-код из веба в мобильные: создание
    трека.
    """

    required_grants = ['auth_forwarding_by_track.create_track']

    required_headers = BaseAuthForwardingView.required_headers + (
        HEADER_CLIENT_COOKIE,
        HEADER_CLIENT_USER_AGENT,
        HEADER_CLIENT_HOST,
    )

    def process_request(self):
        self.get_account_from_session()
        self.check_user_policies()
        self.create_auth_forwarding_track()
        self.statbox.log(action='create_track')
        track_expire_at = get_unixtime() + self.track.ttl
        self.response_values.update(
            track_id=self.track_id,
            track_expire_at=track_expire_at,
        )

    def check_user_policies(self):
        """
        Ищет причину запретить пользователю пробрасывать авторизацию из веба в
        мобильное устройство.
        """
        if self.account.is_strong_password_required:
            raise AccountStrongPasswordPolicyError()
        if (
            self.account.totp_secret.is_set and
            self.is_password_verification_required()
        ):
            raise PasswordRequiredError()

    def create_auth_forwarding_track(self):
        self.create_track(
            track_type='authorize',
            process_name=PROCESS_FORWARD_AUTH_TO_MOBILE_BY_TRACK,
            ttl=settings.AUTH_FORWARDING_TRACK_TTL,
        )
        with self.track_transaction.rollback_on_error():
            self.fill_track_with_account_data(
                allow_create_token=True,
                # Не сохраняем текущую сессию в трек, т.к. предполагается
                # авторизация на другом устройстве, где может быть другая сессия
                #
                # Не сохраняем тип исходного средства авторизации и его
                # идентификатор, потому что сейчас нет способа передать их в
                # Oauth, т.е. Oauth не пишет их в History DB.
            )


class ExchangeByTrackAuthForwardingView(
    BaseByTrackAuthForwardingView,
    BundleAuthNotificationsMixin,
):
    """
    Ручка для проброса аутентикации через QR-код из веба в мобильные: обмена
    трека на токен.
    """

    required_grants = ['auth_forwarding_by_track.exchange']

    basic_form = ExchangeByTrackAuthForwardingForm

    require_track = True
    require_process = True
    allowed_processes = [PROCESS_FORWARD_AUTH_TO_MOBILE_BY_TRACK]

    def process_request(self):
        try:
            self.process_basic_form()
            self.read_track()
            self.check_track_is_auth_forwarding_track()
            self.check_track_not_used_to_issue_token()
            self.get_account_from_track(emails='getall')
            device_status = self.oauth_device_status_failsafe()
            oauth_token = self.issue_oauth_token()
            if not self.track.oauth_token_created_at:
                with self.track_transaction.rollback_on_error():
                    self.track.oauth_token_created_at = get_unixtime()
            if device_status:
                if not device_status['has_auth_on_device']:
                    self.try_send_auth_notifications(
                        device_name=device_status['device_name'],
                    )
                else:
                    log.debug('User notification not required: known device')
            self.statbox.log(action='exchange')
            self.response_values.update(token=oauth_token)
        except (
            AccountGlobalLogoutError,
            AuthAlreadyPassedError,
            InvalidTrackStateError,
            TrackNotFoundError,
        ) as error:
            log.debug('Failed to exchange track to token: %s' % error)
            # Предложить пользователю попробовать другой трек
            raise InvalidTrackStateError()
        except (
            BlackboxTemporaryError,
            OAuthTemporaryError,
            OAuthUnavailableError,
            RedisError,
        ) as error:
            log.debug('Failed to exchange track to token: %s' % error)
            # Предложить пользователю попробовать позже этот же трек
            raise ResourceUnavailableError()

    def check_track_not_used_to_issue_token(self):
        """
        Проверяет, что по треку ещё не получали токен.
        """
        if is_oauth_token_created(self.track):
            raise AuthAlreadyPassedError()

    def issue_oauth_token(self):
        token_response = authorize_oauth(
            client_id=self.form_values['client_id'],
            client_secret=self.form_values['client_secret'],
            env=self.request.env,
            track=self.track,
            **self.device_info
        )
        access_token = token_response.get('access_token')
        if not access_token:
            error_description = token_response.get('error_description') or ''
            log.debug('Failed to get oauth token: %s' % error_description)
            raise OAuthUnavailableError(error_description)
        return access_token

    def oauth_device_status_failsafe(self):
        device_status = None
        try:
            device_status = self.oauth.device_status(self.account.uid, **self.device_info)
        except (OAuthPermanentError, OAuthTemporaryError) as e:
            log.debug('Unable to get device status: %s' % type(e).__name__)
        return device_status

    @cached_property
    def device_info(self):
        return {k: self.form_values[k] for k in DeviceInfoForm.DEVICE_INFO_FIELD_NAMES}

    @cached_property
    def oauth(self):
        return get_oauth()


class GetStatusByTrackAuthForwardingView(BaseByTrackAuthForwardingView):
    """
    Ручка сообщает в каком состоянии находится пробрасывающий аутентикацию
    трек.
    """

    required_grants = ['auth_forwarding_by_track.get_status']

    require_track = True
    require_process = True
    allowed_processes = [PROCESS_FORWARD_AUTH_TO_MOBILE_BY_TRACK]

    required_headers = BaseAuthForwardingView.required_headers + (
        HEADER_CLIENT_COOKIE,
        HEADER_CLIENT_USER_AGENT,
        HEADER_CLIENT_HOST,
    )

    def process_request(self):
        self.read_track()
        self.get_pinned_account_from_session()
        self.check_track_is_auth_forwarding_track()
        token_issued = bool(self.track.oauth_token_created_at)
        track_expire_at = get_unixtime() + self.track.ttl
        self.response_values.update(
            token_issued=token_issued,
            track_expire_at=track_expire_at,
            track_id=self.track_id,
        )
