# -*- coding: utf-8 -*-
import json

from passport.backend.api.views.bundle.auth.base import BundleBaseAuthorizationMixin
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import (
    AccountGlobalLogoutError,
    ActionImpossibleError,
    SecretKeyInvalidError,
)
from passport.backend.api.views.bundle.headers import (
    HEADER_CLIENT_COOKIE,
    HEADER_CLIENT_HOST,
    HEADER_CLIENT_USER_AGENT,
    HEADER_CONSUMER_CLIENT_IP,
)
from passport.backend.api.views.bundle.mixins import (
    BundleAccountGetterMixin,
    BundlePersistentTrackMixin,
)
from passport.backend.api.views.bundle.states import (
    RedirectToCompletion,
    RedirectToSocialCompletion,
)
from passport.backend.core.logging_utils.loggers.statbox import StatboxLogger
from passport.backend.core.models.persistent_track import TRACK_TYPE_AUTH_BY_KEY_LINK
from passport.backend.core.utils.decorators import cached_property

from .forms import SubmitKeyLinkForm


STATE_TO_TRACK_TYPE_MAPPING = {
    RedirectToCompletion: 'authorize',
    RedirectToSocialCompletion: 'complete',
}


class SubmitKeyLinkView(BaseBundleView,
                        BundlePersistentTrackMixin,
                        BundleBaseAuthorizationMixin,
                        BundleAccountGetterMixin,
                        ):
    basic_form = SubmitKeyLinkForm

    required_grants = ['auth_key.base']

    required_headers = [
        HEADER_CLIENT_HOST,
        HEADER_CLIENT_USER_AGENT,
        HEADER_CLIENT_COOKIE,
        HEADER_CONSUMER_CLIENT_IP,
    ]

    statbox_mode = 'key_auth'

    def raise_error_with_logging(self, exception_type):
        self.statbox.log(
            action='finished_with_error',
            error=exception_type.error,
        )
        raise exception_type()

    @cached_property
    def statbox(self):
        statbox_params = dict(
            mode=self.statbox_mode,
            track_id=self.track_id,
            ip=self.client_ip,
            host=self.host,
            user_agent=self.user_agent,
            yandexuid=self.cookies.get('yandexuid'),
            consumer=self.consumer,
        )
        return StatboxLogger(**statbox_params)

    def validate_persistent_track_and_account(self, parsed_track_key):
        uid = parsed_track_key['uid']
        track_id = parsed_track_key['track_id']
        persistent_track = self.read_persistent_track(uid, track_id)

        self.statbox.bind_context(
            uid=uid,
            does_key_exist=persistent_track.exists,
            is_key_expired=persistent_track.is_expired if persistent_track.exists else None,
            key_type=persistent_track.type if persistent_track.exists else None,
        )
        if (not persistent_track.exists or
                persistent_track.is_expired or
                persistent_track.type != TRACK_TYPE_AUTH_BY_KEY_LINK):

            self.raise_error_with_logging(SecretKeyInvalidError)

        # Проверяем что персистентный трек был создан после логаута
        self.get_account_by_uid(uid)
        if self.account.is_logouted_after(persistent_track.created):
            self.raise_error_with_logging(AccountGlobalLogoutError)

    def check_user_policies(self):
        """
        Аутентификация по ключу возможна только в определенных случаях:
        1. Дорегистрация автозарегистрированного пользователя, по ссылке, полученной автоматически при регистрации.
        2. Дорегистрация пользователя с портальным и социальным алиасами, но без пароля; дорегистрация
        автозарегистрированного - по ссылке от саппорта.
        """
        # Если пользователь создан автоматическим образом и не создал пароль - перенаправляем на установку пароля
        if self.account.is_incomplete_autoregistered:
            return RedirectToCompletion()
        # Соц. пользователь, выставивший себе нормальный логин, но еще не имеющий пароля
        if self.account.is_normal and self.account.social_alias and not self.account.have_password:
            return RedirectToSocialCompletion()
        self.raise_error_with_logging(ActionImpossibleError)

    def process_request(self):
        self.process_basic_form()
        secret_key = self.form_values['secret_key']

        self.validate_persistent_track_and_account(secret_key)
        redirect_state = self.check_user_policies()
        track_type = STATE_TO_TRACK_TYPE_MAPPING[type(redirect_state)]
        self.create_track(track_type)
        self.statbox.bind_context(
            track_id=self.track_id,
            state=redirect_state.state,
        )

        with self.track_transaction.rollback_on_error() as self.track:
            if isinstance(redirect_state, RedirectToCompletion):
                self.fill_track_with_account_data()
                self.track.is_complete_autoregistered = True
            elif isinstance(redirect_state, RedirectToSocialCompletion):
                self.track.is_key_auth_passed = True
                self.track.uid = self.account.uid

        self.state = redirect_state
        self.response_values.update(track_id=self.track.track_id)
        self.statbox.log(action='submitted', type='key')
