# -*- coding: utf-8 -*-
import logging

from passport.backend.api.common.authorization import (
    is_user_default_in_multisession,
    is_user_in_multisession,
    user_from_multisession,
)
from passport.backend.api.forms.base import UidForm
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import (
    ActionNotRequiredError,
    SessguardInvalidError,
    SessionidInvalidError,
    UidNotInSessionError,
)
from passport.backend.api.views.bundle.headers import (
    HEADER_CLIENT_COOKIE,
    HEADER_CLIENT_HOST,
    HEADER_CONSUMER_CLIENT_IP,
)
from passport.backend.api.views.bundle.mixins import (
    BundleAccountGetterMixin,
    BundleAccountResponseRendererMixin,
    BundleEditSessionMixin,
    CookieCheckStatus,
)
from passport.backend.api.views.bundle.utils import assert_valid_host
from passport.backend.core.builders.blackbox.constants import (
    BLACKBOX_EDITSESSION_OP_DELETE,
    BLACKBOX_EDITSESSION_OP_SELECT,
    BLACKBOX_SESSIONID_DISABLED_STATUS,
    BLACKBOX_SESSIONID_NEED_RESET_STATUS,
    BLACKBOX_SESSIONID_VALID_STATUS,
)
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.loggers.statbox import StatboxLogger
from passport.backend.core.utils.decorators import cached_property
from passport.backend.utils.string import smart_str
from six.moves.urllib.parse import quote_plus

from .forms import LogoutForm


log = logging.getLogger('passport.api.view.bundle.auth')


class BaseBundleEditSessionView(BundleAccountGetterMixin,
                                BundleEditSessionMixin,
                                BundleAccountResponseRendererMixin,
                                BaseBundleView):
    """Базовый класс работы с мультикукой."""

    required_headers = (
        HEADER_CLIENT_HOST,
        HEADER_CLIENT_COOKIE,
        HEADER_CONSUMER_CLIENT_IP,
    )

    required_grants = ['auth_multi.base']

    require_track = False

    track_type = 'authorize'

    @cached_property
    def statbox(self):
        return StatboxLogger(
            uid=self.form_values['uid'],
            action='cookie_edit',
            yandexuid=self.cookies.get('yandexuid'),
        )

    def get_edit_session_operation(self):
        raise NotImplementedError()  # pragma: no cover

    def get_statbox_params(self):
        raise NotImplementedError()  # pragma: no cover

    def check_session_for_new_default(self, cookie_session_info):
        raise NotImplementedError()  # pragma: no cover

    def check_session_for_current_uid(self, cookie_session_info):
        raise NotImplementedError()  # pragma: no cover

    def update_track(self):
        pass

    def fill_response_with_multisession_accounts(self, cookie_session_info):
        self.response_values['accounts'] = self.get_multisession_accounts(cookie_session_info)

    def process_request(self):
        assert_valid_host(self.request.env)

        self.process_basic_form()
        self.uid = self.form_values['uid']
        self.new_default = self.form_values.get('default_uid')

        # Проверим куку
        cookie_session_info = self.check_session_cookie(allow_scholar=True)
        if cookie_session_info.cookie_status == CookieCheckStatus.Invalid:
            raise SessionidInvalidError()
        elif cookie_session_info.cookie_status == CookieCheckStatus.WrongGuard:
            raise SessguardInvalidError()

        self.check_session_for_current_uid(cookie_session_info)
        self.check_session_for_new_default(cookie_session_info)

        self.read_or_create_track(self.track_type)
        self.update_track()

        statbox_data = self.get_statbox_params()
        statbox_data['authid'] = cookie_session_info.authid
        with self.track_transaction.commit_on_error():
            cookies, default_uid = self.edit_session(
                self.get_edit_session_operation(),
                self.uid,
                cookie_session_info,
                statbox_data,
                new_default_uid=self.new_default,
            )
            self.response_values.update(cookies=cookies)
            if default_uid:
                self.response_values.update(default_uid=default_uid)

        self.response_values['track_id'] = self.track_id
        self.fill_response_with_multisession_accounts(cookie_session_info)


class LogoutView(BaseBundleEditSessionView):
    """Разлогин пользователя в мультикуке."""

    basic_form = LogoutForm

    def get_edit_session_operation(self):
        return BLACKBOX_EDITSESSION_OP_DELETE

    def fill_response_with_multisession_accounts(self, cookie_session_info):
        self.response_values.update(
            accounts=self.get_multisession_accounts(
                cookie_session_info,
                exclude_uid=self.uid,
            ),
            logged_out_uids=[self.uid],
        )

    def update_track(self):
        old_retpath = self.track.retpath
        if (
            old_retpath and
            self.account and
            self.account.is_pdd and
            self.account.domain.domain in settings.PDD_DOMAIN_RETPATH_WRAPPER_MAP
        ):
            quoted_old_retpath = quote_plus(smart_str(old_retpath))
            retpath = settings.PDD_DOMAIN_RETPATH_WRAPPER_MAP[self.account.domain.domain] % quoted_old_retpath
            with self.track_transaction.rollback_on_error() as track:
                track.retpath = retpath

    def check_session_for_current_uid(self, cookie_session_info):
        if not is_user_in_multisession(cookie_session_info, self.uid):
            raise UidNotInSessionError()
        user_session_info = user_from_multisession(cookie_session_info, self.uid)
        if user_session_info.get('status') in (
            BLACKBOX_SESSIONID_VALID_STATUS,
            BLACKBOX_SESSIONID_NEED_RESET_STATUS,
            BLACKBOX_SESSIONID_DISABLED_STATUS,
        ):
            self.parse_account(user_session_info, enabled_required=False)

    def check_session_for_new_default(self, cookie_session_info):
        if self.new_default is None:
            return
        if not is_user_in_multisession(cookie_session_info, self.new_default):
            raise UidNotInSessionError()

    def get_statbox_params(self):
        return {
            'mode': 'logout',
            'global': '0',
            'retpath': self.track.retpath,
            'origin': self.track.origin,
            'user_agent': self.user_agent,
            'referer': self.referer,
        }


class ChangeDefaultView(BaseBundleEditSessionView):
    """Смена пользователя по умолчанию в мультикуке."""
    basic_form = UidForm

    def get_edit_session_operation(self):
        return BLACKBOX_EDITSESSION_OP_SELECT

    def check_session_for_current_uid(self, cookie_session_info):
        if is_user_default_in_multisession(cookie_session_info, self.uid):
            raise ActionNotRequiredError()

    def check_session_for_new_default(self, cookie_session_info):
        if not is_user_in_multisession(cookie_session_info, self.uid):
            raise UidNotInSessionError()

    def get_statbox_params(self):
        return {
            'mode': 'change_default',
        }


__all__ = (
    'LogoutView',
    'ChangeDefaultView',
)
