# -*- coding: utf-8 -*-
import logging

from flask import request
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import ValidationFailedError
from passport.backend.api.views.bundle.headers import HEADER_CONSUMER_CLIENT_IP
from passport.backend.core.builders.oauth import (
    get_oauth,
    OAuthPermanentError,
)
from passport.backend.core.conf import settings
from passport.backend.core.utils.decorators import cached_property

from .forms import IssueCodeForAMForm


log = logging.getLogger('passport.api.view.bundle.oauth')


class IssueCodeForAMView(BaseBundleView):
    """
    Обёртка над аналогичной OAuth-ручкой, не даёт возможности
    переопределять опциональные параметры
    """

    basic_form = IssueCodeForAMForm
    required_headers = [
        HEADER_CONSUMER_CLIENT_IP,
    ]
    required_grants = ['auth_oauth.issue_code']

    @cached_property
    def oauth(self):
        return get_oauth()

    def process_request(self):
        self.process_basic_form()
        if self.form_values['client_id'] and self.form_values['client_secret']:
            client_credentials = {
                'client_id': self.form_values['client_id'],
                'client_secret': self.form_values['client_secret'],
            }
        else:
            client_credentials = settings.OAUTH_APPLICATION_AM_XTOKEN

        try:
            rv = self.oauth.issue_authorization_code(
                code_strength='long',
                ttl=settings.AM_CODE_TTL,
                require_activation=False,
                uid=self.form_values['uid'],
                headers=request.headers,
                **client_credentials
            )
            self.response_values.update(
                code=rv['code'],
                expires_in=settings.AM_CODE_TTL,
            )
        except OAuthPermanentError as e:
            raise ValidationFailedError(errors=[e.message])
