# -*- coding: utf-8 -*-
from datetime import datetime
import logging

from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import (
    AccountDisabledError,
    AccountInvalidTypeError,
    AccountNotFoundError,
    AccountUidMismatchError,
    OAuthUnavailableError,
)
from passport.backend.api.views.bundle.headers import HEADER_CONSUMER_CLIENT_IP
from passport.backend.api.views.bundle.mixins import BundleAccountGetterMixin
from passport.backend.core.builders.oauth import get_oauth
from passport.backend.core.conf import settings
from passport.backend.core.exceptions import UnknownUid
from passport.backend.core.logging_utils.loggers.statbox import StatboxLogger
from passport.backend.core.models.account import Account
from passport.backend.core.runner.context_managers import UPDATE
from passport.backend.core.utils.decorators import cached_property

from .forms import RecreateKolonkishTokenForm


log = logging.getLogger('passport.api.view.bundle.oauth')


class RecreateKolonkishTokenView(BaseBundleView, BundleAccountGetterMixin):
    basic_form = RecreateKolonkishTokenForm
    required_headers = [
        HEADER_CONSUMER_CLIENT_IP,
    ]
    required_grants = ['auth_oauth.recreate_kolonkish_token']

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode='recreate_kolonkish_token',
        )

    def issue_oauth_authorization_code(self, kolonkish):
        client_id = settings.OAUTH_APPLICATION_KOLONKA['client_id']
        client_secret = settings.OAUTH_APPLICATION_KOLONKA['client_secret']

        oauth = get_oauth()

        response = oauth.issue_authorization_code(
            code_strength='long',
            client_id=client_id,
            client_secret=client_secret,
            require_activation=False,
            uid=kolonkish.uid,
            by_uid=True,
            ttl=settings.OAUTH_KOLONKA_CODE_TTL,
            device_id=self.form_values['device_id'],
            device_name=self.form_values['device_name'],
            user_ip=self.client_ip,
        )

        if response['status'] == 'error':
            raise OAuthUnavailableError(response['errors'])
        else:
            self.statbox.log(
                action='oauth_code_issued',
                uid=kolonkish.uid,
                client_id=client_id,
                device_id=self.form_values['device_id'],
                device_name=self.form_values['device_name'],
            )
            return response['code']

    def get_kolonkish_account_by_uid(self):
        # Не использую метод get_account_by_uid, потому что он модифицирует self.account
        # В этом контроллере self.account уже занят под другого пользователя вызовом get_account_by_oauth_token.
        # Рефакторинг get_account_by_uid не выглядит простым с первого раза, поэтому продублировал
        # здесь часть логики оттуда.

        response = self.blackbox.userinfo(
            uid=self.form_values['uid'],
            attributes=settings.BLACKBOX_ATTRIBUTES + ('account.creator_uid',),
        )

        try:
            account = Account().parse(response)
        except UnknownUid:
            raise AccountNotFoundError()

        if not account.is_user_enabled:
            raise AccountDisabledError()
        if not account.is_kolonkish:
            raise AccountInvalidTypeError()
        return account

    def check_creator_account_type(self):
        if not (self.account.is_normal or self.account.is_lite or self.account.is_social or self.account.is_pdd):
            raise AccountInvalidTypeError()

    def process_request(self):
        self.process_basic_form()

        self.get_account_from_session_or_oauth_token(required_scope=settings.OAUTH_KOLONKA_KOLONKISH_SCOPE)

        self.check_creator_account_type()

        kolonkish = self.get_kolonkish_account_by_uid()

        if self.account.uid != kolonkish.creator_uid:
            raise AccountUidMismatchError()

        events = {
            'action': 'recreate_kolonkish_token',
            'consumer': self.consumer,
        }
        with UPDATE(kolonkish, self.request.env, events):
            kolonkish.global_logout_datetime = datetime.now()

        code = self.issue_oauth_authorization_code(kolonkish)

        self.response_values.update(code=code)
