# -*- coding: utf-8 -*-
from datetime import datetime
import logging

from passport.backend.api.common.authorization import authorize_oauth
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import (
    AccountInvalidTypeError,
    OAuthUnavailableError,
)
from passport.backend.api.views.bundle.headers import HEADER_CONSUMER_CLIENT_IP
from passport.backend.api.views.bundle.mixins import BundleAccountGetterMixin
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.loggers.statbox import StatboxLogger
from passport.backend.core.runner.context_managers import UPDATE
from passport.backend.core.utils.decorators import cached_property

from .forms import RecreateYambotTokenForm


log = logging.getLogger('passport.api.view.bundle.oauth')


class RecreateYambotTokenView(BaseBundleView, BundleAccountGetterMixin):
    basic_form = RecreateYambotTokenForm
    required_headers = [
        HEADER_CONSUMER_CLIENT_IP,
    ]
    required_grants = ['auth_oauth.recreate_yambot_token']

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode='recreate_yambot_token',
        )

    def issue_oauth_token(self):
        client_id = settings.OAUTH_APPLICATION_YAMB['client_id']
        client_secret = settings.OAUTH_APPLICATION_YAMB['client_secret']
        response = authorize_oauth(
            client_id=client_id,
            client_secret=client_secret,
            env=self.request.env,
            uid=self.account.uid,
        )
        if 'error' in response:
            raise OAuthUnavailableError(response['error'], response.get('description'))
        else:
            self.statbox.log(
                action='token_created',
                uid=self.account.uid,
                client_id=client_id,
            )
            return response['access_token']

    def process_request(self):
        self.process_basic_form()
        self.get_account_by_uid(uid=self.form_values['uid'])

        if not self.account.yambot_alias.alias:
            raise AccountInvalidTypeError()

        events = {
            'action': 'recreate_yambot_token',
            'consumer': self.consumer,
        }
        with UPDATE(self.account, self.request.env, events):
            self.account.global_logout_datetime = datetime.now()

        token = self.issue_oauth_token()

        self.response_values.update(
            oauth_token=token,
        )
