# -*- coding: utf-8 -*-
import logging

from passport.backend.api.common.authorization import (
    AUTHORIZATION_SESSION_POLICY_PERMANENT,
    is_user_default_in_multisession,
    is_user_valid_in_multisession,
    SessionScope,
)
from passport.backend.api.common.common import extract_tld
from passport.backend.api.common.profile.profile import process_env_profile
from passport.backend.api.views.bundle.auth.base import BundleBaseAuthorizationMixin
from passport.backend.api.views.bundle.base import (
    BaseBundleView,
    OAUTH_HEADER_PREFIX,
)
from passport.backend.api.views.bundle.constants import PDD_PARTNER_OAUTH_TOKEN_SCOPE
from passport.backend.api.views.bundle.exceptions import (
    Account2FAEnabledError,
    InvalidTrackStateError,
    OAuthTokenValidationError,
    ValidationFailedError,
)
from passport.backend.api.views.bundle.headers import (
    HEADER_CLIENT_COOKIE,
    HEADER_CLIENT_HOST,
    HEADER_CLIENT_USER_AGENT,
    HEADER_CONSUMER_AUTHORIZATION,
    HEADER_CONSUMER_CLIENT_IP,
)
from passport.backend.api.views.bundle.mixins import (
    BundleAccountGetterMixin,
    BundleAccountResponseRendererMixin,
    BundleAuthenticateMixin,
    BundleCacheResponseToTrackMixin,
    BundleEditSessionMixin,
    BundleFixPDDRetpathMixin,
    BundlePhoneMixin,
    CookieCheckStatus,
)
from passport.backend.api.views.bundle.states import RedirectToPasswordChange
from passport.backend.api.views.bundle.utils import write_phone_to_log
from passport.backend.core import authtypes
from passport.backend.core.builders import blackbox
from passport.backend.core.builders.blackbox.utils import add_phone_arguments
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.loggers.statbox import StatboxLogger
from passport.backend.core.utils.decorators import cached_property

from .forms import OpenAuthorizationForm


log = logging.getLogger('passport.api.view.bundle.oauth')


class OpenAuthorizationView(BundleAccountResponseRendererMixin,
                            BundleCacheResponseToTrackMixin,
                            BundleAccountGetterMixin,
                            BundleBaseAuthorizationMixin,
                            BundleAuthenticateMixin,
                            BundleEditSessionMixin,
                            BundlePhoneMixin,
                            BundleFixPDDRetpathMixin,
                            BaseBundleView):
    """
    Замена perl-ручке `/passport?mode=oauth`
    Подразумевается обработка только ПДД-пользователей по токенам "type=trusted-pdd-partner"
    :doc http://doc.yandex-team.ru/Passport/passport-modes/reference/oauth.xml
    """

    required_headers = [
        HEADER_CONSUMER_AUTHORIZATION,
        HEADER_CLIENT_HOST,
        HEADER_CLIENT_USER_AGENT,
        HEADER_CLIENT_COOKIE,
        HEADER_CONSUMER_CLIENT_IP,
    ]
    required_grants = ['auth_oauth.base']

    basic_form = OpenAuthorizationForm

    track_type = 'authorize'

    sensitive_response_fields = ['cookies']

    required_scope = PDD_PARTNER_OAUTH_TOKEN_SCOPE

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode='any_auth',
            type='oauth',
            user_agent=self.user_agent,
            ip=self.client_ip,
            track_id=self.track_id,
            yandexuid=self.cookies.get('yandexuid'),
        )

    @cached_property
    def oauth_token(self):
        """
        Ожидается заголовок вида 'Authorization: OAuth AaCbw88AAABbH1dU7mlOTTuX1L-ug_7SfB'
        Возвращаем только токен
        """
        auth_header = self.authorization
        if auth_header.lower().startswith(OAUTH_HEADER_PREFIX):
            auth_header = auth_header[len(OAUTH_HEADER_PREFIX):]
        return auth_header.strip()

    @property
    def default_retpath(self):
        passport_tld = extract_tld(self.request.env.host, settings.PASSPORT_TLDS) or settings.PASSPORT_DEFAULT_TLD
        return settings.DEFAULT_PDD_RETPATH_WITHOUT_TLD % passport_tld

    def check_track(self):
        self.check_auth_not_passed()
        if self.check_redirect_state():
            self.make_cached_response()
            raise InvalidTrackStateError()

    def is_account_type_allowed(self):
        """Принимаем только ПДД-аккаунты"""
        return self.account.is_pdd

    def validate_token_and_parse_account(self):
        """
        Если токен пуст, упадем с ошибкой сразу,
        Если токен не пуст - проверим его в ЧЯ,
          * Если токен невалидный, упадем с ошибкой,
          * Если токен ОК - парсим пользователя из ответа ЧЯ
        :raises: ValidationFailedError
        :raises: OAuthTokenValidationError
        """
        if not self.oauth_token:
            log.info('OAuth token is empty')
            raise ValidationFailedError(['oauth_token.empty'])

        bb_kwargs = dict(ip=self.client_ip)
        bb_kwargs = add_phone_arguments(**bb_kwargs)

        bb_response = blackbox.Blackbox().oauth(
            self.oauth_token,
            **bb_kwargs
        )

        status = bb_response['status']
        if status != blackbox.BLACKBOX_OAUTH_VALID_STATUS:
            self.statbox.log(action='failed', error='oauth_token.invalid')
            raise OAuthTokenValidationError()

        scopes = bb_response['oauth']['scope']
        if self.required_scope not in scopes:
            self.statbox.log(action='failed', error='oauth_scope.invalid')
            raise OAuthTokenValidationError()

        self.parse_account(bb_response)

    def try_experiments(self):
        """Никаких экспериментов в этой ручке"""
        return

    def get_validation_method(self):
        """Заглушка. Нужна для вызова check_user_policies"""
        return

    def respond_redirect(self, redirect_state):
        self.fill_response_briefly()
        self.fill_response_with_account(personal_data_required=True)

        self.state = redirect_state
        return redirect_state

    def process_multisession(self):
        is_session_valid = self.session_info.cookie_status == CookieCheckStatus.Valid

        if not is_session_valid:
            self.make_new_session()

        elif not is_user_valid_in_multisession(self.session_info, self.account.uid):
            self.make_new_session()

        # Нужно только поменять default-аккаунт в сессии
        elif not is_user_default_in_multisession(self.session_info, self.account.uid):
            self.response_values['accounts'] = self.get_multisession_accounts(self.session_info)
            self.change_default(
                self.session_info,
                self.account.uid,
            )
            self.fill_response_with_account(personal_data_required=True)

    def fill_response_briefly(self):
        """
        Этот вызов всегда последний в этой ручке.
        Здесь собираем общую информацию в тело ответа
        """
        self.response_values.update(
            uid=self.account.uid,
            login=self.account.login,
            track_id=self.track_id,
            retpath=self.form_values['retpath'],
        )

    def process_request(self):
        self.process_basic_form()

        self.read_or_create_track(self.track_type)
        self.check_track()

        self.validate_token_and_parse_account()
        log.info('Token %s is OK for uid=%d', self.oauth_token, self.account.uid)

        self.statbox.log(action='submitted')

        self.raise_if_otp_enabled(Account2FAEnabledError(), password_like_otp=False)

        # Ручка сейчас работает только для ПДД
        bb_response = self.hosted_domains(self.account.domain.domain)
        self.account.parse(bb_response)

        with self.track_transaction.rollback_on_error() as self.track:
            self.fill_track_with_account_data()

            # В случае с мультикукой, retpath не будет проверяться и исправляться
            self.fix_pdd_retpath()
            # Функция fix_pdd_retpath записывает retpath в трек только если изменила его в form_values
            if not self.form_values['retpath']:
                self.form_values['retpath'] = self.default_retpath

            self.track.retpath = self.form_values['retpath']
            self.track.is_oauth_pdd_authorization = True

            redirect_state = self.check_user_policies()
            if redirect_state is not None:
                if isinstance(redirect_state, RedirectToPasswordChange):
                    self.save_secure_number_in_track()
                return self.respond_redirect(redirect_state)

            self.session_info = self.check_session_cookie(
                dbfields=[],
                attributes=[],
            )

            self.process_multisession()
            self.fill_response_briefly()

    def make_new_session(self):
        """Выписываем новую сессию в ЧЯ"""

        self.set_old_session_track_fields(self.session_info)
        self.track.authorization_session_policy = AUTHORIZATION_SESSION_POLICY_PERMANENT

        self.fill_response_with_account_and_session(
            auth_type=authtypes.AUTH_TYPE_OAUTH,
            cookie_session_info=self.session_info,
            personal_data_required=True,
            session_scope=SessionScope.xsession,
        )

        self.response_values['accounts'] = self.get_multisession_accounts(self.session_info)

        write_phone_to_log(self.account, self.cookies)
        process_env_profile(self.account, track=self.track)
