# -*- coding: utf-8 -*-

from passport.backend.api.common.authorization import SessionScope
from passport.backend.api.common.common import get_surface_for_track
from passport.backend.api.common.ip import get_ip_autonomous_system
from passport.backend.api.common.profile.profile import process_env_profile
from passport.backend.api.views.bundle.auth.password.base import BasePasswordSubmitAuthView
from passport.backend.api.views.bundle.constants import (
    CRED_STATUS_INVALID,
    CRED_STATUS_VALID,
)
from passport.backend.api.views.bundle.exceptions import (
    AccountInvalidTypeError,
    InvalidCSRFTokenError,
    InvalidTrackStateError,
    OAuthTokenValidationError,
)
from passport.backend.api.views.bundle.headers import (
    HEADER_CLIENT_COOKIE,
    HEADER_CLIENT_HOST,
    HEADER_CLIENT_USER_AGENT,
    HEADER_CONSUMER_CLIENT_IP,
)
from passport.backend.api.views.bundle.mixins import (
    BundleAccountSubscribeMixin,
    BundleAuthenticateMixinV2,
    BundleFixPDDRetpathMixin,
)
from passport.backend.api.views.bundle.states import (
    OtpAuthFinished,
    OtpAuthNotReady,
)
from passport.backend.api.views.bundle.utils import (
    assert_valid_host,
    write_phone_to_log,
)
from passport.backend.core import authtypes
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.loggers.statbox import (
    AntifraudLogger,
    StatboxLogger,
)
from passport.backend.core.utils.decorators import cached_property

from .forms import CommitForm


TYPE_OTP = 'otp'
TYPE_X_TOKEN = 'x_token'


class CommitView(BundleAccountSubscribeMixin,
                 BundleAuthenticateMixinV2,
                 BundleFixPDDRetpathMixin,
                 BasePasswordSubmitAuthView):

    required_headers = (
        HEADER_CLIENT_HOST,
        HEADER_CLIENT_COOKIE,
        HEADER_CLIENT_USER_AGENT,
        HEADER_CONSUMER_CLIENT_IP,
    )

    require_track = True

    type = 'otp'
    track_type = 'authorize'

    required_grants = ['otp.auth']

    sensitive_response_fields = ['cookies']

    basic_form = CommitForm
    auth_by_otp = False
    auth_by_x_token = False

    @cached_property
    def statbox(self):
        return StatboxLogger(
            track_id=self.track_id,
            ip=self.client_ip,
            user_agent=self.user_agent,
            origin=self.track.origin,
            mode='any_auth',
            yandexuid=self.cookies.get('yandexuid'),
        )

    @cached_property
    def antifraud_log(self):
        return AntifraudLogger(
            channel='auth',
            sub_channel=settings.ANTIFRAUD_AUTH_SUB_CHANNEL,
            ip=self.client_ip,
            AS=get_ip_autonomous_system(self.client_ip),
            user_agent=self.user_agent,
            origin=self.track.origin,
            external_id='track-{}'.format(self.track_id),
            uid=self.account.uid if self.account else None,
            service_id='login',
        )

    def process_auth(self):
        self.check_track_for_captcha()
        with self.track_transaction.commit_on_error():
            authorization_session_policy = self.form_values.get('policy')
            if authorization_session_policy:
                self.track.authorization_session_policy = authorization_session_policy

            if self.auth_by_x_token:
                self.get_account_by_uid(
                    self.track.uid,
                    check_disabled_on_deletion=True,
                    emails=True,
                )
                if self.account.totp_secret.is_set:
                    raise AccountInvalidTypeError()
                self.fill_track_with_account_data(
                    auth_source=authtypes.AUTH_SOURCE_XTOKEN,
                    password_passed=False,
                )

            else:
                self.blackbox_login(
                    login=self.track.login,
                    password=self.track.otp,
                    retpath=self.track.retpath or '',
                    service=self.load_service_from_track() or '',
                )

                self.fill_track_with_account_data(password_passed=True)

            if self.account.is_pdd and self.track.retpath:
                self.form_values['retpath'] = self.track.retpath
                self.fix_pdd_retpath()
            # Проверим валидность сессионной куки
            # Будем проверять всегда с multisession=yes.
            # Работает и со второй версией куки.
            cookie_session_info = self.check_session_cookie(dbfields=[])
            self.set_old_session_track_fields(cookie_session_info)

            self.statbox.bind_context(uid=self.account.uid)

            auth_method = settings.AUTH_METHOD_MAGIC if self.auth_by_otp else settings.AUTH_METHOD_MAGIC_X_TOKEN
            self.track.auth_method = auth_method

            self.track.surface = get_surface_for_track(self.consumer, action='otp')

            self.fill_response_with_account_and_session(
                cookie_session_info=cookie_session_info,
                # Не проверяем глогаут для входа по otp,
                # т.к. авторизуем по предъявлению логина-пароля в method=login ЧЯ.
                # Решено 18.12.2014
                logout_check_enabled=self.auth_by_x_token,
                # PASSP-10515 Запишем в auth_log факт того, что это "волшебная авторизация"
                otp_magic_passed=self.auth_by_otp,
                personal_data_required=True,
                session_scope=SessionScope.xsession,
                x_token_magic_passed=self.auth_by_x_token,
            )
            write_phone_to_log(self.account, self.cookies)

            self.statbox.log(action='otp_auth_finished')
            self.response_values['retpath'] = self.track.retpath

            # TODO: склеить запросы в базу, в обычной ручке submit тоже
            process_env_profile(self.account, track=self.track)
            self.subscribe_if_allow_and_update_account(self.account)

    def check_csrf_token(self):
        if self.track.csrf_token != self.form_values['csrf_token']:
            raise InvalidCSRFTokenError()

    def process_request(self, *args, **kwargs):
        assert_valid_host(self.request.env)
        self.process_basic_form()
        self.read_track()
        self.response_values.update(track_id=self.track_id)
        self.check_csrf_token()
        self.check_auth_not_passed()
        if not self.track.is_allow_otp_magic:
            raise InvalidTrackStateError()

        cred_status = self.track.cred_status or self.track.x_token_status

        # Проверим, что трек заполнен правильно
        if (
            (cred_status == CRED_STATUS_VALID and not self.track.uid) or
            (self.track.otp and not self.track.login) or
            (not self.track.otp and self.track.login)
        ):
            raise InvalidTrackStateError()

        if cred_status == CRED_STATUS_INVALID and not (self.track.otp and self.track.login):
            raise OAuthTokenValidationError()

        self.state = OtpAuthNotReady()

        # Расставляем приоритеты, если найдены обе возможности
        if self.track.login and self.track.otp:
            self.auth_by_otp = True
            self.statbox.bind_context(type=TYPE_OTP)

        elif self.track.uid and cred_status == CRED_STATUS_VALID:
            self.auth_by_x_token = True
            self.statbox.bind_context(type=TYPE_X_TOKEN)

        if self.auth_by_otp or self.auth_by_x_token:
            self.state = OtpAuthFinished()
            self.process_auth()
