# -*- coding: utf-8 -*-
from passport.backend.api.common.account import is_auth_by_magic_xtoken_allowed
from passport.backend.api.views.bundle.auth.base import BundleBaseAuthorizationMixin
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.constants import (
    CRED_STATUS_INVALID,
    CRED_STATUS_VALID,
)
from passport.backend.api.views.bundle.exceptions import (
    AccountInvalidTypeError,
    InvalidTrackStateError,
    OAuthTokenValidationError,
)
from passport.backend.api.views.bundle.headers import HEADER_CONSUMER_CLIENT_IP
from passport.backend.api.views.bundle.mixins.account import BundleAccountGetterMixin
from passport.backend.core.logging_utils.loggers.statbox import StatboxLogger
from passport.backend.core.utils.decorators import cached_property


class PrepareAuthWithCredView(BaseBundleView, BundleAccountGetterMixin, BundleBaseAuthorizationMixin):
    required_headers = [
        HEADER_CONSUMER_CLIENT_IP,
    ]

    require_track = True

    track_type = 'authorize'

    required_grants = ['auth_by_token.base']

    @cached_property
    def statbox(self):
        return StatboxLogger(
            track_id=self.track.track_id,
            ip=self.client_ip,
        )

    def process_request(self, *args, **kwargs):
        self.read_track()
        if self.track.track_type != self.track_type or not self.track.is_allow_otp_magic:
            raise InvalidTrackStateError()

        with self.track_transaction.commit_on_error():
            try:
                self.get_account_from_session_or_oauth_token(check_disabled_on_deletion=True)
            except OAuthTokenValidationError:
                self.track.x_token_status = CRED_STATUS_INVALID
                self.track.cred_status = CRED_STATUS_INVALID
                self.statbox.log(action='failed', error='oauth_token.invalid')
                raise
            if not is_auth_by_magic_xtoken_allowed(self.account, allow_passwordless=True):
                self.statbox.log(action='failed', error='account.invalid_type')
                raise AccountInvalidTypeError()
            self.assert_allowed_to_get_cookie()
            self.track.x_token_status = CRED_STATUS_VALID
            self.track.cred_status = CRED_STATUS_VALID
            self.track.uid = self.account.uid
            self.track.otp = ''
            self.track.blackbox_login_status = ''
            self.track.blackbox_password_status = ''

        self.statbox.log(action='auth_prepared_by_cred')
