# -*- coding: utf-8 -*-
import logging

from passport.backend.api.views.bundle.auth.base import BundleBaseAuthorizationMixin
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.mixins import (
    BundleAccountGetterMixin,
    BundleAccountPropertiesMixin,
    BundleAccountResponseRendererMixin,
    BundleAuthenticateMixin,
    BundleAuthNotificationsMixin,
    BundleFrodoMixin,
    BundlePasswordValidationMixin,
    BundlePasswordVerificationMethodMixin,
    BundlePhoneMixin,
)
from passport.backend.api.views.bundle.mixins.challenge import BundleChallengeMixin
from passport.backend.core.conf import settings
from passport.backend.core.types.account.account import (
    ACCOUNT_TYPE_LITE,
    ACCOUNT_TYPE_NORMAL,
    ACCOUNT_TYPE_PDD,
)

from .exceptions import (
    DomainNotHostedError,
    PDDInvalidLoginFormatError,
)


log = logging.getLogger('passport.api.view.bundle.auth.password')


class BasePasswordAuthView(BundlePasswordValidationMixin,
                           BundleAuthenticateMixin,
                           BundleAccountResponseRendererMixin,
                           BundleFrodoMixin,
                           BundleAccountGetterMixin,
                           BundleAccountPropertiesMixin,
                           BundleBaseAuthorizationMixin,
                           BaseBundleView):
    """
    Класс для общих методов, связанных с авторизацией
    """
    required_grants = ['auth_password.base']

    def __init__(self, *args, **kwargs):
        super(BasePasswordAuthView, self).__init__()
        self.totp_check_time = None

    def check_track(self):
        """Проверяем все возможные состояния в треке"""
        self.check_auth_not_passed()
        self.check_track_for_captcha()

    def get_frodo_action(self, account):
        # Отправим в ФО соответствующее ситуации сообщение
        if account.password.is_password_flushed_by_admin:
            return 'change_flushed_password'
        elif account.password.is_changing_required:
            return 'change_password_force'
        elif account.is_strong_password_required:
            return 'change_password_to_strong'

        return 'change_password'


class BasePasswordSubmitAuthView(BundlePasswordVerificationMethodMixin,
                                 BundleAuthNotificationsMixin,
                                 BasePasswordAuthView,
                                 BundleChallengeMixin,
                                 BundlePhoneMixin,
                                 ):

    def check_pdd_domain(self, domain):
        """Проверим, обслуживается ли домен"""
        if not domain:
            raise PDDInvalidLoginFormatError()
        bb_response = self.hosted_domains(domain)
        if not bb_response.get('hosted_domains'):
            raise DomainNotHostedError()

    @property
    def is_sms_validation_required(self):
        if self.track.is_change_password_sms_validation_required is None:
            is_account_type_allowed = self.account.type in (ACCOUNT_TYPE_NORMAL, ACCOUNT_TYPE_LITE, ACCOUNT_TYPE_PDD)
            if is_account_type_allowed:
                self.track.is_change_password_sms_validation_required = (
                    not settings.DISABLE_CHANGE_PASSWORD_PHONE_EXPERIMENT and
                    self.is_secure_number_valid_or_missing
                )
            else:
                self.track.is_change_password_sms_validation_required = False
        return self.track.is_change_password_sms_validation_required


__all__ = (
    'BasePasswordAuthView',
    'BasePasswordSubmitAuthView',
)
