# -*- coding: utf-8 -*-

import logging

from passport.backend.api.common.authorization import user_session_scope
from passport.backend.api.common.profile.profile import process_env_profile
from passport.backend.api.views.bundle.constants import CHANGE_PASSWORD_REASON_HACKED
from passport.backend.api.views.bundle.exceptions import (
    ActionNotRequiredError,
    InvalidTrackStateError,
)
from passport.backend.api.views.bundle.headers import (
    HEADER_CLIENT_COOKIE,
    HEADER_CLIENT_HOST,
    HEADER_CLIENT_USER_AGENT,
    HEADER_CONSUMER_CLIENT_IP,
)
from passport.backend.api.views.bundle.mixins import (
    ActiveDirectoryMixin,
    BindRelatedPhonishAccountMixin,
    BundleAccountSubscribeMixin,
    BundlePasswordChangeMixin,
    BundlePhoneMixin,
)
from passport.backend.api.views.bundle.mixins.kolmogor import KolmogorMixin
from passport.backend.api.views.bundle.mixins.push import BundlePushMixin
from passport.backend.api.views.bundle.utils import (
    assert_valid_host,
    write_phone_to_log,
)
from passport.backend.core.conf import settings
from passport.backend.core.counters.change_password_counter import (
    get_per_phone_number_buckets,
    get_per_user_ip_buckets,
)
from passport.backend.core.dbmanager.exceptions import DBError
from passport.backend.core.logging_utils.loggers.statbox import StatboxLogger
from passport.backend.core.runner.context_managers import UPDATE
from passport.backend.core.utils.decorators import cached_property

from .base import BasePasswordAuthView
from .exceptions import AnswerVerificationRequiredError
from .forms import (
    ChangePasswordForm,
    ChangePasswordIntranetForm,
)


log = logging.getLogger('passport.api.view.bundle.auth.password')


class ForcedChangePasswordView(
    KolmogorMixin,
    BundlePushMixin,
    BasePasswordAuthView,
    BundleAccountSubscribeMixin,
    BundlePhoneMixin,
    BindRelatedPhonishAccountMixin,
    BundlePasswordChangeMixin,
    ActiveDirectoryMixin,
):
    """
    Принудительная смена пароля. Если при авторизации было обнаружено что это
    необходимо.
    """

    required_headers = (
        HEADER_CLIENT_COOKIE,
        HEADER_CONSUMER_CLIENT_IP,
        HEADER_CLIENT_USER_AGENT,
        HEADER_CLIENT_HOST,
    )

    require_track = True

    @property
    def basic_form(self):
        if settings.IS_INTRANET:
            return ChangePasswordIntranetForm
        return ChangePasswordForm

    @property
    def forbidden_change_password_with_bad_frodo_karma(self):
        return settings.FORBIDDEN_CHANGE_PASSWORD_WITH_BAD_FRODO_KARMA

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode='change_password_force',
            track_id=self.track_id,
            user_agent=self.user_agent,
            ip=self.client_ip,
            consumer=self.consumer,
        )

    def log_statbox_error(self, errors):
        self.statbox.log(
            action='error',
            errors=','.join(errors),
            password_quality=self.form_values.get('quality'),
        )

    def log_statbox_success(self):
        self.statbox.log(
            action='changed_password',
            password_quality=self.form_values['quality'],
        )

    def check_track(self):
        super(ForcedChangePasswordView, self).check_track()

        if not self.track.is_password_change:
            raise InvalidTrackStateError('Track is not set to change password')

    def check_if_answer_verification_required(self):
        # Если у пользователя не было защищенного телефона
        # или подтвержденный номер не равен текущему защищенному,
        # то пользователь обязан ответить на КВ или успешно заполнить
        # короткую версию анкеты восстановления пароля.
        if (
            (not self.secure_number or self.secure_number != self.saved_number) and
            (not (self.track.is_fuzzy_hint_answer_checked or self.track.is_short_form_factors_checked))
        ):
            raise AnswerVerificationRequiredError()

    def clear_2fa_promo_flag(self):
        """
        Если у пользователя был установлен атрибут-флаг о показе промо-страницы 2ФА,
        сбрасываем его после успешной принудительной смены пароля
        см. PASSP-11501
        """
        if self.account.show_2fa_promo:
            self.account.show_2fa_promo = False
            return True

    def _process_request(self):
        """Этот вызов может завершиться ошибкой. В этом случае, надо записать ошибку в statbox"""
        # Возможна ошибка валидации
        self.process_basic_form()
        # Сохраняется текущий защищенный телефон в трек
        self.save_secure_number_in_track()

        self.frodo_status = None
        cookie_session_info = self.check_session_cookie(dbfields=[])

        with self.track_transaction.rollback_on_error():
            self.set_old_session_track_fields(cookie_session_info)
            self.validate_and_set_new_password()
            self.fill_response_with_account_and_session(
                cookie_session_info=cookie_session_info,
                session_scope=user_session_scope(cookie_session_info, self.account.uid),
            )

        write_phone_to_log(self.account, self.cookies)
        self.try_bind_related_phonish_account()

        self.send_notifications(self.frodo_status, self.secure_number_or_none)

    def validate_and_set_new_password(self):
        if settings.IS_INTRANET:
            self._validate_and_set_new_password_to_active_directory()
        else:
            self._validate_and_set_new_password_to_db_and_bind_phone_with_sms_2fa()

    def _validate_and_set_new_password_to_active_directory(self):
        # Выполняем только базовые проверки: историю паролей проверит AD. Полностью на откуп AD не отдаём,
        # так как там недостаточно понятные ошибки.
        password, _ = self.validate_password()
        self.change_password_in_active_directory(
            old_password=self.form_values['current_password'],
            new_password=password,
            is_forced=True,
        )

    def _validate_and_set_new_password_to_db_and_bind_phone_with_sms_2fa(self):
        password, quality = self.validate_password(
            required_check_password_history=True,
            is_strong_policy=self.account.is_strong_password_required,
            emails=set([email.address for email in self.account.emails.native]),
        )

        frodo_action = self.get_frodo_action(self.account)
        self.frodo_status = self.frodo_check_spammer(
            frodo_action,
            old_password_quality=self.account.password.quality,
            account_karma=str(self.account.karma.value) if self.account.karma else None,
        )

        self.assert_account_not_compromised(self.frodo_status)

        if self.is_sms_validation_required:
            self.check_if_answer_verification_required()

            phone_binder = self.build_bind_securest_possible_phone(
                account=self.account,
                phone_number=self.saved_number,
            )
            phone_binder.submit()

        with UPDATE(
            self.account,
            self.request.env,
            {'action': 'change_password', 'consumer': self.consumer},
        ):
            self.change_password(
                password,
                quality,
            )
            if self.is_sms_validation_required:
                phone_binder.commit()
                if (
                    phone_binder.is_binding_secure() or
                    phone_binder.is_starting_replacement() or
                    phone_binder.is_replacing()
                ):
                    get_per_phone_number_buckets().incr(self.saved_number.e164)
            self.subscribe_if_allow(self.account)
            is_2fa_promo_cleared = self.clear_2fa_promo_flag()
            process_env_profile(self.account, track=self.track)

            if self.is_enable_sms_2fa_required():
                self.account.sms_2fa_on = True

        if self.is_sms_validation_required:
            self.response_values.update(self.securest_possible_phone_binding_response(phone_binder))

            if phone_binder.is_updating_old_secure():
                self.statbox.log(
                    number=self.saved_number.masked_format_for_statbox,
                    operation='update',
                )
            elif phone_binder.is_binding_secure():
                self.statbox.log(
                    number=self.saved_number.masked_format_for_statbox,
                    operation='save',
                )
            elif (
                phone_binder.is_starting_replacement() or
                phone_binder.is_replacing()
            ):
                self.statbox.log(
                    number=self.saved_number.masked_format_for_statbox,
                    operation='replace',
                )
            elif phone_binder.is_binding_simple():
                self.statbox.log(
                    number=self.saved_number.masked_format_for_statbox,
                    operation='replace',
                    error='phone.isnt_saved',
                )
                self.statbox.log(
                    number=self.saved_number.masked_format_for_statbox,
                    operation='save',
                )

            # PASSP-11501 После привязки нового защищенного телефона ИЛИ
            # подтверждения существующего защищенного телефона, покажем
            # пользователю промо 2ФА
            if (
                is_2fa_promo_cleared and
                (
                    phone_binder.is_binding_secure() or
                    phone_binder.is_updating_old_secure()
                )
            ):
                track_id = self.build_enable_2fa_track(
                    self.track.phone_confirmation_phone_number,
                    skip_phone_check=True,
                )
                if track_id:
                    self.response_values['enable_2fa_track_id'] = track_id

        if self.is_sms_validation_required:
            phone_binder.after_commit()
            self.statbox.bind(**self.phone_binding_stats(phone_binder))

        if is_2fa_promo_cleared:
            self.statbox.bind(show_2fa_promo=True)

    def build_enable_2fa_track(self, phone_number, skip_phone_check=True):
        """
        Если у пользователя установлен атрибут-флаг о показе промо-страницы 2ФА,
        после привязки защищенного телефона, нужно предложить ему включить 2ФА.
        Здесь подготовим трек со всей информацией о привязанном телефоне
        """
        new_track = self.track_manager.create('authorize', self.consumer)
        with self.track_manager.transaction(track=new_track).rollback_on_error():
            new_track.uid = self.account.uid
            new_track.is_it_otp_enable = True
            new_track.phone_confirmation_is_confirmed = True
            new_track.phone_confirmation_phone_number = phone_number
            new_track.has_secure_phone_number = True
            new_track.secure_phone_number = phone_number
            new_track.can_use_secure_number_for_password_validation = True
            # FIXME: Надо переименовать это поле, т.к. уже есть более одного
            # случая, когда надо пропускать проверку телефона при включении 2FA
            new_track.is_otp_restore_passed = skip_phone_check

            return new_track.track_id

    def securest_possible_phone_binding_response(self, phone_binder):
        response = super(ForcedChangePasswordView, self).securest_possible_phone_binding_response(phone_binder)
        response.update(is_yasms_errors_when_replacing_phone=False)
        return response

    def phone_binding_stats(self, phone_binder):
        phone_binding_stats = self.securest_possible_phone_binding_response(phone_binder)
        phone_binding_stats.pop('secure_phone_pending_until', None)
        return phone_binding_stats

    def process_request(self):
        # Запишем, что форма отправлялась
        self.statbox.log(
            action='submitted',
        )

        assert_valid_host(self.request.env)

        self.read_track()
        self.check_track()

        self.get_account_from_track(check_disabled_on_deletion=True, emails=True, need_phones=True)

        # Если у нас нет оснований для принудительной смены пароля, то ничего не делаем.
        if not (
            self.account.is_strong_password_required or
            self.account.password.is_changing_required or
            self.track.is_force_change_password
        ):
            raise ActionNotRequiredError()

        # Если включена 2FA, то смена пароля теряет смысл.
        if self.account.totp_secret.is_set:
            raise ActionNotRequiredError()

        self.statbox.bind_context(
            uid=self.account.uid,
        )

        if self.is_sms_validation_required:
            self.assert_secure_phone(self.secure_number_or_none)

        elif (
            self.track.change_password_reason == CHANGE_PASSWORD_REASON_HACKED and
            not (self.track.is_fuzzy_hint_answer_checked or self.track.is_short_form_factors_checked)
        ):
            raise AnswerVerificationRequiredError()

        try:
            self._process_request()
        except DBError as ex:
            self.log_statbox_error(['backend.database_failed'])
            raise ex

        if self.track.change_password_reason == CHANGE_PASSWORD_REASON_HACKED:
            get_per_user_ip_buckets().incr(self.client_ip)

        # Все прошло успешно - запишем этот факт
        self.log_statbox_success()
        self.send_account_modification_push(event_name='changed_password')


__all__ = ('ForcedChangePasswordView',)
