# -*- coding: utf-8 -*-
import logging

from passport.backend.api.common.account import (
    build_default_person_info,
    fill_person_from_args,
)
from passport.backend.api.common.profile.profile import process_env_profile
from passport.backend.api.forms.register import PHONE_VALIDATION_METHOD
from passport.backend.api.views.bundle.exceptions import (
    ActionNotRequiredError,
    EulaIsNotAcceptedError,
    InvalidTrackStateError,
    UserNotVerifiedError,
)
from passport.backend.api.views.bundle.headers import (
    HEADER_CLIENT_COOKIE,
    HEADER_CLIENT_HOST,
    HEADER_CLIENT_USER_AGENT,
    HEADER_CONSUMER_CLIENT_IP,
)
from passport.backend.api.views.bundle.mixins import (
    BindRelatedPhonishAccountMixin,
    BundleAccountGetterMixin,
    BundleAccountSubscribeMixin,
    BundlePasswordChangeMixin,
    BundlePhoneMixin,
)
from passport.backend.api.views.bundle.mixins.kolmogor import KolmogorMixin
from passport.backend.api.views.bundle.mixins.push import BundlePushMixin
from passport.backend.api.views.bundle.register.exceptions import RegistrationSmsSendPerIPLimitExceededError
from passport.backend.api.views.bundle.utils import (
    assert_valid_host,
    make_hint_question,
    write_phone_to_log,
)
from passport.backend.api.views.register import (
    incr_registration_sms_per_ip_counter,
    is_registration_sms_per_ip_counter_exceed,
)
from passport.backend.core.logging_utils.loggers.statbox import StatboxLogger
from passport.backend.core.runner.context_managers import UPDATE
from passport.backend.core.types.phone_number.phone_number import PhoneNumber
from passport.backend.core.utils.decorators import cached_property

from .base import BasePasswordAuthView
from .forms import (
    CompleteAutoregistered,
    Hint,
)


log = logging.getLogger('passport.api.view.bundle.auth.password')


class CompleteAutoregisteredView(
    BundlePushMixin,
    KolmogorMixin,
    BasePasswordAuthView,
    BundleAccountGetterMixin,
    BundleAccountSubscribeMixin,
    BundlePasswordChangeMixin,
    BindRelatedPhonishAccountMixin,
    BundlePhoneMixin,
):

    required_headers = (
        HEADER_CONSUMER_CLIENT_IP,
        HEADER_CLIENT_USER_AGENT,
        HEADER_CLIENT_COOKIE,
        HEADER_CLIENT_HOST,
    )

    require_track = True

    basic_form = CompleteAutoregistered

    def process_request(self):
        question = None

        assert_valid_host(self.request.env)
        self.read_track()
        track = self.track
        self.check_auth_not_passed()

        self.process_basic_form()

        is_validated_via_phone = self.form_values['validation_method'] == PHONE_VALIDATION_METHOD
        if (
                (is_validated_via_phone and not self.is_phone_confirmed_in_track(allow_by_flash_call=True)) or
                (not is_validated_via_phone and not self.is_captcha_passed)
        ):
            raise UserNotVerifiedError()

        eula_accepted = self.form_values['eula_accepted']
        if not eula_accepted:
            raise EulaIsNotAcceptedError()

        # Проверяем, что пришли сюда из начальной ручки авторизации
        if not track.is_complete_autoregistered:
            raise InvalidTrackStateError('Track is not set to complete autoregistered user')

        self.get_account_from_track(check_disabled_on_deletion=True, need_phones=True)
        if not self.account.is_incomplete_autoregistered:
            raise ActionNotRequiredError()

        self.statbox.bind_context(uid=self.account.uid)

        if is_validated_via_phone and is_registration_sms_per_ip_counter_exceed(
            self.request.env,
            track,
            mode='complete_autoregistered',
        ):
            raise RegistrationSmsSendPerIPLimitExceededError()

        password, quality = self.validate_password(
            # В эту ручку можно попасть двумя способами:
            # 1) при авторизации по паролю
            # 2) при авторизации по одноразовой саппортской ссылке
            # всегда хеш пароля берем из аккаунта
            old_password_hash=self.account.password.serialized_encrypted_password,
            required_check_password_history=True,
            is_strong_policy=self.account.is_strong_password_required,
            emails=set([email.address for email in self.account.emails.native]),
        )

        frodo_action = self.get_frodo_action(self.account)

        cookie_session_info = self.check_session_cookie(dbfields=[])

        if is_validated_via_phone:
            phone_number = PhoneNumber.parse(track.phone_confirmation_phone_number)

            save_phone = self.build_save_secure_phone(
                phone_number=phone_number,
                language=self.form_values['language'],
                # Допускаем любые типы аккаунтов, главное что во время вызова
                # commit аккаунт уже будет портальным.
                check_account_type_on_submit=False,
            )
            save_phone.submit()

        with self.track_transaction.rollback_on_error():
            self.set_old_session_track_fields(cookie_session_info)
            if not is_validated_via_phone:
                self.form_values.update(self.process_form(Hint, self.all_values))
                question = make_hint_question(
                    question=self.form_values['question'],
                    question_id=self.form_values['question_id'],
                    display_language=self.form_values['display_language'],
                )

            events = {
                'action': 'complete_autoregistered',
                'consumer': self.consumer,
            }
            with UPDATE(self.account, self.request.env, events):
                old_password_quality = self.account.password.quality
                self.change_password(
                    password,
                    quality,
                )
                if question:
                    self.account.hint.question = question
                    self.account.hint.answer = self.form_values['answer']
                self.subscribe_if_allow(self.account)
                fill_person_from_args(
                    self.account.person,
                    self.form_values,
                    build_default_person_info(self.client_ip),
                )
                process_env_profile(self.account, track=self.track)

                if is_validated_via_phone:
                    save_phone.commit()

            if is_validated_via_phone:
                save_phone.after_commit()

            self.frodo_check_spammer(
                frodo_action,
                increment_counters=True,
                old_password_quality=old_password_quality or None,
            )
            self.fill_response_with_account_and_session(cookie_session_info=cookie_session_info)
            write_phone_to_log(self.account, self.cookies)

            self.send_account_modification_push(event_name='changed_password')

            self.try_bind_related_phonish_account()

            if is_validated_via_phone:
                incr_registration_sms_per_ip_counter(self.request.env)

    @cached_property
    def statbox(self):
        return StatboxLogger(
            track_id=self.track_id,
            ip=self.client_ip,
            user_agent=self.user_agent,
            mode='any_auth',
            consumer=self.consumer,
        )


__all__ = (
    'CompleteAutoregisteredView',
)
