# -*- coding: utf-8 -*-
import logging

from passport.backend.api.common.account import (
    build_default_person_info,
    fill_person_from_args,
)
from passport.backend.api.common.profile.profile import process_env_profile
from passport.backend.api.views.bundle.exceptions import (
    AccountWithoutPasswordError,
    ActionNotRequiredError,
    EulaIsNotAcceptedError,
    InvalidTrackStateError,
)
from passport.backend.api.views.bundle.headers import (
    HEADER_CLIENT_COOKIE,
    HEADER_CLIENT_HOST,
    HEADER_CLIENT_USER_AGENT,
    HEADER_CONSUMER_CLIENT_IP,
)
from passport.backend.api.views.bundle.mixins import (
    BindRelatedPhonishAccountMixin,
    BundleAccountGetterMixin,
    BundleAccountSubscribeMixin,
    BundlePasswordChangeMixin,
    BundlePhoneMixin,
)
from passport.backend.api.views.bundle.mixins.account import UserMetaDataMixin
from passport.backend.api.views.bundle.mixins.common import BundleCleanWebMixin
from passport.backend.api.views.bundle.mixins.kolmogor import KolmogorMixin
from passport.backend.api.views.bundle.mixins.mail import MailMixin
from passport.backend.api.views.bundle.mixins.push import BundlePushMixin
from passport.backend.api.views.bundle.utils import (
    assert_valid_host,
    make_hint_question,
    write_phone_to_log,
)
from passport.backend.core.runner.context_managers import UPDATE

from .base import BasePasswordAuthView
from .forms import (
    AccountCompletePdd,
    AccountCompletePddPassword,
    AccountCompleteWorkspacePdd,
    AccountCompleteWorkspacePddPassword,
)


log = logging.getLogger('passport.api.view.bundle.auth.password')


class CompletePddView(
    MailMixin,
    UserMetaDataMixin,
    BundlePushMixin,
    KolmogorMixin,
    BasePasswordAuthView,
    BundleAccountGetterMixin,
    BundleAccountSubscribeMixin,
    BundlePasswordChangeMixin,
    BundlePhoneMixin,
    BindRelatedPhonishAccountMixin,
    BundleCleanWebMixin,
):
    require_track = True
    required_headers = (
        HEADER_CONSUMER_CLIENT_IP,
        HEADER_CLIENT_USER_AGENT,
        HEADER_CLIENT_COOKIE,
        HEADER_CLIENT_HOST,
    )

    def determine_basic_form(self, is_workspace_user, with_new_password):
        if is_workspace_user and with_new_password:
            return AccountCompleteWorkspacePddPassword
        elif is_workspace_user:
            return AccountCompleteWorkspacePdd
        elif with_new_password:
            return AccountCompletePddPassword
        else:
            return AccountCompletePdd

    def process_request(self):
        assert_valid_host(self.request.env)
        self.read_track()
        track = self.track
        self.check_track()

        self.get_account_from_track(
            check_disabled_on_deletion=True,
            emails=True,
            need_phones=True,
        )
        if not self.account.is_pdd or self.account.is_complete_pdd:
            raise ActionNotRequiredError()

        is_workspace_user = self.account.is_pdd_workspace_user
        if track.is_complete_pdd_with_password or track.is_complete_pdd and track.have_password:
            self.basic_form = self.determine_basic_form(
                is_workspace_user,
                track.is_complete_pdd_with_password,
            )
        elif track.is_complete_pdd and not track.have_password:
            # Случай ПДД-пользователей, для которых не предусмотрено заведение пароля
            raise AccountWithoutPasswordError()
        else:
            raise InvalidTrackStateError()

        self.process_basic_form()
        self.clean_web_check_form_values()

        if not self.form_values['eula_accepted']:
            raise EulaIsNotAcceptedError()

        if track.is_complete_pdd_with_password:
            password, quality = self.validate_password(
                required_check_password_history=True,
                is_strong_policy=self.account.is_strong_password_required,
                emails=set([email.address for email in self.account.emails.native]),
            )

        cookie_session_info = self.check_session_cookie(dbfields=[])
        with self.track_transaction.rollback_on_error():
            self.set_old_session_track_fields(cookie_session_info)

        with UPDATE(
            self.account,
            self.request.env,
            {
                'action': 'complete_pdd',
                'consumer': self.consumer,
            },
        ):
            # персональные данные
            self.account.is_pdd_agreement_accepted = self.form_values['eula_accepted']
            if not is_workspace_user:
                question = make_hint_question(
                    question=self.form_values['question'],
                    question_id=self.form_values['question_id'],
                    display_language=self.form_values['display_language'],
                )
                self.account.hint.question = question
                self.account.hint.answer = self.form_values['answer']

            ignored_args = set()
            if is_workspace_user:
                # Для пользователей Коннекта правильный язык выставляется заранее
                ignored_args.add('language')

            fill_person_from_args(
                self.account.person,
                self.form_values,
                default_values=build_default_person_info(self.client_ip),
                ignored_args=ignored_args,
            )

            # пароль
            if track.is_complete_pdd_with_password:
                old_password_quality = self.account.password.quality
                self.change_password(
                    password,
                    quality,
                )
            self.clear_password_options()
            self.subscribe_if_allow(self.account)

        # авторизационные профили
        process_env_profile(self.account, track=self.track)

        frodo_kwargs = dict()
        if track.is_complete_pdd_with_password:
            frodo_kwargs.update({'old_password_quality': old_password_quality})

        self.frodo_check_spammer(
            action='complete_pdd',
            increment_counters=True,
            **frodo_kwargs
        )

        # Завершаем трек
        with self.track_transaction.rollback_on_error():
            self.fill_response_with_account_and_session(cookie_session_info=cookie_session_info)

        write_phone_to_log(self.account, self.cookies)
        self.send_account_modification_push(event_name='changed_password')
        self.send_account_modification_mail(event_name='changed_password')
        self.try_bind_related_phonish_account()


__all__ = (
    'CompletePddView',
)
