# -*- coding: utf-8 -*-

from passport.backend.api.common.authorization import (
    get_session_policy_by_ttl,
    SessionScope,
)
from passport.backend.api.common.common import create_csrf_token
from passport.backend.api.common.ip import get_ip_autonomous_system
from passport.backend.api.common.processes import PROCESS_ACCOUNT_DELETE_V2
from passport.backend.api.common.profile.profile import process_env_profile
from passport.backend.api.views.bundle.auth.base import BundleBaseAuthorizationMixin
from passport.backend.api.views.bundle.auth.password.forms import (
    ConfirmCommitMagicForm,
    ConfirmCommitPasswordForm,
    ConfirmSubmitForm,
)
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import (
    AccountWithoutPasswordError,
    ActionNotRequiredError,
    InvalidCSRFTokenError,
    InvalidTrackStateError,
    PasswordNotMatchedError,
    SecondStepRequired,
)
from passport.backend.api.views.bundle.headers import AUTHORIZATION_HEADERS
from passport.backend.api.views.bundle.mixins import (
    BundleAccountGetterMixin,
    BundleAccountPropertiesMixin,
    BundleAccountResponseRendererMixin,
    BundleAssertCaptchaMixin,
    BundleAuthenticateMixinV2,
    BundleCacheResponseToTrackMixin,
    BundlePasswordChangeMixin,
    BundlePasswordVerificationMethodMixin,
    BundlePhoneMixin,
)
from passport.backend.api.views.bundle.states import (
    OtpAuthNotReady,
    RedirectToPasswordChange,
    RfcTotp,
)
from passport.backend.api.views.bundle.utils import write_phone_to_log
from passport.backend.core.builders.blackbox.constants import BLACKBOX_SECOND_STEP_RFC_TOTP
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.loggers import StatboxLogger
from passport.backend.core.logging_utils.loggers.statbox import AntifraudLogger
from passport.backend.core.models.password import get_sha256_hash
from passport.backend.core.types.login.login import normalize_login
from passport.backend.core.utils.decorators import cached_property

from .exceptions import AccountChangedError


class BaseConfirmView(BaseBundleView,
                      BundleAccountGetterMixin,
                      BundleAccountResponseRendererMixin,
                      BundleBaseAuthorizationMixin,
                      BundleAssertCaptchaMixin):
    require_track = True
    required_headers = AUTHORIZATION_HEADERS
    required_grants = ['auth_password.confirm']
    type = 'password'
    track_type = 'authorize'
    allowed_processes = [PROCESS_ACCOUNT_DELETE_V2]

    @cached_property
    def statbox(self):
        return StatboxLogger(
            consumer=self.consumer,
            track_id=self.track_id,
            ip=self.client_ip,
            user_agent=self.user_agent,
            mode='confirm_password',
            yandexuid=self.cookies.get('yandexuid'),
        )

    @cached_property
    def antifraud_log(self):
        return AntifraudLogger(
            channel='auth',
            sub_channel=settings.ANTIFRAUD_AUTH_SUB_CHANNEL,
            ip=self.client_ip,
            AS=get_ip_autonomous_system(self.client_ip),
            user_agent=self.user_agent,
            external_id='track-{}'.format(self.track_id),
            uid=self.account.uid if self.account else None,
            service_id='login',
        )

    def process_request(self):
        self.process_basic_form()

        self.read_track()
        self.response_values['track_id'] = self.track_id
        self.check_auth_not_passed()

        self._process_request()

    def _process_request(self):
        raise NotImplementedError()  # pragma: no cover


class ConfirmSubmit(BaseConfirmView,
                    BundleAccountPropertiesMixin):
    basic_form = ConfirmSubmitForm

    def _process_request(self):
        self.get_pinned_account_from_session()
        self.statbox.bind_context(uid=self.account.uid)
        self.fill_response_with_account(personal_data_required=True)

        if not self.account.have_password:
            raise AccountWithoutPasswordError()

        if not self.is_password_verification_required():
            raise ActionNotRequiredError()

        with self.track_transaction.commit_on_error() as track:
            track.csrf_token = create_csrf_token()
            track.is_allow_otp_magic = True
            track.login_required_for_magic = self.account.login
            track.retpath = self.form_values['retpath']
            track.origin = self.form_values['origin']
            track.authorization_session_policy = get_session_policy_by_ttl(self.session_info.ttl)

        self.response_values.update(csrf_token=track.csrf_token)

        # Капчу проверяем в самом конце: тут брутфорс ещё невозможен,
        # а фронтенду при капче нужны информация об аккаунте и csrf-токен
        self.check_track_for_captcha()

        self.statbox.log(
            action='submitted',
            origin=self.form_values['origin'],
        )


class BaseConfirmCommitView(BaseConfirmView,
                            BundlePasswordChangeMixin,
                            BundlePasswordVerificationMethodMixin,
                            BundlePhoneMixin,
                            BundleAuthenticateMixinV2):

    def process_auth_by_password(self, password, method, force_use_cache):
        try:
            self.blackbox_login(
                uid=self.get_tracked_uid(),
                password=password,
                need_phones=True,
                force_use_cache=force_use_cache,
            )
            redirect_state = None
        except SecondStepRequired as e:
            if BLACKBOX_SECOND_STEP_RFC_TOTP in e.allowed_second_steps:
                redirect_state = RfcTotp()
            else:
                raise  # pragma: no cover

            self.prepare_track_for_second_step(e.allowed_second_steps)

        self.statbox.bind_context(uid=self.account.uid)
        self.fill_track_with_account_data(password_passed=True)
        self.fill_response_with_account(personal_data_required=True)

        redirect_state = redirect_state or self.check_user_policies()
        if redirect_state is not None:
            self.track.password_hash = get_sha256_hash(password)
            if isinstance(redirect_state, RedirectToPasswordChange):
                self.save_secure_number_in_track()
            return redirect_state

        self.track.allow_authorization = True
        self.track.dont_change_default_uid = True
        self.track.session_scope = str(SessionScope.xsession)
        write_phone_to_log(self.account, self.cookies)
        process_env_profile(self.account, track=self.track)

        self.statbox.log(
            action='confirmed',
            method=method,
            origin=self.track.origin,
        )


class ConfirmCommitPassword(BundleCacheResponseToTrackMixin, BaseConfirmCommitView):
    basic_form = ConfirmCommitPasswordForm

    def _process_request(self):
        self.check_track_for_captcha()
        with self.track_transaction.commit_on_error():
            self.state = self.process_auth_by_password(
                password=self.form_values['password'],
                method='password',
                force_use_cache=False,
            )


class ConfirmCommitMagic(BaseConfirmCommitView):
    basic_form = ConfirmCommitMagicForm

    def check_csrf_token(self):
        # FIXME: копипаста из views.bundle.auth.otp.commit
        if self.track.csrf_token != self.form_values['csrf_token']:
            raise InvalidCSRFTokenError()

    def _process_request(self):
        # FIXME: копипаста из views.bundle.auth.otp.commit
        self.check_csrf_token()
        self.check_track_for_captcha()
        if not self.track.is_allow_otp_magic:
            raise InvalidTrackStateError()
        if (self.track.otp and not self.track.login) or (not self.track.otp and self.track.login):
            raise InvalidTrackStateError()
        elif not self.track.otp or not self.track.login:
            self.state = OtpAuthNotReady()
        elif (
            self.track.login_required_for_magic and
            normalize_login(self.track.login_required_for_magic) != normalize_login(self.track.login)
        ):
            raise AccountChangedError()
        else:
            with self.track_transaction.commit_on_error():
                try:
                    self.state = self.process_auth_by_password(
                        password=self.track.otp,
                        method='magic',
                        force_use_cache=True,
                    )
                except PasswordNotMatchedError:
                    # Сбрасываем логин и otp в треке, чтобы можно было повторно провести
                    # магическую авторизацию с другим паролем (пином)
                    self.track.login = None
                    self.track.otp = None
                    raise
