# -*- coding: utf-8 -*-
from datetime import datetime
import logging

from passport.backend.api.common.account import is_auth_by_sms_secure_enough_for_account
from passport.backend.api.common.authorization import SessionScope
from passport.backend.api.common.common import (
    extract_tld,
    is_second_step_allowed,
    should_ignore_per_ip_counters,
)
from passport.backend.api.common.phone import get_accounts_with_actual_secure_phone
from passport.backend.api.common.profile.profile import process_env_profile
from passport.backend.api.views.bundle.auth.password.start import (
    METHOD_MAGIC,
    METHOD_PASSWORD,
)
from passport.backend.api.views.bundle.constants import (
    CRED_STATUS_INVALID,
    CRED_STATUS_VALID,
)
from passport.backend.api.views.bundle.exceptions import (
    AccountInvalidTypeError,
    ActionImpossibleError,
    ActionNotRequiredError,
    CodeInvalidError,
    CompareNotMatchedError,
    EmailConfirmationsLimitExceededError,
    InvalidCSRFTokenError,
    InvalidTrackStateError,
    MagicLinkNotSentError,
    MagicLinkSecretNotMatch,
    OAuthTokenValidationError,
    RateLimitExceedError,
    SecondStepRequired,
)
from passport.backend.api.views.bundle.headers import HEADER_CONSUMER_CLIENT_IP
from passport.backend.api.views.bundle.mixins import (
    BundleAccountSubscribeMixin,
    BundleAuthenticateMixinV2,
    BundleCacheResponseToTrackMixin,
    BundleMagicLinkMixin,
    BundleNeophonishMixin,
    UserMetaDataMixin,
)
from passport.backend.api.views.bundle.mixins.challenge import (
    MobileDeviceStatus,
    MobileProfile,
)
from passport.backend.api.views.bundle.phone.exceptions import PhoneNotConfirmedError
from passport.backend.api.views.bundle.states import (
    EmailCode,
    OtpAuthFinished,
    OtpAuthNotReady,
    RedirectToForcedLiteCompletion,
    RedirectToPasswordChange,
    RfcTotp,
)
from passport.backend.api.views.bundle.utils import (
    assert_valid_host,
    write_phone_to_log,
)
from passport.backend.core import authtypes
from passport.backend.core.builders.blackbox.constants import (
    BLACKBOX_SECOND_STEP_EMAIL_CODE,
    BLACKBOX_SECOND_STEP_RFC_TOTP,
)
from passport.backend.core.builders.historydb_api import get_historydb_api
from passport.backend.core.compare import (
    compare_names,
    STRING_FACTOR_INEXACT_MATCH,
)
from passport.backend.core.conf import settings
from passport.backend.core.counters import login_restore_counter
from passport.backend.core.logging_utils.loggers.statbox import StatboxLogger
from passport.backend.core.mailer.utils import (
    MailInfo,
    make_email_context,
    send_mail_for_account,
)
from passport.backend.core.models.account import get_preferred_language
from passport.backend.core.models.password import get_sha256_hash
from passport.backend.core.types.mobile_device_info import get_app_id_from_track
from passport.backend.core.utils.decorators import cached_property
from passport.backend.utils.common import (
    format_code_by_3,
    generate_random_code,
    normalize_code,
)
from passport.backend.utils.time import (
    datetime_to_integer_unixtime,
    get_unixtime,
    unixtime_to_datetime,
)

from .base import BaseMultiStepAuthView
from .forms import (
    MultiStepAuthAfterLoginRestore2Form,
    MultiStepAuthAfterLoginRestoreForm,
    MultiStepEmailCodeCommitForm,
    MultiStepMagicForm,
    MultiStepMagicLinkConfirmForm,
    MultiStepMagicLinkConfirmRegistrationForm,
    MultiStepMagicLinkInfoForm,
    MultiStepMagicLinkSendForm,
    MultiStepPasswordForm,
    SEND_TO_EMAIL,
)


log = logging.getLogger('passport.api.view.bundle.auth.password.multistep')


class BaseMultiStepCommitView(BaseMultiStepAuthView,
                              BundleAccountSubscribeMixin,
                              BundleAuthenticateMixinV2):
    require_track = True
    track_type = 'authorize'
    method = None
    auth_by_otp = False
    auth_by_x_token = False
    antifraud_auth_type = None

    def check_track(self):
        super(BaseMultiStepCommitView, self).check_track()
        if (
            (self.track.track_type != self.track_type) or
            not self.track.user_entered_login
        ):
            raise InvalidTrackStateError()

    def process_auth_by_password(self, password, force_use_cache, add_avatar_with_secret_url=False):
        password_passed = False
        redirect_state = None

        if self.auth_by_x_token:
            self.get_account_by_uid(
                self.track.uid,
                check_disabled_on_deletion=True,
                emails=True,
                need_phones=True,
            )
            if self.account.totp_secret.is_set:
                raise AccountInvalidTypeError()
            self.track.session_scope = str(SessionScope.xsession)
        else:
            try:
                self.blackbox_login(
                    allow_scholar=self.track.allow_scholar,
                    force_use_cache=force_use_cache,
                    login=self.track.user_entered_login,
                    need_phones=True,
                    password=password,
                    retpath=self.track.retpath or '',
                    service=self.load_service_from_track() or '',
                )
            except SecondStepRequired as e:
                if BLACKBOX_SECOND_STEP_RFC_TOTP in e.allowed_second_steps:
                    redirect_state = RfcTotp()
                elif BLACKBOX_SECOND_STEP_EMAIL_CODE in e.allowed_second_steps:
                    redirect_state = EmailCode()
                else:
                    raise  # pragma: no cover

                self.prepare_track_for_second_step(e.allowed_second_steps)

            password_passed = True

        self.statbox.bind_context(
            uid=self.account.uid,
            login=self.account.login,
            is_2fa_enabled=self.account.totp_secret.is_set or None,
        )

        # Запишем в трек все, что стало известно о пользователе
        self.fill_track_with_account_data(password_passed=password_passed)
        self._update_track_on_successful_auth()

        # Если это вебвью АМ, то заполним мобильный профиль и прикинемся мобилкой
        mobile_profile = None
        if self.track.x_token_client_id or self.track.device_id:
            mobile_profile = MobileProfile()
            mobile_profile.device_status = MobileDeviceStatus.unknown  # чтобы не делать тяжёлый запрос в OAuth
            mobile_profile.device_name = self.track.device_name
            mobile_profile.app_id = self.track.device_application
            mobile_profile.password_source = None

        # Проверить пользователя на "законченность" (не требуется ли смены пароля
        # или заполнения анкеты с персональной информацией) и на требование челленжа перед авторизацией
        redirect_state = (
            redirect_state or
            self.check_user_policies() or
            self.show_challenge_if_necessary(
                allow_new_challenge=True,
                auth_by_x_token=self.auth_by_x_token,
                mobile_profile=mobile_profile,
            )
        )

        if redirect_state is not None:
            if self.method == METHOD_PASSWORD:
                self.track.password_hash = get_sha256_hash(password)
            if isinstance(redirect_state, RedirectToPasswordChange):
                self.save_secure_number_in_track()

            # Отдадим фронту флаг, есть ли у лайта средство восстановления
            if isinstance(redirect_state, RedirectToForcedLiteCompletion):
                self.response_values['has_recovery_method'] = self.check_has_recovery_method()

            self.state = redirect_state
            self.fill_response_with_account(personal_data_required=True, account_info_required=True, add_avatar_with_secret_url=add_avatar_with_secret_url)
            return

        self.track.allow_authorization = True
        write_phone_to_log(self.account, self.cookies)
        process_env_profile(self.account, track=self.track)
        self.subscribe_if_allow_and_update_account(self.account)

    def _process_request(self):
        raise NotImplementedError()  # pragma: no cover

    def _update_track_on_successful_auth(self):
        raise NotImplementedError()  # pragma: no cover

    def process_request(self):
        assert_valid_host(self.request.env)
        if self.basic_form is not None:
            self.process_basic_form()

        self.read_track()
        self.response_values.update(track_id=self.track_id)
        # Проверим, что в треке не сохранено ошибочное состояние (от прошлого запроса)
        self.check_track()

        with self.track_transaction.commit_on_error():
            self._process_request()

        self.statbox.log(action='passed')


class MultiStepCommitPasswordView(BundleCacheResponseToTrackMixin, BaseMultiStepCommitView):
    basic_form = MultiStepPasswordForm
    method = METHOD_PASSWORD
    statbox_type = 'multi_step_password'
    antifraud_auth_type = 'multi_step_password'

    def _update_track_on_successful_auth(self):
        if self.account.totp_secret.is_set:
            self.track.auth_method = settings.AUTH_METHOD_OTP
        else:
            self.track.auth_method = settings.AUTH_METHOD_PASSWORD

    def _process_request(self):
        self.process_auth_by_password(
            password=self.form_values['password'],
            force_use_cache=False,
            add_avatar_with_secret_url=settings.ISSUE_AVATARS_SECRETS,
        )


class MultiStepCommitMagicView(BaseMultiStepCommitView):
    basic_form = MultiStepMagicForm
    method = METHOD_MAGIC
    statbox_type = 'multi_step_magic'
    antifraud_auth_type = 'multi_step_magic'

    def check_csrf_token(self):
        if self.track.csrf_token != self.form_values['csrf_token']:
            raise InvalidCSRFTokenError()

    def _update_track_on_successful_auth(self):
        if self.auth_by_otp:
            self.track.is_otp_magic_passed = True
            self.track.auth_method = settings.AUTH_METHOD_MAGIC
        elif self.auth_by_x_token:
            self.track.is_x_token_magic_passed = True
            self.track.auth_method = settings.AUTH_METHOD_MAGIC_X_TOKEN
            self.track.auth_source = authtypes.AUTH_SOURCE_XTOKEN

    def _process_request(self):
        self.check_csrf_token()
        if not self.track.is_allow_otp_magic:
            raise InvalidTrackStateError()

        cred_status = self.track.cred_status or self.track.x_token_status

        # Проверим, что трек заполнен правильно
        if (
            (cred_status == CRED_STATUS_VALID and not self.track.uid) or
            (self.track.otp and not self.track.login)
        ):
            raise InvalidTrackStateError()

        if cred_status == CRED_STATUS_INVALID and not (self.track.otp and self.track.login):
            raise OAuthTokenValidationError()

        # Расставляем приоритеты, если найдены обе возможности
        if self.track.login and self.track.otp:
            self.auth_by_otp = True

        elif self.track.uid and cred_status == CRED_STATUS_VALID:
            self.auth_by_x_token = True

        if self.auth_by_otp or self.auth_by_x_token:
            self.state = OtpAuthFinished()
            self.process_auth_by_password(
                password=self.track.otp,
                force_use_cache=True,
            )
        else:
            self.state = OtpAuthNotReady()
            if self.track.correct_2fa_picture is not None:
                self.response_values.update(
                    is_2fa_picture_expired=get_unixtime() > int(self.track.correct_2fa_picture_expires_at),
                )


class MultiStepCommitSmsCodeView(BundleCacheResponseToTrackMixin, BaseMultiStepCommitView):
    statbox_type = 'multi_step_sms_code'
    antifraud_auth_type = 'multi_step_sms_code'

    def check_track(self):
        super(MultiStepCommitSmsCodeView, self).check_track()
        if settings.AUTH_METHOD_SMS_CODE not in self.track.allowed_auth_methods:
            # TODO: унести подобную проверку в базовый класс через 3 часа после выкладки PASSP-26598 в прод
            raise InvalidTrackStateError()

    def _process_request(self):
        self.get_account_from_track(need_phones=True)
        self.statbox.bind_context(
            uid=self.account.uid,
            login=self.account.login,
        )

        if not self.is_secure_phone_confirmed_in_track(allow_by_call=False):
            raise PhoneNotConfirmedError()

        self.fill_track_with_account_data(password_passed=False)
        self.track.auth_method = settings.AUTH_METHOD_SMS_CODE
        self.track.session_scope = str(SessionScope.xsession)

        redirect_state = self.check_user_policies() or self.show_challenge_if_necessary(allow_new_challenge=True)
        if redirect_state is not None:
            self.state = redirect_state
            self.fill_response_with_account(personal_data_required=True, account_info_required=True)
            return

        self.track.allow_authorization = True
        self.track.allow_set_xtoken_trusted = True
        write_phone_to_log(self.account, self.cookies)
        process_env_profile(self.account, track=self.track)


class MultiStepAuthAfterLoginRestoreViewBase(BundleCacheResponseToTrackMixin, BaseMultiStepCommitView, BundleNeophonishMixin):
    statbox_type = 'multi_step_after_login_restore'

    def check_track(self):
        # не проверяем тип трека и наличие user_defined_login
        self.check_auth_not_passed()
        self.check_track_for_captcha()

    def check_global_counters(self):
        # Те же счётчики, что и в восстановлении логина
        if login_restore_counter.get_per_phone_buckets().hit_limit(self.account.phones.secure.number.digital):
            raise RateLimitExceedError()

        if should_ignore_per_ip_counters(get_app_id_from_track(self.track), self.account.phones.secure.number.e164):
            log.debug('login_restore_per_ip_limit_counter ignored due to YANGO trusted phone code')
            return
        if login_restore_counter.get_per_ip_buckets().hit_limit_by_ip(self.client_ip):
            raise RateLimitExceedError()

    def increase_global_counters(self):
        # Те же счётчики, что и в восстановлении логина
        login_restore_counter.get_per_phone_buckets().incr(self.account.phones.secure.number.digital)
        login_restore_counter.get_per_ip_buckets().incr(self.client_ip)


class MultiStepAuthAfterLoginRestoreView(MultiStepAuthAfterLoginRestoreViewBase):
    basic_form = MultiStepAuthAfterLoginRestoreForm

    def _process_request(self):
        self.get_account_by_uid(uid=self.form_values['uid'], need_phones=True)
        self.statbox.bind_context(
            uid=self.account.uid,
            login=self.account.login,
        )

        if not (
            is_auth_by_sms_secure_enough_for_account(self.account) and
            (self.account.is_neophonish or settings.ALLOW_AUTH_AFTER_LOGIN_RESTORE_FOR_ALL)
        ):
            raise AccountInvalidTypeError()

        if not self.is_secure_phone_confirmed_in_track(allow_by_call=False):
            raise PhoneNotConfirmedError()

        self.check_global_counters()
        firstname_factor, lastname_factor = compare_names(
            orig_names=[self.account.person.firstname, self.account.person.lastname],
            supplied_names=[self.form_values['firstname'], self.form_values['lastname']]
        )
        if firstname_factor < STRING_FACTOR_INEXACT_MATCH or lastname_factor < STRING_FACTOR_INEXACT_MATCH:
            self.increase_global_counters()
            raise CompareNotMatchedError()
        self.fill_track_with_account_data(password_passed=False)
        self.track.session_scope = str(SessionScope.xsession)

        redirect_state = self.check_user_policies()
        if redirect_state is not None:
            self.state = redirect_state
            self.fill_response_with_account(personal_data_required=True, account_info_required=True)
            return

        self.track.allow_authorization = True
        self.track.allow_set_xtoken_trusted = True
        write_phone_to_log(self.account, self.cookies)
        process_env_profile(self.account, track=self.track)

        language = get_preferred_language(account=self.account)
        location = self.get_location(language)
        browser = self.get_browser()
        self.send_neophonish_auth_notification_to_messenger(location=location, device_name=browser)


class MultiStepAuthAfterSuggestByPhone(MultiStepAuthAfterLoginRestoreViewBase):
    basic_form = MultiStepAuthAfterLoginRestore2Form
    statbox_type = 'multi_step_after_suggest_by_phone'

    def is_lastauth_fresh_enough(self):
        lastauth = get_historydb_api().lastauth(uid=self.account.uid).get('timestamp')
        return (
            lastauth is not None and
            datetime.now() - unixtime_to_datetime(lastauth) <= settings.MAX_LASTAUTH_AGE_FOR_AUTH_SUGGEST
        )

    def did_phone_change_owner(self):
        accounts_with_actual_secure_phone = get_accounts_with_actual_secure_phone(
            accounts=[self.account],
            request_id=self.request.env.request_id,
        )
        return len(accounts_with_actual_secure_phone) == 0

    def _process_request(self):
        if not settings.USE_NEW_SUGGEST_BY_PHONE:
            log.warning('New suggest by phone is disabled by settings')
            raise ActionImpossibleError()
        self.get_account_by_uid(
            uid=self.form_values['uid'],
            need_phones=True,
            emails=True,
        )
        self.statbox.bind_context(
            uid=self.account.uid,
            login=self.account.login,
        )

        if not is_auth_by_sms_secure_enough_for_account(self.account):
            raise AccountInvalidTypeError()

        if not self.is_secure_phone_confirmed_in_track(allow_by_call=False):
            raise PhoneNotConfirmedError()

        if not self.is_lastauth_fresh_enough():
            log.warning('Lastauth is not fresh: forbidding auth')
            raise ActionImpossibleError()

        if settings.USE_PHONE_SQUATTER and self.did_phone_change_owner():
            if settings.PHONE_SQUATTER_DRY_RUN:
                log.debug('Phone number\'s owner changed, but allowing auth due to dry run')
            else:
                log.warning('Phone number\'s owner changed: forbidding auth')
                raise ActionImpossibleError()

        self.fill_track_with_account_data(password_passed=False)
        self.track.session_scope = str(SessionScope.xsession)

        redirect_state = self.check_user_policies() or self.show_challenge_if_necessary(allow_new_challenge=True)
        if redirect_state is not None:
            self.state = redirect_state
            self.fill_response_with_account(
                personal_data_required=True,
                account_info_required=True,
            )
            return

        self.track.allow_authorization = True
        self.track.allow_set_xtoken_trusted = True


class MultiStepEmailCodeBaseView(BaseMultiStepAuthView):
    require_track = True
    statbox_type = 'multi_step_email_code'
    step = None

    def assert_track_ok(self):
        self.check_auth_not_passed()
        if not is_second_step_allowed(self.track, BLACKBOX_SECOND_STEP_EMAIL_CODE):
            raise InvalidTrackStateError()

    def process_request(self):
        self.statbox.bind_context(step=self.step)
        self.read_track()
        self.response_values.update(track_id=self.track_id)
        self.assert_track_ok()

        self.get_account_from_track(emails=True)
        self.statbox.bind_context(
            uid=self.account.uid,
            login=self.account.login,
        )

        if not (self.account.emails.default and self.account.emails.default.is_native):
            raise AccountInvalidTypeError()

        self.process()
        self.statbox.log(action='passed')

    def process(self):
        raise NotImplementedError()  # pragma: no cover


class MultiStepEmailCodeSubmitView(MultiStepEmailCodeBaseView):
    step = 'submit'

    def send_email(self, language, code):
        phrases = settings.translations.NOTIFICATIONS[language]
        user_tld = extract_tld(self.request.env.host, settings.PASSPORT_TLDS) or settings.PASSPORT_DEFAULT_TLD

        context = make_email_context(
            language=language,
            account=self.account,
            context={
                'TLD': user_tld,
                'SHORT_CODE': code,
                'email': self.account.emails.default.address,
                'feedback_key': None,
            },
        )
        info = MailInfo(
            subject=phrases['email_code_sent.subject'],
            from_=phrases['email_sender_display_name'],
            tld=user_tld,
        )
        send_mail_for_account(
            'mail/email_code_sent.html',
            info,
            context,
            self.account,
            send_to_external=False,
            send_to_native=True,
        )

    def process(self):
        email = self.account.emails.default.address
        language = get_preferred_language(account=self.account)

        with self.track_transaction.rollback_on_error() as _track:
            if not self.track.email_confirmation_code or email != self.track.email_confirmation_address:
                code = generate_random_code(settings.EMAIL_VALIDATOR_SHORT_CODE_LENGTH)
                self.track.email_confirmation_address = email
                self.track.email_confirmation_code = code
            else:
                code = self.track.email_confirmation_code

            formatted_code = format_code_by_3(
                code,
                settings.EMAIL_VALIDATOR_SHORT_CODE_DELIMITER,
            )

            self.send_email(
                language=language,
                code=formatted_code,
            )


class MultiStepEmailCodeCommitView(BundleCacheResponseToTrackMixin, MultiStepEmailCodeBaseView):
    step = 'commit'
    basic_form = MultiStepEmailCodeCommitForm

    def assert_track_ok(self):
        super(MultiStepEmailCodeCommitView, self).assert_track_ok()
        if not self.track.email_confirmation_code:
            raise InvalidTrackStateError()

    def check_code(self, code):
        if self.track.email_confirmation_checks_count.get(default=0) >= settings.ALLOWED_EMAIL_SHORT_CODE_FAILED_CHECK_COUNT:
            raise EmailConfirmationsLimitExceededError()

        self.track.email_confirmation_checks_count.incr()
        delimiter = settings.EMAIL_VALIDATOR_SHORT_CODE_DELIMITER
        expected_code = normalize_code(self.track.email_confirmation_code, delimiter)
        actual_code = normalize_code(code, delimiter)
        if expected_code != actual_code:
            raise CodeInvalidError()

    def process(self):
        self.process_basic_form()

        if self.account.emails.default.address != self.track.email_confirmation_address:
            raise InvalidTrackStateError()

        with self.track_transaction.commit_on_error():
            self.check_code(self.form_values['code'])

            redirect_state = self.check_user_policies()
            if redirect_state is not None:
                self.state = redirect_state
                self.fill_response_with_account(personal_data_required=True, account_info_required=True)
                return

            self.track.password_verification_passed_at = get_unixtime()
            self.track.allow_authorization = True
            self.track.is_session_restricted = True
            write_phone_to_log(self.account, self.cookies)
            process_env_profile(self.account, track=self.track)


class MultiStepMagicLinkBaseView(BaseMultiStepAuthView, UserMetaDataMixin, BundleMagicLinkMixin):
    require_track = True
    track_type = 'authorize'
    type = 'multi_step_magic_link'
    allowed_track_types = [track_type]

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode='any_auth',
            track_id=self.track_id,
            ip=self.client_ip,
            user_agent=self.user_agent,
            origin=self.track.origin if self.track else self.form_values.get('origin'),
            yandexuid=self.cookies.get('yandexuid'),
            consumer=self.consumer,
            type='multi_step_magic_link',
        )

    def assert_magic_link_allowed(self):
        if not self.is_magic_link_allowed:
            self.report_login_error_antifraud(['magic_link_not_allowed'], uid=self.account.uid)
            raise AccountInvalidTypeError()

    def check_track_state(self, check_sent=True, check_uid=True, check_code=True, check_expired=True):
        if (
            self.track.track_type not in self.allowed_track_types or
            (check_uid and not self.track.uid) or
            (check_code and not self.track.magic_link_code)
        ):
            raise InvalidTrackStateError()

        if check_sent:
            self.assert_magic_link_sent()
        self.assert_magic_link_not_confirmed()
        self.check_auth_not_passed()
        self.assert_magic_link_not_invalidated()
        if check_expired:
            self.assert_magic_link_not_expired()

    def check_auth_and_get_account_if_required(self, need_account=True, **bb_kwargs):
        if self.track.require_auth_for_magic_link_confirm and self.track.uid:
            self.get_account_from_session_or_oauth_token(multisession_uid=int(self.track.uid), **bb_kwargs)
        elif need_account:
            self.get_account_from_track(**bb_kwargs)


class MultiStepMagicLinkSendView(MultiStepMagicLinkBaseView):
    basic_form = MultiStepMagicLinkSendForm

    def process_request(self):
        self.statbox.bind_context(step='submit')
        self.process_basic_form()
        self.read_track()
        self.check_track_state(check_sent=False, check_expired=False, check_code=False)

        self.get_account_from_track(emails=True)
        self.statbox.bind_context(uid=self.account.uid)
        self.check_magic_link_send_counters()
        self.check_track_confirm_counters()

        self.assert_magic_link_allowed()

        email = self.account.emails.default.address
        language = get_preferred_language(account=self.account)
        location = self.get_location(language)
        browser = self.get_browser()

        with self.track_transaction.rollback_on_error() as track:
            new_link_generated = False
            if not self.track.magic_link_secret or self.should_renew_magic_link_secret:
                track.magic_link_secret = self.make_magic_link_secret()
                time_now = datetime_to_integer_unixtime(datetime.now())
                track.magic_link_start_time = time_now
                track.magic_link_start_location = location
                track.magic_link_start_browser = browser
                track.magic_link_sent_time = time_now
                track.magic_link_code = generate_random_code(settings.MAGIC_LINK_CODE_LENGTH)
                new_link_generated = True

            track.magic_link_send_to = SEND_TO_EMAIL
            self.send_auth_magic_link_to_email(
                email,
                secret=track.magic_link_secret,
                is_new_link_generated=new_link_generated,
                device_name=browser,
            )

        self.incr_magic_link_send_counters()
        self.response_values.update(code=self.track.magic_link_code)


class MultiStepMagicLinkConfirmView(MultiStepMagicLinkBaseView):
    required_headers = (
        HEADER_CONSUMER_CLIENT_IP,
    )
    basic_form = MultiStepMagicLinkConfirmForm

    def process_request(self):
        self.process_basic_form()
        self.read_track()
        self.response_values.update(track_id=self.track_id)
        self.check_track_state(check_code=False)

        self.check_auth_and_get_account_if_required(emails=True)
        self.statbox.bind_context(
            step='confirm_link',
            uid=self.account.uid,
            send_to=self.track.magic_link_send_to,
        )
        self.check_track_confirm_counters()
        self.assert_magic_link_allowed()

        with self.track_transaction.commit_on_error():
            self.track.magic_link_confirms_count.incr()

            uid = self.form_values['secret']['uid']
            secret = self.form_values['secret']['secret']
            if self.track.magic_link_secret != secret or self.account.uid != uid:
                self.statbox.log(error='magic_link.invalid')
                raise MagicLinkSecretNotMatch()

            self.track.magic_link_confirm_time = datetime_to_integer_unixtime(datetime.now())
            self.track.auth_method = settings.AUTH_METHOD_MAGIC_LINK
            self.statbox.log(status='ok')

            self.send_auth_confirm_message_to_email(
                email=self.account.emails.default.address,
            )

        if self.form_values['redirect'] and self.track.retpath:
            self.response_values.update(
                retpath=self.track.retpath,
            )


class MultiStepMagicLinkConfirmRegistrationView(MultiStepMagicLinkBaseView):
    # Ручка относится к процессу регистрации, но лежит тут, так как неразрывно связана с другими ручками для magic_link
    required_headers = (
        HEADER_CONSUMER_CLIENT_IP,
    )
    allowed_track_types = ['register']
    basic_form = MultiStepMagicLinkConfirmRegistrationForm

    def process_request(self):
        self.process_basic_form()
        self.read_track()
        self.response_values.update(track_id=self.track_id)
        self.check_track_state(check_uid=False, check_code=False)

        self.statbox.bind_context(
            step='confirm_link',
            send_to=self.track.magic_link_send_to,
        )
        self.check_track_confirm_counters()
        self.assert_magic_link_allowed()

        with self.track_transaction.commit_on_error():
            self.track.magic_link_confirms_count.incr()

            if self.track.magic_link_secret != self.form_values['secret']:
                self.statbox.log(error='magic_link.invalid')
                self.report_login_error_antifraud(['magic_link_invalid'])
                raise MagicLinkSecretNotMatch()

            self.track.magic_link_confirm_time = datetime_to_integer_unixtime(datetime.now())

        self.statbox.log(status='ok')

        if self.form_values['redirect'] and self.track.retpath:
            self.response_values.update(
                retpath=self.track.retpath,
            )


class MultiStepMagicLinkInvalidateView(MultiStepMagicLinkBaseView):
    required_headers = (
        HEADER_CONSUMER_CLIENT_IP,
    )
    allowed_track_types = ['authorize', 'register']

    def process_request(self):
        self.read_track()
        self.check_track_state(check_uid=False, check_code=False)
        if self.track.uid:
            self.get_account_from_track()
            self.statbox.bind_context(uid=self.account.uid)

        with self.track_transaction.rollback_on_error() as track:
            track.magic_link_invalidate_time = datetime_to_integer_unixtime(datetime.now())

        self.statbox.log(
            action='invalidate',
            start_time=self.track.magic_link_start_time,
            sent_time=self.track.magic_link_sent_time,
        )


class MultiStepMagicLinkStatusView(BundleCacheResponseToTrackMixin, MultiStepMagicLinkBaseView):
    antifraud_auth_type = 'multi_step_magic_link_status'

    def process_request(self):
        self.read_track()

        magic_link_confirmed = False
        self.response_values.update(track_id=self.track_id)

        try:
            self.check_track_for_captcha(log_fail_to_statbox=False)
            self.check_track_state()
        except ActionNotRequiredError:
            magic_link_confirmed = True
            self.get_account_from_track(emails=True, need_phones=True)
            self.statbox.bind_context(
                step='confirmed',
                uid=self.account.uid,
            )

            with self.track_transaction.commit_on_error():
                self.fill_track_with_account_data(password_passed=False)
                self.track.session_scope = str(SessionScope.xsession)

                redirect_state = self.check_user_policies() or self.show_challenge_if_necessary(allow_new_challenge=True)

                if redirect_state is not None:
                    self.state = redirect_state
                    self.fill_response_with_account(personal_data_required=True, account_info_required=True)
                    self.response_values.update(magic_link_confirmed=magic_link_confirmed)
                    return

                self.track.allow_authorization = True
                write_phone_to_log(self.account, self.cookies)
                process_env_profile(self.account, track=self.track)
                self.statbox.log(
                    action='passed',
                    start_time=self.track.magic_link_start_time,
                    sent_time=self.track.magic_link_sent_time,
                    confirm_time=self.track.magic_link_confirm_time,
                    confirms_count=self.track.magic_link_confirms_count.get(default=0),
                )

        if not self.track.magic_link_sent_time:
            raise MagicLinkNotSentError()

        self.response_values.update(magic_link_confirmed=magic_link_confirmed)


class MultiStepMagicLinkInfoView(MultiStepMagicLinkBaseView):
    allowed_track_types = ['authorize', 'register']
    basic_form = MultiStepMagicLinkInfoForm

    def process_request(self):
        self.process_basic_form()

        self.read_track()
        self.check_track_state(check_uid=False, check_code=False)
        self.check_auth_and_get_account_if_required(need_account=bool(self.track.uid), emails=True)

        self.assert_magic_link_allowed()
        self.response_values.update(
            start_time=self.track.magic_link_start_time,
            location=self.track.magic_link_start_location,
            browser=self.track.magic_link_start_browser,
        )
        if self.track.magic_link_code:
            self.response_values.update(
                code=self.track.magic_link_code,
            )
        if self.account:
            # авторизационный процесс
            self.response_values.update(
                login=self.account.login,
                display_name=self.account.person.display_name.name,
                avatar_url=settings.GET_AVATAR_URL % (
                    self.account.person.default_avatar,
                    self.form_values['avatar_size'],
                ),
            )
        else:
            # регистрационный процесс
            self.response_values.update(
                login=self.track.user_entered_login,
            )
