# -*- coding: utf-8 -*-
from datetime import datetime
import logging
import random
import uuid

from passport.backend.api.common.account import (
    get_allowed_auth_methods,
    get_masked_magic_link_email,
)
from passport.backend.api.common.common import (
    create_csrf_token,
    get_surface_for_track,
    looks_like_yandex_email,
)
from passport.backend.api.common.pdd import does_domain_belong_to_pdd
from passport.backend.api.common.social_api import try_get_social_profiles_for_auth
from passport.backend.api.common.suggest import get_countries_suggest
from passport.backend.api.forms import DeviceInfoForm
from passport.backend.api.views.bundle.exceptions import (
    AccountDisabledOnDeletionError,
    AccountNotFoundError,
    ActionImpossibleError,
    ValidationFailedError,
)
from passport.backend.api.views.bundle.mixins import (
    BundleDeviceInfoMixin,
    BundleFederalMixin,
    BundleFixPDDRetpathMixin,
    BundlePushMixin,
    KolmogorMixin,
)
from passport.backend.api.views.bundle.phone.helpers import dump_number
from passport.backend.api.views.bundle.utils import assert_valid_host
from passport.backend.core import validators
from passport.backend.core.am_pushes.subscription_manager import (
    DeviceIdFilter,
    get_pushes_subscription_manager,
)
from passport.backend.core.builders.blackbox.exceptions import BlackboxInvalidParamsError
from passport.backend.core.builders.kolmogor import BaseKolmogorError
from passport.backend.core.conf import settings
from passport.backend.core.cookies.cookie_l import (
    CookieL,
    CookieLUnpackError,
)
from passport.backend.core.eav_type_mapping import ALIAS_NAME_TO_TYPE
from passport.backend.core.geobase import Region
from passport.backend.core.models.domain import Domain
from passport.backend.core.types.account.account import ACCOUNT_TYPE_FEDERAL
from passport.backend.core.types.login.login import login_is_scholar
from passport.backend.core.utils.decorators import cached_property
from passport.backend.core.utils.experiments import is_experiment_enabled_by_phone
from passport.backend.utils.time import (
    datetime_to_integer_unixtime,
    get_unixtime,
)

from .base import BaseMultiStepAuthView
from .forms import MultiStepStartForm


log = logging.getLogger('passport.api.view.bundle.auth.password.multistep')


class MultiStepStartView(
    BaseMultiStepAuthView,
    BundleFixPDDRetpathMixin,
    BundleDeviceInfoMixin,
    BundleFederalMixin,
    BundlePushMixin,
    KolmogorMixin,
):
    require_track = False
    basic_form = MultiStepStartForm
    statbox_type = 'multi_step_start'

    @cached_property
    def pushes_subscription_manager(self):
        return get_pushes_subscription_manager(
            uid=self.account.uid,
            push_service='2fa',
            event='2fa_pictures',
        )

    @cached_property
    def yakey_2fa_pictures_shown_counter(self):
        return self.build_counter(
            keyspace=settings.YAKEY_2FA_PICTURES_SHOWN_KEYSPACE,
            name=settings.YAKEY_2FA_PICTURES_SHOWN_COUNTER % self.account.uid,
            limit=settings.COUNTERS[settings.YAKEY_2FA_PICTURES_SHOWN_COUNTER],
        )

    @staticmethod
    def generate_cloud_token():
        return 'cl-%s' % uuid.uuid4().get_hex()

    def put_mobile_params_to_track(self, track):
        # Если домик вызывается в вебвью АМ - надо сохранить в трек и мобильные параметры
        track.avatar_size = self.form_values['avatar_size']
        track.captcha_scale_factor = self.form_values['captcha_scale_factor']
        track.display_language = self.form_values['display_language']
        track.language = self.form_values['display_language']
        track.cloud_token = self.form_values['cloud_token'] or self.generate_cloud_token()
        track.payment_auth_retpath = self.form_values['payment_auth_retpath']
        track.x_token_client_id = self.form_values['x_token_client_id']
        track.x_token_client_secret = self.form_values['x_token_client_secret']
        if self.form_values['client_id']:
            track.client_id = self.form_values['client_id']
            track.client_secret = self.form_values['client_secret']

    def put_user_agent_and_ip_params_to_track(self, track):
        user_agent_info = self.request.env.user_agent_info
        track.browser_id = settings.BROWSER_ENCODE.get(user_agent_info.get('BrowserName')) or None
        track.os_family_id = settings.OS_FAMILY_ENCODE.get(user_agent_info.get('OSFamily')) or None

        track.region_id = Region(ip=self.client_ip).id

    def are_2fa_pictures_available(self):
        if not self.account.totp_secret.is_set:
            log.debug('Cannot show 2fa pictures: 2fa not enabled')
            return False
        elif not self.account.totp_secret.yakey_device_ids:
            log.debug('Cannot show 2fa pictures: no yakey_device_ids')
            return False
        elif not self.pushes_subscription_manager.has_yakey_compatible_subscriptions():
            log.debug('Cannot show 2fa pictures: no YaKey compatible push subscriptions')
            return False

        filtered_subscriptions = self.pushes_subscription_manager.filter_subscriptions(
            subscriptions=self.pushes_subscription_manager.yakey_compatible_subscriptions,
            filters=[
                DeviceIdFilter(self.account.totp_secret.yakey_device_ids),
            ],
        )
        if not filtered_subscriptions:
            log.debug('Cannot show 2fa pictures: no push subscriptions with trusted device_ids')
            return False

        return True

    def get_2fa_pictures_available_count(self):
        # Не используем метод из миксина, так как нам важен не только факт непревышения лимита,
        # но и текущее значение счётчика
        counter = self.yakey_2fa_pictures_shown_counter
        try:
            values = self.kolmogor.get(
                counter.key_space,
                [settings.YAKEY_2FA_PICTURES_DENY_FLAG_COUNTER, counter.name],
            )
            if values[settings.YAKEY_2FA_PICTURES_DENY_FLAG_COUNTER]:
                log.debug('Cannot show 2fa pictures: user denied')
                return 0
            counter_value = values[counter.name]
            available_count = counter.limit - counter_value
            if available_count == 0:
                log.debug('Cannot show 2fa pictures: limit exceeded')
            return available_count
        except BaseKolmogorError as e:
            log.debug('Cannot show 2fa pictures: kolmogor failed: %s %s', e.__class__.__name__, e)
            return 0


    def fill_track_with_pictures_and_send_push(self):
        all_2fa_pictures = random.sample(
            range(settings.YAKEY_2FA_PICTURES_TOTAL_COUNT),
            settings.YAKEY_2FA_PICTURES_COUNT,
        )
        self.track.correct_2fa_picture = random.choice(all_2fa_pictures)
        self.track.correct_2fa_picture_expires_at = get_unixtime() + settings.YAKEY_2FA_PICTURES_TTL

        self.send_push_with_pictures(
            pictures=all_2fa_pictures,
            expires_at=self.track.correct_2fa_picture_expires_at,
        )

    def create_and_fill_track(self, track_type, login=None, allowed_auth_methods=None):
        self.create_track(track_type)

        with self.track_transaction.rollback_on_error() as track:
            self.put_mobile_params_to_track(track)

            track.retpath = self.form_values['retpath']
            # Для ПДД проверим retpath (и поправим при необходимости)
            if self.account and self.account.is_pdd:
                self.fix_pdd_retpath()

            track.origin = self.form_values['origin']
            track.service = self.form_values['service'].slug if self.form_values['service'] else None

            if track_type == 'authorize':
                track.user_entered_login = login
                track.fretpath = self.form_values['fretpath']
                track.clean = self.form_values['clean']
                if allowed_auth_methods:
                    track.allowed_auth_methods = allowed_auth_methods
                    if {settings.AUTH_METHOD_MAGIC, settings.AUTH_METHOD_MAGIC_X_TOKEN} & set(allowed_auth_methods):
                        track.uid = self.account.uid
                        track.is_allow_otp_magic = True
                        track.csrf_token = create_csrf_token()
                        self.response_values.update(csrf_token=track.csrf_token)

                        if self.form_values['with_2fa_pictures'] and self.are_2fa_pictures_available():
                            available_count = self.get_2fa_pictures_available_count()
                            if not available_count:
                                self.response_values['2fa_pictures'] = {
                                    'count_left': 0,
                                }
                            else:
                                self.failsafe_inc_kolmogor_counters([self.yakey_2fa_pictures_shown_counter])
                                self.fill_track_with_pictures_and_send_push()
                                self.response_values['2fa_pictures'] = {
                                    'correct': self.track.correct_2fa_picture,
                                    'expires_at': self.track.correct_2fa_picture_expires_at,
                                    'count_left': available_count - 1,
                                }

                        self.put_user_agent_and_ip_params_to_track(track)

                    if settings.AUTH_METHOD_MAGIC_LINK in allowed_auth_methods:
                        track.uid = self.account.uid
                        track.magic_link_start_time = datetime_to_integer_unixtime(datetime.now())
                        magic_link_email = get_masked_magic_link_email(
                            account=self.account,
                            user_entered_login=login,
                        )
                        self.response_values.update(magic_link_email=magic_link_email)
                    if settings.AUTH_METHOD_SMS_CODE in allowed_auth_methods:
                        track.uid = self.account.uid
                        track.has_secure_phone_number = bool(self.secure_number)
                        track.secure_phone_number = self.secure_number.e164
                        self.response_values.update(
                            secure_phone_number=dump_number(self.secure_number, only_masked=True),
                        )

                if self.form_values.get('old_track_id'):
                    old_track = self.track_manager.read(self.form_values.get('old_track_id'))
                    if self.is_phone_confirmed_in_track(track=old_track, allow_by_flash_call=True):
                        track.copy_phone_confirmation_from_other_track(old_track)

                if self.in_social_suggest_mode():
                    self.update_track_with_social_suggest_data(track)

                self.track.surface = get_surface_for_track(self.consumer, action='password')

                self.track.allow_scholar = self.form_values.get('allow_scholar')

            track.authorization_session_policy = self.form_values['policy']

        device_info = self.process_form(DeviceInfoForm, self.all_values)
        self.save_device_params_to_track(device_info)

        self.response_values.update(track_id=self.track_id)
        self.log_statbox_decision_reached()

    def try_parse_phone_number(self, raw_number, state, country_list=None):
        """Пытаемся распарсить номер со всеми переданными странами"""
        country_list = country_list or [None]
        error = None
        for country in country_list:
            try:
                result = validators.PhoneNumber().to_python(
                    {
                        'phone_number': raw_number,
                        'country': country,
                    },
                    state,
                )
                return result['phone_number'], country
            except validators.Invalid as e:
                error = e

        raise ValidationFailedError.from_invalid(error)

    def try_validate_login(self, login, login_type, login_validator, state):
        if looks_like_yandex_email(login):
            # если юзер ввёл яндексовый логин с доменной частью - отрежем яндексовый домен
            login_part, domain_part = login.rsplit('@', 1)
            login = login_part.strip()

        if login_type == 'lite' and not settings.ALLOW_LITE_REGISTRATION:
            raise validators.Invalid('invalid', login, state)

        login_validator.to_python(login, state)
        availability_validator = validators.Availability(login_type=ALIAS_NAME_TO_TYPE[login_type])
        availability_validator.to_python({'login': login}, state)

        if login_type == 'lite':
            if does_domain_belong_to_pdd(login):
                # Лайтов на ПДД-домене мы не разрешаем
                raise AccountNotFoundError()

        return login, login_type

    def try_register(self, account_not_found_error, phone_number=None, country=None):
        user_entered_value = self.form_values['login']
        login, account_type = None, None

        if self.form_values.get('allow_scholar') and login_is_scholar(user_entered_value):
            raise account_not_found_error

        if looks_like_yandex_email(user_entered_value):
            self.response_values.update(looks_like_yandex_email=True)

        state = validators.State(self.request.env)
        if phone_number:
            account_type = 'portal'  # допустимы обе регистрации, но портальную считаем приоритетной
        else:
            validation_error = None  # фоллбек на всякий (недостижимый) случай

            for login_type, login_validator in (
                ('lite', validators.LiteLogin()),
                ('portal', validators.Login()),  # предпочтительный тип должен стоять последним
            ):
                if (
                    login_type == 'lite' and
                    self.form_values['origin'] in settings.DONT_PROMOTE_LITE_REGISTRATION_IN_WEB_FOR_ORIGINS
                ):
                    continue

                try:
                    login, account_type = self.try_validate_login(
                        login=user_entered_value,
                        login_type=login_type,
                        login_validator=login_validator,
                        state=state,
                    )
                    # всё хорошо, мы нашли, какому типу аккаунта соответствует логин
                    break
                except validators.Invalid as e:
                    validation_error = e

            if account_type is None:
                raise ValidationFailedError.from_invalid(validation_error)

        # Переданный логин или телефон валидны, можно предложить регистрацию, если она возможна
        if settings.ALLOW_REGISTRATION:
            allowed_account_types = [account_type]
            if phone_number is not None and settings.ALLOW_NEOPHONISH_REGISTRATION:
                allowed_account_types.append('neophonish')

            self.response_values.update(
                can_register=True,
                account_type=account_type,
                allowed_account_types=allowed_account_types,
                login=login,
                phone_number=dump_number(phone_number) if phone_number else None,
                country=country,
            )
            self.create_and_fill_track('register')
        else:
            raise account_not_found_error

    def try_auth(self):
        self.response_values.update(
            primary_alias_type=self.get_alias_type(),
        )

        profiles = try_get_social_profiles_for_auth(self.account)
        saml_settings = None
        if self.account.is_federal:
            if not self.account.domain.domain:
                domain_id = self.account.federal_alias.alias.split('/', 1)[0]
                domain_info = self.blackbox.hosted_domains(domain_id=domain_id)
                self.account.parse(domain_info)
            else:
                domain_id = self.account.domain.id
            saml_settings = self.get_saml_settings(domain_id=domain_id, only_enabled=True)
        auth_methods = get_allowed_auth_methods(
            account=self.account,
            social_profiles=profiles,
            user_ip=self.client_ip,
            cookies=self.cookies,
            user_entered_login=self.form_values['login'],
            saml_settings=saml_settings,
        )

        if self.form_values.get('social_track_id'):
            auth_methods = self.filter_auth_methods_compatible_with_social_suggest(auth_methods)

        if not auth_methods:
            raise ActionImpossibleError()

        self.create_and_fill_track(
            'authorize',
            login=self.form_values['login'],
            allowed_auth_methods=auth_methods,
        )
        self.response_values.update(
            can_authorize=True,
            auth_methods=auth_methods,
            preferred_auth_method=auth_methods[0],
        )

    def try_saml_sso_auth(self):
        # надо проверить что домен федерала поддержан и может после этого отдать его на авторизацию
        user_entered_login = self.form_values['login']
        if settings.SAML_SSO_ENABLED and user_entered_login and '@' in user_entered_login:
            idp_domain = user_entered_login.rsplit('@', 1)[1]
            domain_info = self.blackbox.hosted_domains(domain=idp_domain)
            if domain_info['hosted_domains']:
                domain = Domain().parse(domain_info)
                if domain.is_enabled and self.get_saml_settings(domain_id=domain.id, only_enabled=True, with_enabled_jit_provisioning=True) is not None:
                    self.response_values.update(
                        primary_alias_type=ACCOUNT_TYPE_FEDERAL,
                        auth_methods=[settings.AUTH_METHOD_SAML_SSO],
                    )
                    self.create_and_fill_track(
                        'authorize',
                        login=user_entered_login,
                        allowed_auth_methods=[settings.AUTH_METHOD_SAML_SSO],
                    )
                    self.response_values.update(can_authorize=True)
                    return True

    def filter_auth_methods_compatible_with_social_suggest(self, auth_methods):
        """
        Оставляет методы входа, которые можно использовать логинясь в аккаунт с
        включением соц. входа.
        """
        auth_methods = [m for m in auth_methods if not m.startswith(settings.AUTH_METHOD_SOCIAL_PREFIX)]

        if self.account.have_password:
            # Если у аккаунта есть пароль, то для включения соц. входа, его
            # нужно будет запросить, поэтому предлагаем только вход по паролю.
            password_auth_methods = {
                settings.AUTH_METHOD_MAGIC,
                settings.AUTH_METHOD_OTP,
                settings.AUTH_METHOD_PASSWORD,
            }
            auth_methods = [m for m in auth_methods if m in password_auth_methods]
        return auth_methods

    def in_social_suggest_mode(self):
        """
        Признак, что данная авторизация продолжает соц. авторизацию, чтобы
        помочь пользователю не создавать аккаунт-дубль.
        """
        return self.form_values.get('social_track_id')

    def update_track_with_social_suggest_data(self, track):
        """
        Сохраняет в данном треке некоторые атрибуты из соц. авторизационного
        трека.
        """
        social_track = self.track_manager.read(self.form_values['social_track_id'])
        track.social_task_data = social_track.social_task_data
        track.social_task_id = social_track.social_task_id
        track.social_track_id = social_track.track_id

    def log_statbox_submit(self):
        cookie_info = {}
        if 'L' in self.cookies:
            try:
                cookie_info = CookieL().unpack(self.cookies['L'])
            except CookieLUnpackError as e:
                log.debug('Error while unpacking cookie: %s', e.message)

        self.statbox.log(
            action='submitted',
            referer=self.referer,
            retpath=self.form_values['retpath'],
            input_login=self.form_values['login'],
            l_login=cookie_info.get('login'),
            l_uid=cookie_info.get('uid'),
            cookie_yp=self.cookies.get('yp'),
            cookie_ys=self.cookies.get('ys'),
            cookie_my=self.cookies.get('my'),
        )

    def log_statbox_decision_reached(self):
        # Данная запись используется фронтом (для связки записей с треком и записей с process_uuid)
        # и nile-джобой профиля (для определения профилей, записанных после авторизации)
        self.statbox.log(
            action='decision_reached',
            track_type=self.track.track_type,
            track_id=self.track_id,
            referer=self.referer,
            retpath=self.form_values['retpath'],
        )

    def process_request(self):
        assert_valid_host(self.request.env)
        self.process_basic_form()
        self.log_statbox_submit()

        if self.form_values['is_pdd']:
            _, _, domain = self.form_values['login'].partition('@')
            self.check_pdd_domain(domain)

        countries = get_countries_suggest() or [None]
        country = countries[0]
        self.response_values.update(use_new_suggest_by_phone=False)
        try:
            state = validators.State(self.request.env)
            phone_number, country = self.try_parse_phone_number(
                self.form_values['login'],
                state,
                countries,
            )
            if (
                settings.USE_NEW_SUGGEST_BY_PHONE and
                is_experiment_enabled_by_phone(
                   phone_number,
                   settings.NEW_SUGGEST_BY_PHONE_DENOMINATOR,
                )
            ):
                self.response_values.update(use_new_suggest_by_phone=True)
            self.response_values.update(
                phone_number=dump_number(phone_number),
                country=country,
            )
        except ValidationFailedError:
            phone_number = None

        try:
            self.get_account_by_alt_logins(
                allow_federal=True,
                allow_scholar=self.form_values.get('allow_scholar'),
                check_disabled_on_deletion=True,
                emails=True,
                login=self.form_values['login'],
                need_phones=True,
            )
            # Пользователь нашёлся и не заблокирован. Проверим, можно ли им авторизоваться
            self.try_auth()
        except AccountNotFoundError as blackbox_error:
            if not self.try_saml_sso_auth():
                self.try_register(
                    account_not_found_error=blackbox_error,
                    phone_number=phone_number,
                    country=country,
                )
        except AccountDisabledOnDeletionError:
            self.response_values.update(
                can_be_restored=self.can_restore_disabled_account(self.account),
            )
            raise
        except BlackboxInvalidParamsError:
            # возможность регистрации не проверяем, логин заведомо невалидный
            raise ValidationFailedError(['login.prohibitedsymbols'])
        finally:
            if not self.track_id:
                self.create_and_fill_track('restore')
