# -*- coding: utf-8 -*-
from passport.backend.api.common import is_second_step_allowed
from passport.backend.api.common.profile.profile import process_env_profile
from passport.backend.api.views.bundle.auth.password.base import BasePasswordSubmitAuthView
from passport.backend.api.views.bundle.auth.password.exceptions import RfcOtpInvalidError
from passport.backend.api.views.bundle.exceptions import (
    ActionImpossibleError,
    CaptchaRequiredError,
    InvalidTrackStateError,
)
from passport.backend.api.views.bundle.headers import (
    HEADER_CLIENT_COOKIE,
    HEADER_CLIENT_HOST,
    HEADER_CLIENT_USER_AGENT,
    HEADER_CONSUMER_CLIENT_IP,
)
from passport.backend.api.views.bundle.mixins import (
    BundleAssertCaptchaMixin,
    BundleCacheResponseToTrackMixin,
)
from passport.backend.api.views.bundle.utils import write_phone_to_log
from passport.backend.core.builders.blackbox.constants import (
    BLACKBOX_CHECK_RFC_TOTP_VALID_STATUS,
    BLACKBOX_SECOND_STEP_RFC_TOTP,
)
from passport.backend.core.counters import bad_rfc_otp_counter
from passport.backend.core.logging_utils.loggers import StatboxLogger
from passport.backend.core.runner.context_managers import UPDATE
from passport.backend.core.utils.decorators import cached_property
from passport.backend.utils.time import get_unixtime

from .forms import RfcOtpForm


class RfcOtpView(BundleCacheResponseToTrackMixin,
                 BasePasswordSubmitAuthView,
                 BundleAssertCaptchaMixin):
    require_track = True
    required_grants = ['auth_password.base']
    required_headers = (
        HEADER_CLIENT_HOST,
        HEADER_CLIENT_USER_AGENT,
        HEADER_CLIENT_COOKIE,
        HEADER_CONSUMER_CLIENT_IP,
    )
    basic_form = RfcOtpForm

    @cached_property
    def statbox(self):
        return StatboxLogger(
            consumer=self.consumer,
            track_id=self.track_id,
            ip=self.client_ip,
            user_agent=self.user_agent,
            mode='any_auth',
            yandexuid=self.cookies.get('yandexuid'),
        )

    def read_and_check_track(self):
        self.read_track()
        self.response_values['track_id'] = self.track_id
        self.check_auth_not_passed()
        self.check_track_for_captcha()
        if not is_second_step_allowed(self.track, BLACKBOX_SECOND_STEP_RFC_TOTP):
            raise InvalidTrackStateError()

    def check_otp(self, otp):
        if bad_rfc_otp_counter.get_counter().hit_limit(self.account.uid):
            self.track.is_captcha_required = True
            self.check_track_for_captcha()

        rv = self.blackbox.check_rfc_totp(
            uid=self.account.uid,
            totp=otp,
        )

        if rv['status'] == BLACKBOX_CHECK_RFC_TOTP_VALID_STATUS:
            events = {'action': 'set_rfc_otp_check_time'}
            with UPDATE(self.account, self.request.env, events):
                self.account.rfc_totp_secret.check_time = rv['time']
            self.track.password_verification_passed_at = get_unixtime()
        else:
            if bad_rfc_otp_counter.incr_counter_and_check_limit_afterwards(self.account.uid):
                self.invalidate_captcha()
                self.track.is_captcha_required = True
                raise CaptchaRequiredError()
            raise RfcOtpInvalidError(rv['status'])

    def process_request(self):
        self.process_basic_form()
        self.statbox.log(
            action='rfc_otp_submitted',
        )

        self.read_and_check_track()
        self.get_account_from_track(
            check_disabled_on_deletion=True,
            need_phones=True,
        )
        self.statbox.bind_context(uid=self.account.uid)

        if not self.account.rfc_totp_secret.is_set:
            raise ActionImpossibleError()

        with self.track_transaction.commit_on_error():
            self.check_otp(self.form_values['otp'])

            redirect_state = self.check_user_policies()
            if redirect_state is not None:
                self.state = redirect_state
                self.fill_response_with_account(personal_data_required=True, account_info_required=True)
                return

            self.track.allow_authorization = True
            self.track.is_session_restricted = True
            write_phone_to_log(self.account, self.cookies)
            process_env_profile(self.account, track=self.track)
