# -*- coding: utf-8 -*-
import logging

from passport.backend.api.common.authorization import SessionScope
from passport.backend.api.common.common import (
    create_csrf_token,
    get_surface_for_track,
)
from passport.backend.api.common.ip import get_ip_autonomous_system
from passport.backend.api.common.profile.profile import process_env_profile
from passport.backend.api.views.bundle.constants import (
    CRED_STATUS_INVALID,
    CRED_STATUS_VALID,
)
from passport.backend.api.views.bundle.exceptions import (
    AccountInvalidTypeError,
    InvalidCSRFTokenError,
    InvalidTrackStateError,
    OAuthDeviceCodeNotFound,
    OAuthInvalidScopeError,
    OAuthTokenValidationError,
    SecondStepRequired,
)
from passport.backend.api.views.bundle.headers import (
    HEADER_CLIENT_COOKIE,
    HEADER_CLIENT_HOST,
    HEADER_CLIENT_USER_AGENT,
    HEADER_CONSUMER_CLIENT_IP,
)
from passport.backend.api.views.bundle.mixins import (
    BundleAccountSubscribeMixin,
    BundleAuthenticateMixinV2,
    BundleCacheResponseToTrackMixin,
    BundleDeviceInfoMixin,
    BundleFixPDDRetpathMixin,
)
from passport.backend.api.views.bundle.states import (
    EmailCode,
    OtpAuthFinished,
    OtpAuthNotReady,
    RedirectToForcedLiteCompletion,
    RedirectToPasswordChange,
    RfcTotp,
)
from passport.backend.api.views.bundle.utils import (
    assert_valid_host,
    write_phone_to_log,
)
from passport.backend.core import authtypes
from passport.backend.core.builders import oauth
from passport.backend.core.builders.blackbox.constants import (
    BLACKBOX_SECOND_STEP_EMAIL_CODE,
    BLACKBOX_SECOND_STEP_RFC_TOTP,
)
from passport.backend.core.builders.oauth import get_oauth
from passport.backend.core.conf import settings
from passport.backend.core.cookies.cookie_l import (
    CookieL,
    CookieLUnpackError,
)
from passport.backend.core.geobase import Region
from passport.backend.core.logging_utils.loggers.statbox import (
    AntifraudLogger,
    StatboxLogger,
)
from passport.backend.core.models.password import get_sha256_hash
from passport.backend.core.utils.decorators import cached_property
from passport.backend.utils.string import smart_text

from .base import BasePasswordSubmitAuthView
from .forms import (
    CommitMagicForm,
    CommitPasswordForm,
    SubmitForm,
)


log = logging.getLogger('passport.api.view.bundle.auth.password')


METHOD_PASSWORD = 'password'
METHOD_MAGIC = 'magic'
METHOD_MAGIC_X_TOKEN = 'magic_x_token'

OAUTH_SCOPE_CREATE_SESSION = 'passport:create_session'


# TODO: пересмотреть иерархию
class BaseStartView(BasePasswordSubmitAuthView):
    required_headers = (
        HEADER_CLIENT_HOST,
        HEADER_CLIENT_USER_AGENT,
        HEADER_CLIENT_COOKIE,
        HEADER_CONSUMER_CLIENT_IP,
    )

    track_type = 'authorize'

    @cached_property
    def statbox(self):
        return StatboxLogger(
            track_id=self.track_id,
            ip=self.client_ip,
            user_agent=self.user_agent,
            origin=self.track.origin if self.track else self.form_values.get('origin'),
            mode='any_auth',
            yandexuid=self.cookies.get('yandexuid'),
            consumer=self.consumer,
        )

    @cached_property
    def antifraud_log(self):
        return AntifraudLogger(
            channel='auth',
            sub_channel=settings.ANTIFRAUD_AUTH_SUB_CHANNEL,
            ip=self.client_ip,
            AS=get_ip_autonomous_system(self.client_ip),
            user_agent=self.user_agent,
            origin=self.track.origin if self.track else self.form_values.get('origin'),
            external_id='track-{}'.format(self.track_id),
            uid=self.account.uid if self.account else None,
            service_id='login',
        )

    def fill_track_with_request_data(self):
        """Записываем в трек все, что относится к запросу"""
        if self.form_values['retpath']:
            self.track.retpath = self.form_values['retpath']
        if self.form_values['origin']:
            self.track.origin = self.form_values['origin']
        if self.form_values['service']:
            self.track.service = self.form_values['service'].slug
        if self.form_values['fretpath']:
            self.track.fretpath = self.form_values['fretpath']
        if self.form_values['clean']:
            self.track.clean = self.form_values['clean']

        self.track.authorization_session_policy = self.form_values['policy']

        self.track.surface = get_surface_for_track(self.consumer, action='password')


class StartSubmitView(BaseStartView, BundleDeviceInfoMixin):
    basic_form = SubmitForm
    require_track = False

    def prepare_for_magic(self):
        self.track.is_allow_otp_magic = True
        self.track.csrf_token = create_csrf_token()
        self.response_values.update(csrf_token=self.track.csrf_token)

    def fill_track_with_user_agent_and_ip_data(self):
        user_agent_info = self.request.env.user_agent_info
        self.track.browser_id = settings.BROWSER_ENCODE.get(user_agent_info.get('BrowserName')) or None
        self.track.os_family_id = settings.OS_FAMILY_ENCODE.get(user_agent_info.get('OSFamily')) or None

        self.track.region_id = Region(ip=self.client_ip).id

        if self.form_values['device_name']:
            self.track.device_name = self.form_values['device_name']

    def process_request(self):
        assert_valid_host(self.request.env)
        self.process_basic_form()

        self.read_or_create_track(self.track_type)

        if self.form_values['with_code']:
            oauth_response = get_oauth().issue_device_code(
                client_bound=True,
                client_id=settings.TV_AUTH_CLIENT_ID,
                code_strength=settings.TV_AUTH_CODE_STRENGTH,
                **self.form_to_oauth_params(self.form_values)
            )
            device_code = oauth_response['device_code']
            user_code = oauth_response['user_code']
            verification_url = oauth_response['verification_url']
            expires_in = oauth_response['expires_in']
        else:
            device_code = user_code = verification_url = expires_in = None
        self.response_values.update(track_id=self.track_id)
        with self.track_transaction.commit_on_error():
            self.fill_track_with_request_data()
            self.prepare_for_magic()

            if self.form_values['with_code']:
                self.track.magic_qr_device_code = device_code
                self.response_values.update(
                    user_code=user_code,
                    # TODO Для известных client_id отдавать захардкоженный verification_url, а не из oauth
                    verification_url=verification_url,
                    expires_in=expires_in,
                )
                self.fill_track_with_user_agent_and_ip_data()


class BaseStartCommitView(BundleFixPDDRetpathMixin,
                          BundleAccountSubscribeMixin,
                          BundleAuthenticateMixinV2,
                          BaseStartView):
    sensitive_response_fields = ['cookies']
    method = None
    auth_by_otp = False
    auth_by_x_token = False
    auth_by_device_code = False
    antifraud_auth_type = None

    def process_auth_by_password(self, login, password, force_use_cache, need_oauth_token=False, add_avatar_with_secret_url=False):
        password_passed = False
        redirect_state = None
        if self.auth_by_x_token or self.auth_by_device_code:
            self.get_account_by_uid(
                self.track.uid,
                check_disabled_on_deletion=True,
                emails=True,
                need_phones=True,
            )
            if self.account.totp_secret.is_set:
                raise AccountInvalidTypeError()
            self.track.session_scope = str(SessionScope.xsession)
        else:
            try:
                self.blackbox_login(
                    login=login,
                    password=password,
                    retpath=self.track.retpath or '',
                    service=self.load_service_from_track() or '',
                    need_phones=True,
                    force_use_cache=force_use_cache,
                )
            except SecondStepRequired as e:
                if BLACKBOX_SECOND_STEP_RFC_TOTP in e.allowed_second_steps:
                    redirect_state = RfcTotp()
                elif BLACKBOX_SECOND_STEP_EMAIL_CODE in e.allowed_second_steps:
                    redirect_state = EmailCode()
                else:
                    raise  # pragma: no cover

                self.prepare_track_for_second_step(e.allowed_second_steps)

            password_passed = True

        self.statbox.bind_context(
            uid=self.account.uid,
            is_2fa_enabled=self.account.totp_secret.is_set or None,
        )

        # Запишем в трек все, что стало известно о пользователе
        self.fill_track_with_account_data(password_passed=password_passed)
        self._update_track_on_successful_auth()
        # Для ПДД проверим retpath (и поправим при необходимости)
        if self.account.is_pdd:
            self.fix_pdd_retpath()

        # Проверить пользователя на "законченность" (не требуется ли смены пароля
        # или заполнения анкеты с персональной информацией) и на требование челленжа перед авторизацией
        redirect_state = (
            redirect_state or
            self.check_user_policies()
        )
        # Не показываем челлендж на странице с QR-кодом на телевизоре или ГУ
        if not self.auth_by_device_code:
            redirect_state = (
                redirect_state or
                self.show_challenge_if_necessary(allow_new_challenge=True, auth_by_x_token=self.auth_by_x_token)
            )

        if redirect_state is not None:
            if self.method == METHOD_PASSWORD:
                self.track.password_hash = get_sha256_hash(password)
            if isinstance(redirect_state, RedirectToPasswordChange):
                self.save_secure_number_in_track()

            # Отдадим фронту флаг, есть ли у лайта средство восстановления
            if isinstance(redirect_state, RedirectToForcedLiteCompletion):
                self.response_values['has_recovery_method'] = self.check_has_recovery_method()

            self.state = redirect_state
            self.fill_response_with_account(personal_data_required=True, account_info_required=True, add_avatar_with_secret_url=add_avatar_with_secret_url)
            return

        if need_oauth_token:
            self.track.allow_oauth_authorization = True
        else:
            self.track.allow_authorization = True
            self.track.surface = get_surface_for_track(self.consumer, action='password')

        write_phone_to_log(self.account, self.cookies)
        process_env_profile(self.account, track=self.track)
        self.subscribe_if_allow_and_update_account(self.account)

    def _process_request(self):
        raise NotImplementedError()  # pragma: no cover

    def _get_track(self):
        raise NotImplementedError()  # pragma: no cover

    def _update_track_on_successful_auth(self):
        raise NotImplementedError()  # pragma: no cover

    def process_request(self):
        assert_valid_host(self.request.env)
        self.process_basic_form()

        self._get_track()

        cookie_info = {}
        if 'L' in self.cookies:
            try:
                cookie_info = CookieL().unpack(self.cookies['L'])
            except CookieLUnpackError as e:
                log.debug('Error while unpacking cookie: %s', smart_text(e))

        self.statbox.log(
            action='submitted',
            type=self.method,
            referer=self.referer,
            retpath=self.track.retpath,
            input_login=self.form_values.get('login'),
            l_login=cookie_info.get('login'),
            l_uid=cookie_info.get('uid'),
            cookie_yp=self.cookies.get('yp'),
            cookie_ys=self.cookies.get('ys'),
            cookie_my=self.cookies.get('my'),
        )

        self.response_values.update(track_id=self.track_id)
        # Проверим, что в треке не сохранено ошибочное состояние (от прошлого запроса)
        self.check_track()

        with self.track_transaction.commit_on_error():
            self._process_request()


class StartCommitPasswordView(BundleCacheResponseToTrackMixin, BaseStartCommitView):
    require_track = False
    basic_form = CommitPasswordForm
    method = METHOD_PASSWORD
    antifraud_auth_type = 'auth_by_password'

    def _get_track(self):
        if self.track_id:
            self.read_track()
        else:
            # хотим уметь ходить в commit, минуя submit
            self.create_track(self.track_type)

        with self.track_transaction.commit_on_error():
            self.fill_track_with_request_data()

    def _update_track_on_successful_auth(self):
        if self.account.totp_secret.is_set:
            self.track.auth_method = settings.AUTH_METHOD_OTP
        else:
            self.track.auth_method = settings.AUTH_METHOD_PASSWORD

    def _process_request(self):
        self.track.user_entered_login = self.form_values['login']

        if self.form_values['is_pdd']:
            _, _, domain = self.form_values['login'].partition('@')
            self.check_pdd_domain(domain)

        self.process_auth_by_password(
            login=self.form_values['login'],
            password=self.form_values['password'],
            force_use_cache=False,
            add_avatar_with_secret_url=settings.ISSUE_AVATARS_SECRETS,
        )


class StartCommitMagicView(BaseStartCommitView):
    require_track = True
    basic_form = CommitMagicForm
    method = METHOD_MAGIC
    antifraud_auth_type = 'auth_by_magic'

    def check_csrf_token(self):
        if self.track.csrf_token != self.form_values['csrf_token']:
            raise InvalidCSRFTokenError()

    def _get_track(self):
        self.read_track()

    def _update_track_on_successful_auth(self):
        if self.auth_by_otp:
            self.track.is_otp_magic_passed = True
            self.track.auth_method = settings.AUTH_METHOD_MAGIC
        elif self.auth_by_x_token:
            self.track.is_x_token_magic_passed = True
            self.track.auth_method = settings.AUTH_METHOD_MAGIC_X_TOKEN
            self.track.auth_source = authtypes.AUTH_SOURCE_XTOKEN

    def _process_request(self):
        self.check_csrf_token()
        if not getattr(self.track, 'is_allow_otp_magic', None):
            raise InvalidTrackStateError()

        cred_status = self.track.cred_status or self.track.x_token_status

        # Проверим, что трек заполнен правильно
        if (
            (cred_status == CRED_STATUS_VALID and not self.track.uid) or
            (self.track.otp and not self.track.login)
        ):
            raise InvalidTrackStateError()

        if cred_status == CRED_STATUS_INVALID and not (self.track.otp and self.track.login):
            raise OAuthTokenValidationError()

        # Расставляем приоритеты, если найдены обе возможности
        if self.track.login and self.track.otp:
            self.auth_by_otp = True
        elif self.track.uid and cred_status == CRED_STATUS_VALID:
            self.auth_by_x_token = True
        elif self.track.magic_qr_device_code:
            self.auth_by_device_code = True
            try:
                oauth_response = get_oauth().check_device_code(self.track.magic_qr_device_code)
            except oauth.OAuthDeviceCodeNotFound:  # ошибка из билдера
                raise OAuthDeviceCodeNotFound()  # ошибка апишная, наружу
            except oauth.OAuthDeviceCodeNotAccepted:
                self.state = OtpAuthNotReady()
                return
            else:
                oauth_uid = oauth_response['uid']
                if not self.track.uid:
                    self.track.uid = oauth_uid
                elif int(self.track.uid) != oauth_uid:
                    raise InvalidTrackStateError()
                scopes = oauth_response['scopes']
                if OAUTH_SCOPE_CREATE_SESSION not in scopes:
                    raise OAuthInvalidScopeError()

        if self.auth_by_otp or self.auth_by_x_token or self.auth_by_device_code:
            self.statbox.bind_context(
                type=(
                    (self.auth_by_otp and 'auth_by_otp')
                    or (self.auth_by_x_token and 'auth_by_x_token')
                    or 'auth_by_device_code'
                ),
            )
            self.state = OtpAuthFinished()
            self.process_auth_by_password(
                login=self.track.login,
                password=self.track.otp,
                force_use_cache=True,
                need_oauth_token=self.form_values['need_oauth_token'],
            )
            self.statbox.log(action='start_commit_magic')
        else:
            self.state = OtpAuthNotReady()
