# -*- coding: utf-8 -*-
import logging

from passport.backend.api.common.authorization import SessionScope
from passport.backend.api.common.common import get_surface_for_track
from passport.backend.api.common.ip import get_ip_autonomous_system
from passport.backend.api.common.profile.profile import process_env_profile
from passport.backend.api.views.bundle.headers import (
    HEADER_CLIENT_COOKIE,
    HEADER_CLIENT_HOST,
    HEADER_CLIENT_USER_AGENT,
    HEADER_CONSUMER_CLIENT_IP,
)
from passport.backend.api.views.bundle.mixins import (
    BundleAccountSubscribeMixin,
    BundleCacheResponseToTrackMixin,
    BundleFixPDDRetpathMixin,
)
from passport.backend.api.views.bundle.states import (
    RedirectToForcedLiteCompletion,
    RedirectToPasswordChange,
)
from passport.backend.api.views.bundle.utils import (
    assert_valid_host,
    write_phone_to_log,
)
from passport.backend.core.conf import settings
from passport.backend.core.cookies.cookie_l import (
    CookieL,
    CookieLUnpackError,
)
from passport.backend.core.logging_utils.loggers.statbox import (
    AntifraudLogger,
    StatboxLogger,
)
from passport.backend.core.models.password import get_sha256_hash
from passport.backend.core.utils.decorators import cached_property

from .base import BasePasswordSubmitAuthView
from .forms import (
    LoginPasswordForm,
    SubmitAuthorizationForm,
)


log = logging.getLogger('passport.api.view.bundle.auth.password')


class SubmitAuthorizationView(BundleFixPDDRetpathMixin,
                              BundleAccountSubscribeMixin,
                              BundleCacheResponseToTrackMixin,
                              BasePasswordSubmitAuthView):
    require_track = False
    required_headers = (
        HEADER_CLIENT_HOST,
        HEADER_CLIENT_USER_AGENT,
        HEADER_CLIENT_COOKIE,
        HEADER_CONSUMER_CLIENT_IP,
    )

    type = 'password'
    track_type = 'authorize'

    basic_form = SubmitAuthorizationForm

    sensitive_response_fields = ['cookies']
    antifraud_auth_type = 'auth_password_submit_old'

    @cached_property
    def statbox(self):
        return StatboxLogger(
            track_id=self.track.track_id,
            ip=self.client_ip,
            user_agent=self.user_agent,
            origin=self.track.origin if self.track.origin is not None else self.form_values['origin'],
            mode='any_auth',
            yandexuid=self.cookies.get('yandexuid'),
            input_login=self.form_values['login'],
        )

    @cached_property
    def antifraud_log(self):
        return AntifraudLogger(
            channel='auth',
            sub_channel=settings.ANTIFRAUD_AUTH_SUB_CHANNEL,
            ip=self.client_ip,
            AS=get_ip_autonomous_system(self.client_ip),
            user_agent=self.user_agent,
            origin=self.track.origin if self.track.origin is not None else self.form_values['origin'],
            external_id='track-{}'.format(self.track_id),
            uid=self.account.uid if self.account else None,
            service_id='login',
        )

    def fill_track_with_request_data(self):
        """Записываем в трек все, что относится к запросу"""
        if self.form_values['retpath']:
            self.track.retpath = self.form_values['retpath']
        if self.form_values['origin']:
            self.track.origin = self.form_values['origin']
        if self.form_values['service']:
            self.track.service = self.form_values['service'].slug
        if self.form_values['fretpath']:
            self.track.fretpath = self.form_values['fretpath']
        if self.form_values['clean']:
            self.track.clean = self.form_values['clean']

        self.track.user_entered_login = self.form_values['login']
        self.track.authorization_session_policy = self.form_values['policy']

        self.track.surface = get_surface_for_track(self.consumer, action='password')

    def is_password_and_login_empty(self):
        """Отвечает на вопрос: Были ли переданы пустые логин и пароль?"""
        return not self.form_values['password'] and not self.form_values['login']

    def process_auth_by_cookie(self):
        """
        Обрабатываем запрос на авторизацию по куке
        """
        assert_valid_host(self.request.env)
        self.get_account_from_session(dbfields=[], enabled_required=False)
        if self.account.is_pdd:
            # Получим данные о домене и обновим аккаунт
            response = self.hosted_domains(self.account.domain.domain)
            self.account.parse(response)
            # Для ПДД проверим retpath (и поправим при необходимости)
            self.fix_pdd_retpath()

        self.response_values['retpath'] = self.form_values['retpath']
        self.response_values['fretpath'] = self.form_values['fretpath']
        self.response_values['clean'] = self.form_values['clean']

    def process_auth_by_password(self):
        """
        Обрабатываем запрос на авторизацию по паролю:
        * Сначала проверим что это не повторный запрос, которому нужна капча;
        * Запросим авторизацию в ЧЯ по паролю;
        ** Покажем ошибку если таковая пришла от ЧЯ
        * Проверим валидность сессионной куки через ЧЯ;
        ** Соберем новую куку для пользователя если нужно,
        * Составим тело ответа
        ** Прицепим новые куки, если нужно
        * Запишем в логи все, что может быть интересно,
        * Отправим гендерную статистику
        """
        assert_valid_host(self.request.env)

        self.process_form(LoginPasswordForm(), self.all_values)

        self.read_or_create_track(self.track_type)

        cookie_info = {}
        if 'L' in self.cookies:
            try:
                cookie_info = CookieL().unpack(self.cookies['L'])
            except CookieLUnpackError as e:
                log.debug('Error while unpacking cookie: %s', e.message)

        self.statbox.log(
            action='submitted',
            type='password',
            referer=self.referer,
            retpath=self.form_values['retpath'] or self.track.retpath,
            l_login=cookie_info.get('login'),
            l_uid=cookie_info.get('uid'),
            cookie_yp=self.cookies.get('yp'),
            cookie_ys=self.cookies.get('ys'),
            cookie_my=self.cookies.get('my'),
        )

        # Проверим, что в треке не сохранено ошибочное состояние (от прошлого запроса)
        self.response_values['track_id'] = self.track_id
        self.check_track()

        with self.track_transaction.commit_on_error() as track:
            # Если пытаемся авторизоваться с ПДДшной странички,
            # то проверим сначала, обслуживается ли домен
            if self.form_values['is_pdd']:
                _, _, domain = self.form_values['login'].partition('@')
                self.check_pdd_domain(domain)

            # Запишем в трек все, что знаем об этом запросе
            self.fill_track_with_request_data()

            # Поскольку пришел пароль, сразу спрашиваем ЧЯ и парсим инфо о пользователе, если все удачно
            self.blackbox_login(login=self.form_values['login'], need_phones=True)

            self.statbox.bind_context(
                uid=self.account.uid,
                is_2fa_enabled=self.account.totp_secret.is_set or None,
            )

            # Для ПДД проверим retpath (и поправим при необходимости)
            if self.account.is_pdd:
                self.fix_pdd_retpath()

            # Записать в трек все, что стало известно о пользователе
            self.fill_track_with_account_data(password_passed=True)
            if self.account.totp_secret.is_set:
                self.track.auth_method = settings.AUTH_METHOD_OTP
            else:
                self.track.auth_method = settings.AUTH_METHOD_PASSWORD

            # Проверить пользователя на "законченность" - не требуется ли смены пароля
            # или заполнения анкеты с персональной информацией
            redirect_state = self.check_user_policies() or self.show_challenge_if_necessary(allow_new_challenge=True)

            if redirect_state is not None:
                self.track.password_hash = get_sha256_hash(self.form_values['password'])
                if isinstance(redirect_state, RedirectToPasswordChange):
                    self.save_secure_number_in_track()

                # Отдадим фронту флаг, есть ли у лайта средство восстановления
                if isinstance(redirect_state, RedirectToForcedLiteCompletion):
                    self.response_values['has_recovery_method'] = self.check_has_recovery_method()

                self.fill_response_with_account(personal_data_required=True, account_info_required=True)
                return redirect_state

            # Проверим валидность сессионной куки
            # Будем проверять всегда с multisession=yes.
            # Работает и со второй версией куки.
            cookie_session_info = self.check_session_cookie(
                dbfields=[],
                attributes=[],
            )

            self.set_old_session_track_fields(cookie_session_info)

            # Если пользователь дошел сюда - он определенно имеет пароль (поскольку ввел его)
            self.fill_response_with_account_and_session(
                cookie_session_info=cookie_session_info,
                # Не проверяем глогаут, т.к. авторизуем по предъявлению логина-пароля в method=login ЧЯ,
                # Глогаут имеет смысл только для кук или треков. Решено 18.12.2014
                logout_check_enabled=False,
                personal_data_required=True,
                session_scope=SessionScope.xsession,
            )
            write_phone_to_log(self.account, self.cookies)

            if self.profile_load_result and self.profile_load_result.success:
                ufo_profile = self.profile_load_result.content
            else:
                ufo_profile = None

            process_env_profile(
                self.account,
                ufo_profile=ufo_profile,
                track=self.track,
            )

            self.response_values['retpath'] = track.retpath

            service = self.form_values['service']
            self.subscribe_if_allow_and_update_account(self.account, service=service)

    def process_request(self, *args, **kwargs):

        self.process_basic_form()

        if self.is_password_and_login_empty():
            # Если пароля не было - проверим ТОЛЬКО валидность сессионной куки
            # В трек ничего писать не будем
            self.state = self.process_auth_by_cookie()
        else:
            # Если пароль был введен - работаем с треком
            self.state = self.process_auth_by_password()
